/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.NetworkFilterParameterBuilder;
import org.ovirt.engine.sdk4.types.NetworkFilterParameter;

/**
 * This service manages a parameter for a network filter.
 */
public interface NicNetworkFilterParameterService extends Service {
    /**
     * Retrieves a representation of the network filter parameter.
     */
    public interface GetRequest extends Request<GetRequest, GetResponse> {
    }
    
    /**
     * Retrieves a representation of the network filter parameter.
     */
    public interface GetResponse extends Response {
        /**
         * The representation of the network filter parameter.
         */
        NetworkFilterParameter parameter();
    }
    
    /**
     * Retrieves a representation of the network filter parameter.
     */
    GetRequest get();
    
    /**
     * Removes the filter parameter.
     * 
     * For example, to remove the filter parameter with id `123` on NIC `456` of virtual machine `789`
     * send a request like this:
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/vms/789/nics/456/networkfilterparameters/123
     * ----
     */
    public interface RemoveRequest extends Request<RemoveRequest, RemoveResponse> {
    }
    
    /**
     * Removes the filter parameter.
     * 
     * For example, to remove the filter parameter with id `123` on NIC `456` of virtual machine `789`
     * send a request like this:
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/vms/789/nics/456/networkfilterparameters/123
     * ----
     */
    public interface RemoveResponse extends Response {
    }
    
    /**
     * Removes the filter parameter.
     * 
     * For example, to remove the filter parameter with id `123` on NIC `456` of virtual machine `789`
     * send a request like this:
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/vms/789/nics/456/networkfilterparameters/123
     * ----
     */
    RemoveRequest remove();
    
    /**
     * Updates the network filter parameter.
     * 
     * For example, to update the network filter parameter having with with id `123` on NIC `456` of
     * virtual machine `789` send a request like this:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/vms/789/nics/456/networkfilterparameters/123
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <network_filter_parameter>
     *   <name>updatedName</name>
     *   <value>updatedValue</value>
     * </network_filter_parameter>
     * ----
     */
    public interface UpdateRequest extends Request<UpdateRequest, UpdateResponse> {
        UpdateRequest parameter(NetworkFilterParameter parameter);
        /**
         * The network filter parameter that is being updated.
         */
        UpdateRequest parameter(NetworkFilterParameterBuilder parameter);
    }
    
    /**
     * Updates the network filter parameter.
     * 
     * For example, to update the network filter parameter having with with id `123` on NIC `456` of
     * virtual machine `789` send a request like this:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/vms/789/nics/456/networkfilterparameters/123
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <network_filter_parameter>
     *   <name>updatedName</name>
     *   <value>updatedValue</value>
     * </network_filter_parameter>
     * ----
     */
    public interface UpdateResponse extends Response {
        /**
         * The network filter parameter that is being updated.
         */
        NetworkFilterParameter parameter();
    }
    
    /**
     * Updates the network filter parameter.
     * 
     * For example, to update the network filter parameter having with with id `123` on NIC `456` of
     * virtual machine `789` send a request like this:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/vms/789/nics/456/networkfilterparameters/123
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <network_filter_parameter>
     *   <name>updatedName</name>
     *   <value>updatedValue</value>
     * </network_filter_parameter>
     * ----
     */
    UpdateRequest update();
    
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

