/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.DataCenterBuilder;
import org.ovirt.engine.sdk4.types.DataCenter;

public interface DataCenterService extends Service {
    public interface GetRequest extends Request<GetRequest, GetResponse> {
        /**
         * Indicates if the results should be filtered according to the permissions of the user.
         */
        GetRequest filter(Boolean filter);
    }
    
    public interface GetResponse extends Response {
        DataCenter dataCenter();
    }
    
    GetRequest get();
    
    /**
     * Removes the data center.
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/datacenters/123
     * ----
     * 
     * Without any special parameters, the storage domains attached to the data center are detached and then removed
     * from the storage. If something fails when performing this operation, for example if there is no host available to
     * remove the storage domains from the storage, the complete operation will fail.
     * 
     * If the `force` parameter is `true` then the operation will always succeed, even if something fails while removing
     * one storage domain, for example. The failure is just ignored and the data center is removed from the database
     * anyway.
     */
    public interface RemoveRequest extends Request<RemoveRequest, RemoveResponse> {
        /**
         * Indicates if the remove should be performed asynchronously.
         */
        RemoveRequest async(Boolean async);
        /**
         * Indicates if the operation should succeed, and the storage domain removed from the database, even if
         * something fails during the operation.
         * 
         * This parameter is optional, and the default value is `false`.
         */
        RemoveRequest force(Boolean force);
    }
    
    /**
     * Removes the data center.
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/datacenters/123
     * ----
     * 
     * Without any special parameters, the storage domains attached to the data center are detached and then removed
     * from the storage. If something fails when performing this operation, for example if there is no host available to
     * remove the storage domains from the storage, the complete operation will fail.
     * 
     * If the `force` parameter is `true` then the operation will always succeed, even if something fails while removing
     * one storage domain, for example. The failure is just ignored and the data center is removed from the database
     * anyway.
     */
    public interface RemoveResponse extends Response {
    }
    
    /**
     * Removes the data center.
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/datacenters/123
     * ----
     * 
     * Without any special parameters, the storage domains attached to the data center are detached and then removed
     * from the storage. If something fails when performing this operation, for example if there is no host available to
     * remove the storage domains from the storage, the complete operation will fail.
     * 
     * If the `force` parameter is `true` then the operation will always succeed, even if something fails while removing
     * one storage domain, for example. The failure is just ignored and the data center is removed from the database
     * anyway.
     */
    RemoveRequest remove();
    
    /**
     * Updates the data center.
     * 
     * The `name`, `description`, `storage_type`, `version`, `storage_format` and `mac_pool` elements are updatable
     * post-creation. For example, to change the name and description of data center `123` send a request like this:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/datacenters/123
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <data_center>
     *   <name>myupdatedname</name>
     *   <description>An updated description for the data center</description>
     * </data_center>
     * ----
     */
    public interface UpdateRequest extends Request<UpdateRequest, UpdateResponse> {
        /**
         * Indicates if the update should be performed asynchronously.
         */
        UpdateRequest async(Boolean async);
        UpdateRequest dataCenter(DataCenter dataCenter);
        /**
         * The data center that is being updated.
         */
        UpdateRequest dataCenter(DataCenterBuilder dataCenter);
    }
    
    /**
     * Updates the data center.
     * 
     * The `name`, `description`, `storage_type`, `version`, `storage_format` and `mac_pool` elements are updatable
     * post-creation. For example, to change the name and description of data center `123` send a request like this:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/datacenters/123
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <data_center>
     *   <name>myupdatedname</name>
     *   <description>An updated description for the data center</description>
     * </data_center>
     * ----
     */
    public interface UpdateResponse extends Response {
        /**
         * The data center that is being updated.
         */
        DataCenter dataCenter();
    }
    
    /**
     * Updates the data center.
     * 
     * The `name`, `description`, `storage_type`, `version`, `storage_format` and `mac_pool` elements are updatable
     * post-creation. For example, to change the name and description of data center `123` send a request like this:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/datacenters/123
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <data_center>
     *   <name>myupdatedname</name>
     *   <description>An updated description for the data center</description>
     * </data_center>
     * ----
     */
    UpdateRequest update();
    
    ClustersService clustersService();
    IscsiBondsService iscsiBondsService();
    /**
     * Returns a reference to the service, that manages the networks, that are associated with the data center.
     */
    NetworksService networksService();
    AssignedPermissionsService permissionsService();
    QossService qossService();
    QuotasService quotasService();
    /**
     * Attach and detach storage domains to and from a data center.
     * 
     * For attaching a single storage domain we should use the folowing POST request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/datacenters/123/storagedomains
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <storage_domain>
     *   <name>data1</name>
     * </storage_domain>
     * ----
     * 
     * For detaching a single storage domain we should use the folowing DELETE request:
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/datacenters/123/storagedomains/123
     * ----
     */
    AttachedStorageDomainsService storageDomainsService();
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

