/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.NetworkBuilder;
import org.ovirt.engine.sdk4.types.Network;

/**
 * Manages logical networks.
 * 
 * The engine creates a default `ovirtmgmt` network on installation. This network acts as the management network for
 * access to hypervisor hosts. This network is associated with the `Default` cluster and is a member of the `Default`
 * data center.
 */
public interface NetworksService extends Service {
    /**
     * Creates a new logical network, or associates an existing network with a data center.
     * 
     * Creation of a new network requires the `name` and `data_center` elements.
     * 
     * For example, to create a network named `mynetwork` for data center `123` send a request like this:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/networks
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <network>
     *   <name>mynetwork</name>
     *   <data_center id="123"/>
     * </network>
     * ----
     * 
     * 
     * To associate the existing network `456` with the data center `123` send a request like this:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/datacenters/123/networks
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <network>
     *   <name>ovirtmgmt</name>
     * </network>
     * ----
     */
    public interface AddRequest extends Request<AddRequest, AddResponse> {
        AddRequest network(Network network);
        AddRequest network(NetworkBuilder network);
    }
    
    /**
     * Creates a new logical network, or associates an existing network with a data center.
     * 
     * Creation of a new network requires the `name` and `data_center` elements.
     * 
     * For example, to create a network named `mynetwork` for data center `123` send a request like this:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/networks
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <network>
     *   <name>mynetwork</name>
     *   <data_center id="123"/>
     * </network>
     * ----
     * 
     * 
     * To associate the existing network `456` with the data center `123` send a request like this:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/datacenters/123/networks
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <network>
     *   <name>ovirtmgmt</name>
     * </network>
     * ----
     */
    public interface AddResponse extends Response {
        Network network();
    }
    
    /**
     * Creates a new logical network, or associates an existing network with a data center.
     * 
     * Creation of a new network requires the `name` and `data_center` elements.
     * 
     * For example, to create a network named `mynetwork` for data center `123` send a request like this:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/networks
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <network>
     *   <name>mynetwork</name>
     *   <data_center id="123"/>
     * </network>
     * ----
     * 
     * 
     * To associate the existing network `456` with the data center `123` send a request like this:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/datacenters/123/networks
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <network>
     *   <name>ovirtmgmt</name>
     * </network>
     * ----
     */
    AddRequest add();
    
    /**
     * List logical networks.
     * 
     * For example:
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/networks
     * ----
     * 
     * Will respond:
     * 
     * [source,xml]
     * ----
     * <networks>
     *   <network href="/ovirt-engine/api/networks/123" id="123">
     *     <name>ovirtmgmt</name>
     *     <description>Default Management Network</description>
     *     <link href="/ovirt-engine/api/networks/123/permissions" rel="permissions"/>
     *     <link href="/ovirt-engine/api/networks/123/vnicprofiles" rel="vnicprofiles"/>
     *     <link href="/ovirt-engine/api/networks/123/networklabels" rel="networklabels"/>
     *     <mtu>0</mtu>
     *     <stp>false</stp>
     *     <usages>
     *       <usage>vm</usage>
     *     </usages>
     *     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
     *   </network>
     *   ...
     * </networks>
     * ----
     */
    public interface ListRequest extends Request<ListRequest, ListResponse> {
        /**
         * Indicates if the search performed using the `search` parameter should be performed taking case into
         * account. The default value is `true`, which means that case is taken into account. If you want to search
         * ignoring case set it to `false`.
         */
        ListRequest caseSensitive(Boolean caseSensitive);
        /**
         * Sets the maximum number of networks to return. If not specified all the networks are returned.
         */
        ListRequest max(Integer max);
        /**
         * Sets the maximum number of networks to return. If not specified all the networks are returned.
         */
        ListRequest max(Long max);
        /**
         * Sets the maximum number of networks to return. If not specified all the networks are returned.
         */
        ListRequest max(BigInteger max);
        /**
         * A query string used to restrict the returned networks.
         */
        ListRequest search(String search);
    }
    
    /**
     * List logical networks.
     * 
     * For example:
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/networks
     * ----
     * 
     * Will respond:
     * 
     * [source,xml]
     * ----
     * <networks>
     *   <network href="/ovirt-engine/api/networks/123" id="123">
     *     <name>ovirtmgmt</name>
     *     <description>Default Management Network</description>
     *     <link href="/ovirt-engine/api/networks/123/permissions" rel="permissions"/>
     *     <link href="/ovirt-engine/api/networks/123/vnicprofiles" rel="vnicprofiles"/>
     *     <link href="/ovirt-engine/api/networks/123/networklabels" rel="networklabels"/>
     *     <mtu>0</mtu>
     *     <stp>false</stp>
     *     <usages>
     *       <usage>vm</usage>
     *     </usages>
     *     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
     *   </network>
     *   ...
     * </networks>
     * ----
     */
    public interface ListResponse extends Response {
        List<Network> networks();
    }
    
    /**
     * List logical networks.
     * 
     * For example:
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/networks
     * ----
     * 
     * Will respond:
     * 
     * [source,xml]
     * ----
     * <networks>
     *   <network href="/ovirt-engine/api/networks/123" id="123">
     *     <name>ovirtmgmt</name>
     *     <description>Default Management Network</description>
     *     <link href="/ovirt-engine/api/networks/123/permissions" rel="permissions"/>
     *     <link href="/ovirt-engine/api/networks/123/vnicprofiles" rel="vnicprofiles"/>
     *     <link href="/ovirt-engine/api/networks/123/networklabels" rel="networklabels"/>
     *     <mtu>0</mtu>
     *     <stp>false</stp>
     *     <usages>
     *       <usage>vm</usage>
     *     </usages>
     *     <data_center href="/ovirt-engine/api/datacenters/456" id="456"/>
     *   </network>
     *   ...
     * </networks>
     * ----
     */
    ListRequest list();
    
    NetworkService networkService(String id);
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

