/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.HostDeviceBuilder;
import org.ovirt.engine.sdk4.types.HostDevice;

/**
 * A service to manage host devices attached to a virtual machine.
 */
public interface VmHostDevicesService extends Service {
    /**
     * Attach target device to given virtual machine.
     * 
     * Example:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/vms/123/hostdevices
     * ----
     * 
     * With request body of type <<types/host_device,HostDevice>>, for example
     * 
     * [source,xml]
     * ----
     * <host_device id="123" />
     * ----
     * 
     * NOTE: A necessary precondition for a successful host device attachment is that the virtual machine must be pinned
     * to *exactly* one host. The device ID is then taken relative to this host.
     * 
     * NOTE: Attachment of a PCI device that is part of a bigger IOMMU group will result in attachment of the remaining
     * devices from that IOMMU group as "placeholders". These devices are then identified using the `placeholder`
     * attribute of the <<types/host_device,HostDevice>> type set to `true`.
     * 
     * In case you want attach a device that already serves as an IOMMU placeholder, simply issue an explicit Add operation
     * for it, and its `placeholder` flag will be cleared, and the device will be accessible to the virtual machine.
     */
    public interface AddRequest extends Request<AddRequest, AddResponse> {
        AddRequest device(HostDevice device);
        /**
         * The host device to be attached to given virtual machine.
         */
        AddRequest device(HostDeviceBuilder device);
    }
    
    /**
     * Attach target device to given virtual machine.
     * 
     * Example:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/vms/123/hostdevices
     * ----
     * 
     * With request body of type <<types/host_device,HostDevice>>, for example
     * 
     * [source,xml]
     * ----
     * <host_device id="123" />
     * ----
     * 
     * NOTE: A necessary precondition for a successful host device attachment is that the virtual machine must be pinned
     * to *exactly* one host. The device ID is then taken relative to this host.
     * 
     * NOTE: Attachment of a PCI device that is part of a bigger IOMMU group will result in attachment of the remaining
     * devices from that IOMMU group as "placeholders". These devices are then identified using the `placeholder`
     * attribute of the <<types/host_device,HostDevice>> type set to `true`.
     * 
     * In case you want attach a device that already serves as an IOMMU placeholder, simply issue an explicit Add operation
     * for it, and its `placeholder` flag will be cleared, and the device will be accessible to the virtual machine.
     */
    public interface AddResponse extends Response {
        /**
         * The host device to be attached to given virtual machine.
         */
        HostDevice device();
    }
    
    /**
     * Attach target device to given virtual machine.
     * 
     * Example:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/vms/123/hostdevices
     * ----
     * 
     * With request body of type <<types/host_device,HostDevice>>, for example
     * 
     * [source,xml]
     * ----
     * <host_device id="123" />
     * ----
     * 
     * NOTE: A necessary precondition for a successful host device attachment is that the virtual machine must be pinned
     * to *exactly* one host. The device ID is then taken relative to this host.
     * 
     * NOTE: Attachment of a PCI device that is part of a bigger IOMMU group will result in attachment of the remaining
     * devices from that IOMMU group as "placeholders". These devices are then identified using the `placeholder`
     * attribute of the <<types/host_device,HostDevice>> type set to `true`.
     * 
     * In case you want attach a device that already serves as an IOMMU placeholder, simply issue an explicit Add operation
     * for it, and its `placeholder` flag will be cleared, and the device will be accessible to the virtual machine.
     */
    AddRequest add();
    
    /**
     * List the host devices assigned to given virtual machine.
     */
    public interface ListRequest extends Request<ListRequest, ListResponse> {
        /**
         * Sets the maximum number of devices to return. If not specified all the devices are returned.
         */
        ListRequest max(Integer max);
        /**
         * Sets the maximum number of devices to return. If not specified all the devices are returned.
         */
        ListRequest max(Long max);
        /**
         * Sets the maximum number of devices to return. If not specified all the devices are returned.
         */
        ListRequest max(BigInteger max);
    }
    
    /**
     * List the host devices assigned to given virtual machine.
     */
    public interface ListResponse extends Response {
        /**
         * Retrieved list of host devices attached to given virtual machine.
         */
        List<HostDevice> device();
    }
    
    /**
     * List the host devices assigned to given virtual machine.
     */
    ListRequest list();
    
    /**
     * Returns a reference to the service that manages a specific host device attached to given virtual machine.
     */
    VmHostDeviceService deviceService(String id);
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

