/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.xml;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import org.ovirt.api.metamodel.runtime.util.ArrayListWithHref;
import org.ovirt.api.metamodel.runtime.util.ListWithHref;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.engine.sdk4.internal.containers.CpuTypeContainer;
import org.ovirt.engine.sdk4.types.CpuType;

public class XmlCpuTypeReader {
    
    public static CpuType readOne(XmlReader reader) {
        // Do nothing if there aren't more tags:
        if (!reader.forward()) {
            return null;
        }
        
        // Create the object:
        CpuTypeContainer object = new CpuTypeContainer();
        
        // Process the inner elements:
        reader.next();
        while (reader.forward()) {
            String name = reader.getLocalName();
            switch (name) {
                case "architecture":
                object.architecture(XmlArchitectureReader.readOne(reader));
                break;
                case "level":
                object.level(reader.readInteger());
                break;
                case "name":
                object.name(reader.readString());
                break;
                case "link":
                readLink(reader, object);
                break;
                default:
                reader.skip();
                break;
            }
        }
        
        // Discard the end tag:
        reader.next();
        
        return object;
    }
    
    public static Iterator<CpuType> iterateMany(final XmlReader reader) {
        return new Iterator<CpuType>() {
            private boolean first = true;
            
            @Override
            public void remove() {
                // Empty on purpose
            }
            
            @Override
            public boolean hasNext() {
                if (first) {
                    if (!reader.forward()) {
                        return false;
                    }
                    reader.next();
                    first = false;
                }
                if (!reader.forward()) {
                    reader.next();
                    return false;
                }
                return true;
            }
            
            @Override
            public CpuType next() {
                CpuType next = readOne(reader);
                if (next == null) {
                    throw new NoSuchElementException();
                }
                return next;
            }
        };
    }
    
    public static List<CpuType> readMany(XmlReader reader) {
        List<CpuType> list = new ArrayList<>();
        Iterator<CpuType> iterator = iterateMany(reader);
        while (iterator.hasNext()) {
            list.add(iterator.next());
        }
        return list;
    }
    
    private static void readLink(XmlReader reader, CpuTypeContainer object) {
        // Process the attributes:
        String rel = null;
        String href = null;
        for (int i = 0; i < reader.getAttributeCount(); i++) {
            String attrName = reader.getAttributeLocalName(i);
            String attrVal = reader.getAttributeValue(i);
            switch (attrName) {
                case "href":
                href = attrVal;
                break;
                case "rel":
                rel = attrVal;
                break;
                default:
                reader.skip();
                break;
            }
        }
        reader.skip();
    }
}

