/*
Copyright (c) 2015 Red Hat, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package services;

import annotations.Area;
import org.ovirt.api.metamodel.annotations.In;
import org.ovirt.api.metamodel.annotations.InputDetail;
import org.ovirt.api.metamodel.annotations.Out;
import org.ovirt.api.metamodel.annotations.Service;
import types.Cluster;
import types.Disk;
import types.Image;
import types.StorageDomain;
import types.Template;

import static org.ovirt.api.metamodel.language.ApiLanguage.mandatory;
import static org.ovirt.api.metamodel.language.ApiLanguage.optional;
import static org.ovirt.api.metamodel.language.ApiLanguage.or;

@Service
@Area("Storage")
public interface ImageService {
    interface Get {
        @Out Image image();
    }

    /**
     * Imports an image.
     *
     * If the `import_as_template` parameter is `true` then the image will be imported as a template, otherwise it will
     * be imported as a disk.
     *
     * When imported as a template, the name of the template can be specified by the optional `template.name`
     * parameter. If that parameter is not specified, then the name of the template will be automatically assigned by the
     * engine as `GlanceTemplate-x` (where `x` will be seven random hexadecimal characters).
     *
     * When imported as a disk, the name of the disk can be specified by the optional `disk.name` parameter. If
     * that parameter is not specified, then the name of the disk will be automatically assigned by the engine as
     * `GlanceDisk-x` (where `x` will be the seven hexadecimal characters of the image identifier).
     *
     * It is recommended to always explicitly specify the template or disk name, to avoid these automatic names
     * generated by the engine.
     *
     * @author Juan Hernandez <juan.hernandez@redhat.com>
     * @author Tahlia Richardson <trichard@redhat.com>
     * @date 24 Mar 2017
     * @status updated_by_docs
     */
    interface Import {
        @InputDetail
        default void inputDetail() {
            or(mandatory(storageDomain().id()), mandatory(storageDomain().name()));
            optional(importAsTemplate());
            or(optional(cluster().id()), optional(cluster().name()));
            or(optional(disk().alias()), optional(disk().name()));
        }

        /**
         * The cluster to which the image should be imported if the `import_as_template` parameter
         * is set to `true`.
         *
         * @author Ondra Machacek <omachace@redhat.com>
         * @author Tahlia Richardson <trichard@redhat.com>
         * @date 24 Mar 2017
         * @status updated_by_docs
         */
        @In Cluster cluster();

        /**
         * The disk to import.
         *
         * @author Ondra Machacek <omachace@redhat.com>
         * @author Tahlia Richardson <trichard@redhat.com>
         * @date 24 Mar 2017
         * @status updated_by_docs
         */
        @In Disk disk();

        /**
         * Specifies if a template should be created from the imported disk.
         *
         * @author Ondra Machacek <omachace@redhat.com>
         * @author Tahlia Richardson <trichard@redhat.com>
         * @date 24 Mar 2017
         * @status updated_by_docs
         */
        @In Boolean importAsTemplate();

        /**
         * The name of the template being created if the
         * `import_as_template` parameter is set to `true`.
         *
         * @author Ondra Machacek <omachace@redhat.com>
         * @author Tahlia Richardson <trichard@redhat.com>
         * @date 24 Mar 2017
         * @status updated_by_docs
         */
        @In Template template();

        /**
         * The storage domain to which the disk should be imported.
         *
         * @author Ondra Machacek <omachace@redhat.com>
         * @author Tahlia Richardson <trichard@redhat.com>
         * @date 24 Mar 2017
         * @status updated_by_docs
         */
        @In StorageDomain storageDomain();

        /**
         * Indicates if the import should be performed asynchronously.
         *
         * @author Tahlia Richardson <trichard@redhat.com>
         * @date 24 Mar 2017
         * @status updated_by_docs
         */
        @In Boolean async();
    }
}
