! This file is part of mctc-lib.
!
! Licensed under the Apache License, Version 2.0 (the "License");
! you may not use this file except in compliance with the License.
! You may obtain a copy of the License at
!
!     http://www.apache.org/licenses/LICENSE-2.0
!
! Unless required by applicable law or agreed to in writing, software
! distributed under the License is distributed on an "AS IS" BASIS,
! WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
! See the License for the specific language governing permissions and
! limitations under the License.

module testsuite_structure
   use mctc_env_accuracy, only : wp
   use mctc_io_structure, only : structure_type, new
   implicit none
   private

   public :: get_structure


contains


subroutine get_structure(self, name)
   type(structure_type), intent(out) :: self
   character(len=*), intent(in) :: name

   select case(name)
   case('mindless01'); call mindless01(self)
   case('mindless02'); call mindless02(self)
   case('mindless03'); call mindless03(self)
   case('mindless04'); call mindless04(self)
   case('mindless05'); call mindless05(self)
   case('mindless06'); call mindless06(self)
   case('mindless07'); call mindless07(self)
   case('mindless08'); call mindless08(self)
   case('mindless09'); call mindless09(self)
   case('mindless10'); call mindless10(self)
   case('x01'); call x01(self)
   case('x02'); call x02(self)
   case('x03'); call x03(self)
   case('x04'); call x04(self)
   case('x05'); call x05(self)
   end select

end subroutine get_structure


subroutine mindless01(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=2) ::&
      & "Na", "H", "O", "H", "F", "H", "H", "O", "N", "H", "H", "Cl", "B", "B", "N", "Al"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.85528263484662_wp,  3.58670515364616_wp, -2.41763729306344_wp, &
      &  4.40178023537845_wp,  0.02338844412653_wp, -4.95457749372945_wp, &
      & -2.98706033463438_wp,  4.76252065456814_wp,  1.27043301573532_wp, &
      &  0.79980886075526_wp,  1.41103455609189_wp, -5.04655321620119_wp, &
      & -4.20647469409936_wp,  1.84275767548460_wp,  4.55038084858449_wp, &
      & -3.54356121843970_wp, -3.18835665176557_wp,  1.46240021785588_wp, &
      &  2.70032160109941_wp,  1.06818452504054_wp, -1.73234650374438_wp, &
      &  3.73114088824361_wp, -2.07001543363453_wp,  2.23160937604731_wp, &
      & -1.75306819230397_wp,  0.35951417150421_wp,  1.05323406177129_wp, &
      &  5.41755788583825_wp, -1.57881830078929_wp,  1.75394002750038_wp, &
      & -2.23462868255966_wp, -2.13856505054269_wp,  4.10922285746451_wp, &
      &  1.01565866207568_wp, -3.21952154552768_wp, -3.36050963020778_wp, &
      &  2.42119255723593_wp,  0.26626435093114_wp, -3.91862474360560_wp, &
      & -3.02526098819107_wp,  2.53667889095925_wp,  2.31664984740423_wp, &
      & -2.00438948664892_wp, -2.29235136977220_wp,  2.19782807357059_wp, &
      &  1.12226554109716_wp, -1.36942007032045_wp,  0.48455055461782_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine mindless01


subroutine mindless02(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=2) ::&
      & "H", "S", "B", "O", "Mg", "H", "H", "H", "Si", "H", "B", "Li", "F", "H", "H", "S"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.79537625851198_wp, -3.77866422935275_wp, -1.07883558363403_wp, &
      & -2.68278833302782_wp,  0.38892666265890_wp,  1.66214865238427_wp, &
      &  0.11484649791305_wp,  1.48857933226955_wp,  3.65660396510375_wp, &
      & -1.07998879593946_wp, -0.16259121615748_wp, -4.55703065871422_wp, &
      &  0.60302832999383_wp,  4.08816149622342_wp, -0.02589373148029_wp, &
      & -1.22534089315880_wp, -1.79981382478068_wp, -3.70773173318592_wp, &
      & -1.33460982049866_wp, -4.24819082475503_wp,  2.72791902701083_wp, &
      & -0.16278082578516_wp,  2.41267994179303_wp,  5.69030695190570_wp, &
      &  2.87802444057103_wp, -0.33120525058830_wp,  1.88311373530297_wp, &
      &  0.68489327931487_wp,  0.32790204044961_wp, -4.20547693710673_wp, &
      & -1.20919773588330_wp, -2.87253762561437_wp,  0.94064204223101_wp, &
      & -3.25572604597922_wp,  2.21241092990940_wp, -2.86715549314771_wp, &
      & -1.83147468262373_wp,  5.20527293771933_wp, -2.26976270603341_wp, &
      &  4.90885865772880_wp, -1.92576561961811_wp,  2.99069919443735_wp, &
      &  1.26806242248758_wp, -2.60409341782411_wp,  0.55162805282247_wp, &
      &  4.11956976339902_wp,  1.59892866766766_wp, -1.39117477789609_wp],&
      & shape(xyz))
   integer, parameter :: uhf = 2
   call new(self, sym, xyz, uhf=uhf)
end subroutine mindless02


subroutine mindless03(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=2) ::&
      & "C", "O", "H", "Li", "Mg", "Al", "C", "H", "H", "H", "F", "S", "C", "H", "Na", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -0.02148551327524_wp, -0.67161751504297_wp, -4.75078512817560_wp, &
      &  1.37792545875526_wp, -3.24818416423144_wp,  3.83896600631495_wp, &
      & -2.23986953822894_wp,  1.64550402751694_wp,  3.42773272178522_wp, &
      & -0.87622711432790_wp, -2.74068400827752_wp,  1.43723692979592_wp, &
      &  1.29492470653815_wp,  1.86470311043681_wp, -1.04536500695239_wp, &
      & -3.65768365013010_wp,  0.45437052179208_wp, -1.41566056087159_wp, &
      & -0.23245910487384_wp, -1.83274112101585_wp, -2.43395808606122_wp, &
      &  0.30373451850419_wp, -3.84228931776777_wp, -2.44882782867802_wp, &
      & -3.36159503902161_wp,  4.20056392581975_wp,  1.63352684198071_wp, &
      &  0.49372989648081_wp, -1.56245253044952_wp, -6.53610501083288_wp, &
      &  4.38566058812996_wp,  1.86127331114460_wp,  0.56178822055152_wp, &
      & -1.17545963764009_wp,  2.49456345795141_wp, -4.90195191215762_wp, &
      & -1.86623614216854_wp,  2.76329843590746_wp,  1.71572598870213_wp, &
      &  1.02361259176985_wp, -4.24377370348987_wp,  5.32418288889440_wp, &
      &  4.71194535010347_wp, -1.03648125005561_wp,  3.35573062118779_wp, &
      & -0.16051737061546_wp,  3.89394681976155_wp,  2.23776331451663_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine mindless03


subroutine mindless04(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=2) ::&
      & "H", "B", "H", "F", "B", "H", "H", "Si", "H", "H", "C", "Al", "Si", "O", "H", "B"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.34544890768411_wp,  2.85946545334720_wp,  3.11183388215396_wp, &
      & -0.36293929605305_wp,  4.15983774640545_wp,  1.36413101934678_wp, &
      & -3.36268280924844_wp,  4.92951597114402_wp, -3.59085684882314_wp, &
      &  3.78143178536443_wp, -4.97181356229699_wp,  1.59003443639387_wp, &
      &  3.44227417874042_wp, -3.46504338606415_wp,  3.62082644591507_wp, &
      &  1.88917586252014_wp,  3.42088101960529_wp,  1.28872629783483_wp, &
      & -0.32747529934233_wp, -4.29711514977711_wp, -3.55330460209973_wp, &
      & -3.58768360829779_wp, -1.39509759062952_wp, -1.10396714572410_wp, &
      & -0.39440896193088_wp,  6.31837673143592_wp,  1.99105318714945_wp, &
      &  4.34376903295874_wp, -4.12502353873667_wp,  5.57829602371555_wp, &
      & -1.39570266622309_wp, -2.60410756418652_wp, -4.03149806979915_wp, &
      &  0.21788515354592_wp,  0.28610741675369_wp,  1.29731097788136_wp, &
      & -2.00000183598828_wp,  3.04473467156937_wp, -2.00578147078785_wp, &
      &  2.12833842504876_wp, -1.30141517432227_wp,  3.38069910888504_wp, &
      & -2.48411958079522_wp, -2.81581487156584_wp, -5.76829803496286_wp, &
      & -0.54241147261516_wp, -0.04348817268188_wp, -3.16920520707912_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine mindless04


subroutine mindless05(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=2) ::&
      & "B", "P", "H", "H", "B", "P", "H", "Cl", "N", "H", "P", "Si", "H", "H", "P", "N"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.68391902268453_wp,  0.21679405065309_wp, -2.81441127558071_wp, &
      & -2.67199537993843_wp, -3.97743927106200_wp,  0.03497540139192_wp, &
      &  2.02325266152397_wp, -0.16048070975416_wp, -0.41980608052722_wp, &
      &  4.26224346168617_wp,  3.65384961705338_wp, -2.81836810458488_wp, &
      & -2.80378310343644_wp,  1.84796600006216_wp,  0.15107304476153_wp, &
      &  1.58317082705122_wp,  3.77079801391042_wp, -2.86230158107979_wp, &
      &  2.63670178694113_wp,  3.13142099211650_wp,  2.24139937019049_wp, &
      & -6.27112533979613_wp, -3.92471014080274_wp,  1.62562669834852_wp, &
      & -0.92594349239390_wp, -2.94451283088352_wp,  2.60616476876177_wp, &
      & -1.79532342290201_wp, -1.56841672860834_wp,  3.65515689388732_wp, &
      & -3.01460634915379_wp, -0.47748181717446_wp, -2.44834110183776_wp, &
      &  2.18249449208515_wp, -2.23505035804805_wp,  1.77725119258081_wp, &
      &  3.26068149442689_wp, -4.54078259646428_wp,  0.57204329987377_wp, &
      &  1.73744972267909_wp, -1.18654391698320_wp, -4.24063427353503_wp, &
      &  0.94405328902426_wp,  4.99525793054843_wp,  1.18501287451328_wp, &
      & -1.83118967048165_wp,  3.39933176543682_wp,  1.75515887283605_wp],&
      & shape(xyz))
   integer, parameter :: uhf = 1
   call new(self, sym, xyz, uhf=uhf)
end subroutine mindless05


subroutine mindless06(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=2) ::&
      & "B", "N", "H", "O", "B", "H", "Al", "H", "B", "Mg", "H", "H", "H", "H", "C", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.10912945825730_wp,  1.64180252123600_wp,  0.27838149792131_wp, &
      & -2.30085163837888_wp,  0.87765138232225_wp, -0.60457694150897_wp, &
      &  2.78083551168063_wp,  4.95421363506113_wp,  0.40788634984219_wp, &
      & -5.36229602768251_wp, -7.29510945515334_wp,  0.06097106408867_wp, &
      &  2.13846114572058_wp, -0.99012126457352_wp,  0.93647189687052_wp, &
      &  0.09330150731888_wp, -2.75648066796634_wp, -3.70294675694565_wp, &
      & -1.52684105316140_wp, -2.44981814860506_wp, -1.02727325811774_wp, &
      & -0.45240334635443_wp,  5.86105501765814_wp,  0.30815308772432_wp, &
      & -3.95419048213910_wp, -5.52061943693205_wp, -0.31702321028260_wp, &
      &  2.68706169520082_wp, -0.13577304635533_wp, -3.57041492458512_wp, &
      & -3.79914135008731_wp,  2.06429808651079_wp, -0.77285245656187_wp, &
      &  0.89693752015341_wp,  4.58640300917890_wp,  3.09718012019731_wp, &
      &  2.76317093138142_wp, -0.62928000132252_wp,  3.08807601371151_wp, &
      &  1.00075543259914_wp, -3.11885279872042_wp,  1.08659460804098_wp, &
      &  0.86969979951508_wp,  4.43363816376984_wp,  1.02355776570620_wp, &
      &  4.05637089597643_wp, -1.52300699610852_wp, -0.29218485610105_wp],&
      & shape(xyz))
   integer, parameter :: uhf = 1
   call new(self, sym, xyz, uhf=uhf)
end subroutine mindless06


subroutine mindless07(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=2) ::&
      & "C", "H", "B", "H", "H", "Cl", "F", "N", "C", "H", "S", "H", "H", "O", "F", "Mg"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -3.75104222741336_wp, -5.81308736205268_wp, -1.22507366840233_wp, &
      & -1.45226572768296_wp, -3.01878767879831_wp,  2.38723142561073_wp, &
      & -1.99423317853240_wp, -3.52953889999752_wp, -1.30301724065129_wp, &
      & -4.33750965171233_wp, -6.65936981001909_wp,  0.55979831484564_wp, &
      & -4.51833920602637_wp, -6.72398616322561_wp, -2.90031439001886_wp, &
      & -1.25657105633503_wp, -2.39389339457851_wp, -4.58765484136593_wp, &
      & -0.14864209579028_wp,  4.40065007854051_wp,  1.35717716022989_wp, &
      & -0.91662354168326_wp, -2.22680612180354_wp,  0.71122632634918_wp, &
      &  1.83282041695179_wp,  5.36061635978157_wp,  3.22095765094686_wp, &
      &  0.66518416413161_wp,  6.30980889882630_wp,  4.62705414435961_wp, &
      &  3.68701623423530_wp,  2.79957532381681_wp,  4.21336212424745_wp, &
      &  1.69373321407504_wp,  0.01030275402386_wp, -3.74820290941150_wp, &
      &  3.35791986589808_wp,  2.52513229318111_wp, -3.46078430541625_wp, &
      &  2.79199182665654_wp,  1.01759578021447_wp, -2.59243571461852_wp, &
      &  3.05358934464082_wp,  7.15252337445235_wp,  1.82164153773112_wp, &
      &  1.29297161858681_wp,  0.78926456763834_wp,  0.91903438556425_wp],&
      & shape(xyz))
   integer, parameter :: uhf = 1
   call new(self, sym, xyz, uhf=uhf)
end subroutine mindless07


subroutine mindless08(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=2) ::&
      & "C", "O", "B", "F", "H", "Al", "H", "H", "O", "B", "Be", "C", "H", "H", "B", "F"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.27823293129313_wp,  0.06442674490989_wp,  2.76980447300615_wp, &
      &  2.05039033278229_wp,  0.64690940303039_wp, -0.29571013189632_wp, &
      & -0.07388472989895_wp,  2.46033979750309_wp, -1.30590420482375_wp, &
      &  1.10019432741349_wp,  4.43501067437330_wp, -2.64796515354449_wp, &
      & -1.89008873387150_wp,  0.02064696008121_wp,  4.74727599156952_wp, &
      &  0.81013963557610_wp,  1.41165582964016_wp, -6.35835508532445_wp, &
      &  2.51638337449170_wp,  1.74086425451198_wp,  3.45340860505386_wp, &
      &  2.62048878651566_wp, -1.58024532804571_wp,  2.87415150030394_wp, &
      & -0.92472602392464_wp, -3.37659091509259_wp, -0.68138826965952_wp, &
      & -2.19962829538645_wp, -2.53092502025386_wp,  1.35654623095955_wp, &
      &  0.92594749614406_wp, -1.61669775704536_wp, -1.93872059141561_wp, &
      &  1.63141903847248_wp,  0.18081362275364_wp,  2.42899361614054_wp, &
      & -3.96336280784845_wp, -3.68611886004249_wp,  2.18920954455515_wp, &
      & -1.17097381446263_wp,  1.08303722364990_wp, -3.04753977323348_wp, &
      & -2.18263847972349_wp,  2.31604957286801_wp,  1.11461091308323_wp, &
      &  2.02857282501340_wp, -1.56917620284149_wp, -4.65841766477431_wp],&
      & shape(xyz))
   integer, parameter :: uhf = 1
   call new(self, sym, xyz, uhf=uhf)
end subroutine mindless08


subroutine mindless09(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=2) ::&
      & "H", "H", "H", "H", "Li", "H", "C", "B", "H", "H", "Si", "H", "Cl", "F", "H", "B"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.97360649552839_wp,  1.71723751297383_wp, -0.51862929250676_wp, &
      &  0.16903666216522_wp,  1.73154352333176_wp, -0.40099024352959_wp, &
      & -3.94463844105182_wp, -1.24346369608005_wp,  0.09565841726334_wp, &
      &  2.21647168119803_wp,  4.10625979391554_wp,  2.61391340002321_wp, &
      & -0.04488993380842_wp, -2.16288302687041_wp,  4.48488595610432_wp, &
      &  3.52287141817194_wp, -0.90500888687059_wp, -5.00916337263077_wp, &
      &  1.95336082370762_wp, -0.83849036872324_wp, -3.65515970516029_wp, &
      &  2.05706981818495_wp,  1.70095588601056_wp, -2.06303335904159_wp, &
      & -6.40097100472159_wp, -1.71072935987273_wp,  3.14621771036234_wp, &
      &  2.04751538182937_wp, -2.55691868000982_wp, -2.49926722310562_wp, &
      &  2.03251078714394_wp,  1.35094356516468_wp,  2.02150308748654_wp, &
      &  0.20477572129201_wp, -0.93291693232462_wp, -4.76431390827476_wp, &
      & -2.67673272939098_wp,  1.40764602033672_wp,  4.10347165469140_wp, &
      & -2.75901984658887_wp, -3.73954809548334_wp,  3.19373273207227_wp, &
      &  1.96938102642596_wp,  3.74070925169244_wp, -3.03185101883736_wp, &
      & -4.32034786008576_wp, -1.66533650719069_wp,  2.28302516508337_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine mindless09


subroutine mindless10(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=2) ::&
      & "H", "Si", "H", "Cl", "C", "H", "F", "H", "C", "N", "B", "H", "Mg", "C", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.57062307661218_wp, -1.68792229443234_wp,  2.78939425857465_wp, &
      & -2.08994110527129_wp,  3.25317728228563_wp, -0.42147881550833_wp, &
      &  2.13532981939105_wp, -1.71356933061236_wp, -2.49234593851880_wp, &
      & -2.46885241522113_wp, -4.41076598859264_wp, -0.58746410797603_wp, &
      &  3.86605901148259_wp, -0.50808683490216_wp,  1.10929274542242_wp, &
      & -4.57284898019279_wp, -1.54920337824862_wp, -2.63711913350102_wp, &
      & -4.99945502320431_wp,  0.09990896897876_wp, -3.20268495970371_wp, &
      &  1.63618508154720_wp,  2.66791559582643_wp, -3.16904643876699_wp, &
      & -2.28445827511587_wp,  0.42792856662334_wp,  2.04433546457507_wp, &
      &  0.78486183614848_wp,  1.96692225005484_wp, -1.58921219981020_wp, &
      & -0.92003258313224_wp, -1.56076484060483_wp,  0.46494611026243_wp, &
      & -1.07970143095156_wp,  1.19037461384346_wp,  3.56880222429743_wp, &
      &  3.27327901654007_wp,  3.47628642644825_wp,  1.85050408639730_wp, &
      &  1.64922592697103_wp, -0.66726875777723_wp, -0.77306391492380_wp, &
      &  5.67004330685832_wp, -1.05218123504276_wp,  0.25282456342591_wp, &
      & -4.17031726246173_wp,  0.06724895615223_wp,  2.79231605575371_wp],&
      & shape(xyz))
   integer, parameter :: uhf = 1
   call new(self, sym, xyz, uhf=uhf)
end subroutine mindless10


subroutine x01(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 32
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "O", "O", "O", "O", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  8.66499304004865_wp,  9.33379706146984_wp,  7.91683397818847_wp, &
      & 11.12335640640559_wp,  2.15906103856442_wp, 12.28606540623589_wp, &
      &  2.69933763795316_wp,  8.69604933222823_wp,  5.94619038919326_wp, &
      &  4.40748472136020_wp,  2.73477407301820_wp,  1.34585013180550_wp, &
      &  8.94474331294941_wp,  5.24347823723191_wp,  6.03793064749213_wp, &
      & 10.69920497119952_wp,  4.18256695916720_wp,  4.01290993162606_wp, &
      & 10.03588165896165_wp,  5.08392793928533_wp,  1.39546189302066_wp, &
      &  8.49376544841568_wp,  7.47023679468653_wp,  1.28534952220463_wp, &
      &  9.07166275596235_wp,  9.29463739048189_wp,  3.43672461445391_wp, &
      &  8.87252167134021_wp,  8.08457959995134_wp,  6.01243046499065_wp, &
      &  2.61511156242260_wp,  9.98111860346994_wp,  7.83909475310702_wp, &
      &  1.94055488237084_wp,  8.92607461070971_wp, 10.39160309677660_wp, &
      &  3.94058158183755_wp,  9.62934843688878_wp, 12.34968145677051_wp, &
      &  4.41149849250265_wp, -0.13918998706016_wp, 12.14066417045890_wp, &
      &  4.70428751489632_wp,  1.17418327733924_wp,  9.63142334925047_wp, &
      &  2.97542523741995_wp,  0.23498193061563_wp,  7.50807857572976_wp, &
      & 11.00218693527931_wp, 10.01811156711939_wp,  3.26258886294108_wp, &
      &  6.51725105145864_wp,  6.87025658460207_wp,  1.40398487849503_wp, &
      &  9.16505707508531_wp,  6.55996790419566_wp, 12.22499217557358_wp, &
      &  0.47155553650782_wp,  4.68566204197289_wp,  3.97665598787901_wp, &
      & 10.71305929405516_wp,  2.12295125271661_wp,  4.04673290893442_wp, &
      &  7.00559444915079_wp,  4.60292552897028_wp,  5.68707427274543_wp, &
      &  7.82621004905011_wp, 10.93721749961744_wp,  3.36440794575946_wp, &
      &  9.47562563356497_wp,  4.56669825883223_wp,  7.90994201013468_wp, &
      &  5.72226645413659_wp,  8.73073901566233_wp, 11.81706046742993_wp, &
      &  1.10806523402693_wp,  1.08795999732802_wp,  7.75198050780745_wp, &
      &  4.49303750240828_wp,  3.20681326576017_wp,  9.88850009827761_wp, &
      &  3.68234836535172_wp,  0.82971735354827_wp,  5.66899365523961_wp, &
      &  6.67318941946786_wp,  0.84421918391682_wp,  9.09225122218742_wp, &
      & 12.30020769805011_wp,  9.84555048261104_wp, 11.35093359840778_wp, &
      &  1.70097366892267_wp,  6.88253129604813_wp, 10.28258692216369_wp, &
      &  3.02585151643235_wp, 10.74223721977969_wp,  1.45047192019470_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 12.17191845_wp,  0.09308248_wp,  0.41423668_wp, &
      &  0.08731987_wp, 12.55440630_wp,  0.31568135_wp, &
      &  0.42750296_wp, -1.80907922_wp, 12.77606779_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine x01


subroutine x02(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 32
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "O", "O", "O", "O", "O", "O", "O", "O", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  4.63745097550818_wp,  2.37915954409831_wp, 19.12271349268357_wp, &
      &  6.72709904800693_wp,  0.45406655172709_wp, 16.05143697871524_wp, &
      &  6.91256964714223_wp,  2.38436998330633_wp,  6.39788951244788_wp, &
      &  4.82391466561283_wp,  0.46056271598603_wp,  3.32509104154164_wp, &
      &  1.18919515335071_wp,  5.93351886762742_wp,  9.40656871308612_wp, &
      &  3.27471753885763_wp,  7.86164871844023_wp,  6.33444054660705_wp, &
      &  0.99780233239929_wp,  7.85689402199700_wp, 19.06217051148892_wp, &
      &  3.08952816179886_wp,  5.93388907848718_wp, 22.13323554052032_wp, &
      &  5.35416029613234_wp,  2.30873365987837_wp, 16.93028032095304_wp, &
      &  4.75001882722751_wp,  4.26813080508463_wp, 14.98920903273459_wp, &
      &  6.19847745199579_wp,  2.31366251977415_wp,  4.20458897657195_wp, &
      &  6.80665649236884_wp,  4.27192054545680_wp,  2.26362280139482_wp, &
      &  3.16548840193314_wp,  9.74798782169150_wp, 10.46928111352906_wp, &
      &  2.56099108285706_wp,  7.78911812897077_wp,  8.52781851045641_wp, &
      &  1.71531586300926_wp,  7.78766523049261_wp, 21.25437643855244_wp, &
      &  1.11185021611237_wp,  9.74796815379632_wp, 23.19468740594458_wp, &
      &  3.61329543151651_wp,  3.43205969905233_wp, 13.49003378404895_wp, &
      &  7.17673435114791_wp, 10.13506068796258_wp, 17.37760565299595_wp, &
      &  3.71404617620063_wp,  5.82813222850270_wp, 15.82034897510546_wp, &
      &  6.46923526186126_wp,  4.97322450985026_wp, 14.10746213158056_wp, &
      &  5.08841444241702_wp,  4.98373937264637_wp,  1.38536619864770_wp, &
      &  0.56730092315824_wp,  5.82834639286665_wp,  3.09197971230966_wp, &
      &  0.65639014106239_wp,  3.43360236266220_wp,  0.75941943516012_wp, &
      &  4.37810806504803_wp, 10.14190879051040_wp,  4.65109383610143_wp, &
      &  4.20736555115953_wp,  0.34920021575627_wp,  9.63882258640363_wp, &
      &  4.29177512440358_wp,  8.91048751253497_wp, 11.97544245280817_wp, &
      &  0.74057850072650_wp,  4.66235021937769_wp,  8.08112406897593_wp, &
      &  1.44463798101514_wp, 10.46104535442793_wp, 11.34128971196497_wp, &
      &  0.06308951988013_wp,  0.34832836424057_wp, 22.36345606641216_wp, &
      & -0.01363861729261_wp,  8.91069125987469_wp, 24.70168579125730_wp, &
      &  2.83286106954203_wp, 10.46215767099852_wp, 24.06567365395194_wp, &
      &  3.53590396893376_wp,  4.66141487828773_wp, 20.80838381072642_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      &  7.28162822_wp, -0.00060036_wp,  0.00257376_wp, &
      &  0.00297342_wp, 10.95374740_wp,  0.00144914_wp, &
      & -0.01715980_wp, -0.00425337_wp, 25.44979900_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine x02


subroutine x03(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 52
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "C", "C", "C", "C", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  6.85499925334061_wp,  0.58339620591570_wp, 15.88596621412105_wp, &
      &  9.24413370501684_wp,  2.95651635041689_wp,  2.46485240796786_wp, &
      &  5.31512352904071_wp,  9.21510777326209_wp,  9.20126499327503_wp, &
      &  2.94513117890964_wp,  6.85725126528710_wp,  5.80174646207598_wp, &
      &  0.57238636973286_wp,  9.22086291490312_wp,  9.19505291377807_wp, &
      &  2.95101143232547_wp, 11.60011353636975_wp,  5.81736060783860_wp, &
      &  2.94175316098990_wp,  9.21047863759430_wp, 10.88860009988064_wp, &
      &  5.31187062289578_wp,  6.85199657074529_wp,  7.49949099919000_wp, &
      &  5.31674998211317_wp, 11.58947903551133_wp,  7.51493725091216_wp, &
      &  2.95263788539794_wp,  9.23399965125764_wp,  4.11911238860320_wp, &
      &  0.57451326990455_wp,  6.85762660061152_wp,  7.49311102565259_wp, &
      &  9.20935263162105_wp,  2.94137782566549_wp, 12.48510453059833_wp, &
      &  9.23462521013167_wp,  5.32488224747550_wp,  0.76979417550117_wp, &
      & 11.58885347663730_wp,  2.94563162600886_wp, 14.16404493518103_wp, &
      &  9.23174763931115_wp,  0.58201997639285_wp,  0.77919624176683_wp, &
      &  6.85862749480996_wp,  5.32025311180771_wp, 15.87606046573401_wp, &
      &  6.84586609377985_wp,  2.94825897327976_wp, 14.18704641872381_wp, &
      & 11.59535928892716_wp,  5.31737554098720_wp, 15.85373055835306_wp, &
      &  0.57989307622116_wp, 11.59410817117911_wp,  7.50771780717246_wp, &
      & 11.59248171810665_wp,  0.57989307622116_wp, 15.86330051865918_wp, &
      &  5.11732181307401_wp,  0.55412005061133_wp,  0.27349939190652_wp, &
      &  7.06155879354367_wp, 11.62288387938426_wp,  6.34908103396994_wp, &
      &  5.10243351187221_wp,  2.94813386150496_wp, 13.01380286400142_wp, &
      &  0.82974129050674_wp,  0.55111736801601_wp,  0.23471586856066_wp, &
      & 11.62375966180790_wp,  7.05930678159717_wp, 14.68300541423754_wp, &
      &  6.82459709206300_wp,  7.05993234047120_wp, 14.70231322889024_wp, &
      &  0.54636312057342_wp,  0.82323547821688_wp,  8.68012189169256_wp, &
      &  9.23550099255530_wp, 11.35727158147325_wp,  1.96419237932130_wp, &
      &  1.26100157825957_wp,  9.24175658129555_wp,  2.87552123092879_wp, &
      &  2.93774958419614_wp,  7.50645626475024_wp, 12.11422659522601_wp, &
      &  4.65465846984512_wp,  9.23550099255530_wp,  2.89079958861049_wp, &
      &  2.94362983761198_wp, 10.90023826811060_wp, 12.13487756220238_wp, &
      &  2.95351366782157_wp,  0.83074218470518_wp,  4.64159864250933_wp, &
      & 11.34400973334392_wp,  9.21861090295662_wp, 10.37383697183559_wp, &
      &  2.94513117890964_wp,  5.12307695471503_wp,  4.61456770199555_wp, &
      & 11.35089088095820_wp, 11.62838879747568_wp,  6.33464214649053_wp, &
      &  0.54285999087888_wp,  5.11244245385661_wp,  8.65846356047344_wp, &
      &  7.05542831657822_wp,  6.82334597431495_wp,  6.33162005376228_wp, &
      &  0.83124263180440_wp,  5.34690191984118_wp,  0.22699274269958_wp, &
      &  5.34840326113884_wp,  0.81710500125144_wp,  8.69019553412005_wp, &
      &  7.05167496333407_wp,  9.21022841404469_wp, 10.38525376658675_wp, &
      &  5.12057471921893_wp,  5.35666063827596_wp,  0.26208259715536_wp, &
      &  6.81821639154794_wp, 11.34963976321015_wp, 14.72044578525974_wp, &
      & 10.95128387223103_wp,  2.95826791526416_wp,  3.68611365826132_wp, &
      &  9.20284681933119_wp,  1.23885679411908_wp, 11.25746330676746_wp, &
      &  7.55775209242028_wp,  2.95864325058858_wp,  3.71566300938197_wp, &
      &  9.20272170755639_wp,  4.63263879747945_wp, 11.24084179676209_wp, &
      &  9.24150635774594_wp,  7.06506192323821_wp,  1.94773876335639_wp, &
      &  0.81009874186236_wp,  2.94200338453951_wp, 12.97434776449373_wp, &
      &  5.33952032512768_wp,  5.10481063559351_wp,  8.66249301744444_wp, &
      & 11.34300883914548_wp,  6.83035223370403_wp,  6.32356113982028_wp, &
      & 11.61750407306765_wp, 11.34613663351561_wp, 14.69710851363604_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 12.51117748_wp,  0.00000000_wp,  0.00000000_wp, &
      &  0.00000000_wp, 12.51117748_wp,  0.00000000_wp, &
      &  0.00000000_wp,  0.00000000_wp, 16.78940405_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine x03


subroutine x04(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "N", "N", &
      & "N", "N"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  4.15467418193883_wp,  3.33328901394025_wp,  1.66323391111685_wp, &
      &  3.33328901394025_wp,  1.66323391111685_wp,  4.15467418193883_wp, &
      &  1.66323391111685_wp,  4.15467418193883_wp,  3.33328901394025_wp, &
      &  9.14844968255857_wp,  3.06025186813034_wp,  9.33489558920461_wp, &
      &  8.23223100208403_wp,  4.72395947311570_wp,  6.86439258737082_wp, &
      &  6.65104086899334_wp,  2.17993918288896_wp,  7.71770472210706_wp, &
      &  7.71770472210706_wp,  6.65104086899334_wp,  2.17993918288896_wp, &
      &  9.33489558920461_wp,  9.14844968255857_wp,  3.06025186813034_wp, &
      &  6.86439258737082_wp,  8.23223100208403_wp,  4.72395947311570_wp, &
      &  4.72395947311570_wp,  6.86439258737082_wp,  8.23223100208403_wp, &
      &  2.17993918288896_wp,  7.71770472210706_wp,  6.65104086899334_wp, &
      &  3.06025186813034_wp,  9.33489558920461_wp,  9.14844968255857_wp, &
      &  2.59764243614222_wp,  2.59764243614222_wp,  2.59764243614222_wp, &
      &  7.55541720275509_wp,  3.76255039758519_wp,  8.43506671658056_wp, &
      &  8.43506671658056_wp,  7.55541720275509_wp,  3.76255039758519_wp, &
      &  3.76255039758519_wp,  8.43506671658056_wp,  7.55541720275509_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      &  9.47387737_wp,  0.00000000_wp,  0.00000000_wp, &
      &  0.00000000_wp,  9.47387737_wp,  0.00000000_wp, &
      &  0.00000000_wp,  0.00000000_wp,  9.47387737_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine x04


subroutine x05(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 48
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  9.36760291026725_wp,  7.80244140446007_wp,  2.94420668811823_wp, &
      & -2.60635659089003_wp, 11.13901918927945_wp, 16.83643328201347_wp, &
      &  7.22248107756189_wp,  8.57559920837242_wp,  7.03672928400636_wp, &
      & -1.76362632094559_wp,  0.64297750017893_wp, 14.40186664953440_wp, &
      &  5.83962338438617_wp, 10.13957405739921_wp,  8.57608868644902_wp, &
      & -0.31075648460643_wp, 10.41940511029398_wp, 12.90511986423551_wp, &
      &  0.36884840129967_wp,  7.95718616550666_wp, 13.74008757124138_wp, &
      &  5.13899142512075_wp,  1.27780458134152_wp,  7.74094984443455_wp, &
      & -0.43135113991668_wp,  7.08361972732341_wp, 16.04852770237523_wp, &
      &  5.85948725126428_wp,  2.12703296273982_wp,  5.39640022648409_wp, &
      &  7.98202471665911_wp,  9.37739667910611_wp,  4.56827791982260_wp, &
      &  2.58521074631042_wp,  6.23620255015096_wp,  0.63970266218240_wp, &
      & -7.37287492489702_wp,  6.83831475498402_wp, 13.72263180036320_wp, &
      & -0.33592003395561_wp,  6.47879071615157_wp,  7.13632985901710_wp, &
      & -9.65567668851609_wp,  2.93064163769937_wp, 14.45885460740137_wp, &
      &  0.44488747834642_wp,  5.60046986687546_wp,  4.70073641648637_wp, &
      &-10.35051844686700_wp,  3.77330440377895_wp, 16.93226888683481_wp, &
      & -2.67740619557892_wp,  2.59670085855714_wp,  7.83370501910091_wp, &
      & -8.03719513476919_wp,  7.73259684150050_wp, 16.06666801328784_wp, &
      & -1.95562894546284_wp,  1.67604311050164_wp,  5.51722154256241_wp, &
      &  1.91027980799140_wp,  7.09482655347090_wp,  3.06297438409324_wp, &
      & -8.19632876596279_wp,  4.41220669444970_wp, 12.90802915938188_wp, &
      &  8.06439765037426_wp,  1.37696801270714_wp,  1.32184680649995_wp, &
      &  0.30124800389322_wp,  2.27996651956055_wp,  1.45618778825843_wp, &
      & 10.17812550339424_wp,  8.63514254729744_wp,  0.55738672304122_wp, &
      &  7.30609985362735_wp,  0.55954890552544_wp,  3.73485042789423_wp, &
      & -1.86811945051845_wp,  5.02597852648671_wp,  8.64608290497032_wp, &
      & -0.39194685290862_wp,  3.14311853344512_wp,  3.87021821970436_wp, &
      &  0.34646325134892_wp, 11.08026825220325_wp, 11.06935462688116_wp, &
      &  5.28103743231810_wp,  9.50927498680126_wp, 10.45446654094714_wp, &
      & -2.26789230598577_wp,  2.53398791223687_wp, 13.76284852738644_wp, &
      &  7.76197910212852_wp,  6.69602202299020_wp,  7.67831443128346_wp, &
      &  9.83472737946160_wp,  5.89603575648236_wp,  3.55909477405225_wp, &
      &  5.35897816853694_wp,  4.02359018773944_wp,  4.77723376533483_wp, &
      &  7.54765344185304_wp,  3.26594082001099_wp,  0.68608024951558_wp, &
      & -6.21783671028862_wp,  7.98605223285624_wp, 12.46513175710043_wp, &
      & -3.91122464008637_wp,  1.48801747180495_wp,  9.05064606532323_wp, &
      & -2.61184696118149_wp, 11.15215041991691_wp,  4.87991477050055_wp, &
      & -0.35396280221428_wp,  0.43333061103608_wp,  0.82127690631710_wp, &
      &  2.51885417779770_wp,  8.96364971376210_wp,  3.67683565997560_wp, &
      &  5.60295691186475_wp,  1.08117572257201_wp, 13.82633961558057_wp, &
      &  0.27625154557950_wp,  8.34365186719871_wp,  7.75926129035577_wp, &
      & -7.67466304247654_wp,  3.74319313352409_wp, 11.03238946502150_wp, &
      & -2.49374222683799_wp,  5.73280097562707_wp, 10.47517387698890_wp, &
      &  1.55529102490321_wp,  6.77673381130491_wp, 12.54402499606942_wp, &
      &  4.03617550083984_wp,  2.48938700817517_wp,  8.98510135702578_wp, &
      & -7.43995626641283_wp,  9.60481600971518_wp, 16.67385502383442_wp, &
      &  0.10707944729661_wp,  5.20506134431823_wp, 16.69319327980730_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 15.90091151_wp,  0.00000000_wp,  0.00000000_wp, &
      &  0.00000000_wp, 11.32002641_wp,  0.00000000_wp, &
      &-12.11389533_wp,  0.00000000_wp, 17.11350086_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine x05


end module testsuite_structure
