/* XRACER (C) 1999-2000 Richard W.M. Jones <rich@annexia.org> and other AUTHORS
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: backdrop.c,v 1.2 2000/03/19 23:48:47 rich Exp $
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <GL/glu.h>

#include "xracer.h"
#include "xracer-math.h"
#include "xracer-log.h"
#include "common.h"

/* The backdrop is modelled as a cylinder which surrounds the
 * craft at a fixed distance. We initialize a display list
 * describing this cylinder around the craft, and then each
 * frame use the model view to transform it around the craft.
 * This is not very efficient, because we end up always
 * drawing the entire backdrop (XXX).
 */

#define BACKDROP_TOP 3
#define BACKDROP_BASE -3
#define BACKDROP_NR_SECTS 40
#define BACKDROP_RADIUS 8
#define BACKDROP_X_SCALE 2

static int backdrop_dlist;
static GLuint backdrop_tex;

int
backdrop_load ()
{
  int i;
  GLfloat t;
  GLfloat a;

  backdrop_tex = xrTextureLoad ("backdrops/mountains4.jpg",
				"backdrops/mountains4-alpha.jpg",
				0, 0, 0);
  if (backdrop_tex == 0)
    {
      xrLog (LOG_ERROR, "cannot load backdrop texture: backdrops/mountains4*.jpg");
      return -1;
    }

  backdrop_dlist = glGenLists (1);
  if (backdrop_dlist == 0)
    {
      xrLog (LOG_ERROR,
	     "glGenLists: cannot allocate display list: %s",
	     gluErrorString (glGetError ()));
      return -1;
    }

  glNewList (backdrop_dlist, GL_COMPILE);

  glBindTexture (GL_TEXTURE_2D, backdrop_tex);
  glBegin (GL_QUAD_STRIP);

  for (i = 0; i <= BACKDROP_NR_SECTS; ++i)
    {
      t = (GLfloat) i / BACKDROP_NR_SECTS; /* Parameter from 0 to 1. */
      a = 2 * M_PI * t;		/* Angle from 0 to 2*pi. */

      glTexCoord2f (t * BACKDROP_X_SCALE, 0);
      glVertex3f (BACKDROP_RADIUS * cos (a), BACKDROP_RADIUS * sin (a),
		  BACKDROP_BASE);
      glTexCoord2f (t * BACKDROP_X_SCALE, 1);
      glVertex3f (BACKDROP_RADIUS * cos (a), BACKDROP_RADIUS * sin (a),
		  BACKDROP_TOP);
    }

  glEnd ();
  glEndList ();

  return 0;
}

void
backdrop_unload ()
{
  glDeleteLists (backdrop_dlist, 1);
  xrTextureUnload (backdrop_tex);
}

void
backdrop_display (const GLfloat *posn)
{
  glMatrixMode (GL_MODELVIEW);
  glPushMatrix ();
  glTranslatef (posn[0], posn[1], posn[2]);

  glCallList (backdrop_dlist);

  glPopMatrix ();
}
