//
// Copyright (c) 2017 Joey <majunjiev@gmail.com>.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
package ovirtsdk

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/http/httputil"
	"net/url"
	"strings"
)

//
//
type FollowService struct {
	BaseService
}

func NewFollowService(connection *Connection, path string) *FollowService {
	var result FollowService
	result.connection = connection
	result.path = path
	return &result
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FollowService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *FollowService) String() string {
	return fmt.Sprintf("FollowService:%s", op.path)
}

//
// This annotation is intended to specify what oVirt area is the annotated concept related to. Currently the following
// areas are in use, and they are closely related to the oVirt teams, but not necessarily the same:
// - Infrastructure
// - Network
// - SLA
// - Storage
// - Virtualization
// A concept may be associated to more than one area, or to no area.
// The value of this annotation is intended for reporting only, and it doesn't affect at all the generated code or the
// validity of the model
//
type AreaService struct {
	BaseService
}

func NewAreaService(connection *Connection, path string) *AreaService {
	var result AreaService
	result.connection = connection
	result.path = path
	return &result
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AreaService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AreaService) String() string {
	return fmt.Sprintf("AreaService:%s", op.path)
}

//
// Manages the ser of labels attached to a network or to a host NIC.
//
type NetworkLabelsService struct {
	BaseService
}

func NewNetworkLabelsService(connection *Connection, path string) *NetworkLabelsService {
	var result NetworkLabelsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Attaches label to logical network.
// You can attach labels to a logical network to automate the association of that logical network with physical host
// network interfaces to which the same label has been attached.
// For example, to attach the label `mylabel` to a logical network having id `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/networks/123/networklabels
// ----
// With a request body like this:
// [source,xml]
// ----
// <network_label id="mylabel"/>
// ----
//
type NetworkLabelsServiceAddRequest struct {
	NetworkLabelsService *NetworkLabelsService
	header               map[string]string
	query                map[string]string
	label                *NetworkLabel
}

func (p *NetworkLabelsServiceAddRequest) Header(key, value string) *NetworkLabelsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkLabelsServiceAddRequest) Query(key, value string) *NetworkLabelsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkLabelsServiceAddRequest) Label(label *NetworkLabel) *NetworkLabelsServiceAddRequest {
	p.label = label
	return p
}

func (p *NetworkLabelsServiceAddRequest) Send() (*NetworkLabelsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkLabelsService.connection.URL(), p.NetworkLabelsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkLabelWriteOne(writer, p.label, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &NetworkLabelsServiceAddResponse{label: result}, nil
}

func (p *NetworkLabelsServiceAddRequest) MustSend() *NetworkLabelsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Attaches label to logical network.
// You can attach labels to a logical network to automate the association of that logical network with physical host
// network interfaces to which the same label has been attached.
// For example, to attach the label `mylabel` to a logical network having id `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/networks/123/networklabels
// ----
// With a request body like this:
// [source,xml]
// ----
// <network_label id="mylabel"/>
// ----
//
type NetworkLabelsServiceAddResponse struct {
	label *NetworkLabel
}

func (p *NetworkLabelsServiceAddResponse) Label() (*NetworkLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *NetworkLabelsServiceAddResponse) MustLabel() *NetworkLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Attaches label to logical network.
// You can attach labels to a logical network to automate the association of that logical network with physical host
// network interfaces to which the same label has been attached.
// For example, to attach the label `mylabel` to a logical network having id `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/networks/123/networklabels
// ----
// With a request body like this:
// [source,xml]
// ----
// <network_label id="mylabel"/>
// ----
//
func (p *NetworkLabelsService) Add() *NetworkLabelsServiceAddRequest {
	return &NetworkLabelsServiceAddRequest{NetworkLabelsService: p}
}

//
// Returns the list of labels attached to the network or host NIC.
// The order of the returned list of labels isn't guaranteed.
//
type NetworkLabelsServiceListRequest struct {
	NetworkLabelsService *NetworkLabelsService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *NetworkLabelsServiceListRequest) Header(key, value string) *NetworkLabelsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkLabelsServiceListRequest) Query(key, value string) *NetworkLabelsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkLabelsServiceListRequest) Follow(follow string) *NetworkLabelsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *NetworkLabelsServiceListRequest) Max(max int64) *NetworkLabelsServiceListRequest {
	p.max = &max
	return p
}

func (p *NetworkLabelsServiceListRequest) Send() (*NetworkLabelsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkLabelsService.connection.URL(), p.NetworkLabelsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkLabelsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkLabelsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkLabelsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkLabelsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkLabelsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkLabelReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &NetworkLabelsServiceListResponse{labels: result}, nil
}

func (p *NetworkLabelsServiceListRequest) MustSend() *NetworkLabelsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of labels attached to the network or host NIC.
// The order of the returned list of labels isn't guaranteed.
//
type NetworkLabelsServiceListResponse struct {
	labels *NetworkLabelSlice
}

func (p *NetworkLabelsServiceListResponse) Labels() (*NetworkLabelSlice, bool) {
	if p.labels != nil {
		return p.labels, true
	}
	return nil, false
}

func (p *NetworkLabelsServiceListResponse) MustLabels() *NetworkLabelSlice {
	if p.labels == nil {
		panic("labels in response does not exist")
	}
	return p.labels
}

//
// Returns the list of labels attached to the network or host NIC.
// The order of the returned list of labels isn't guaranteed.
//
func (p *NetworkLabelsService) List() *NetworkLabelsServiceListRequest {
	return &NetworkLabelsServiceListRequest{NetworkLabelsService: p}
}

//
//
func (op *NetworkLabelsService) LabelService(id string) *NetworkLabelService {
	return NewNetworkLabelService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *NetworkLabelsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.LabelService(path), nil
	}
	return op.LabelService(path[:index]).Service(path[index+1:])
}

func (op *NetworkLabelsService) String() string {
	return fmt.Sprintf("NetworkLabelsService:%s", op.path)
}

//
// Manages the filters used by an scheduling policy.
//
type FiltersService struct {
	BaseService
}

func NewFiltersService(connection *Connection, path string) *FiltersService {
	var result FiltersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a filter to a specified user defined scheduling policy.
//
type FiltersServiceAddRequest struct {
	FiltersService *FiltersService
	header         map[string]string
	query          map[string]string
	filter         *Filter
}

func (p *FiltersServiceAddRequest) Header(key, value string) *FiltersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FiltersServiceAddRequest) Query(key, value string) *FiltersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FiltersServiceAddRequest) Filter(filter *Filter) *FiltersServiceAddRequest {
	p.filter = filter
	return p
}

func (p *FiltersServiceAddRequest) Send() (*FiltersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FiltersService.connection.URL(), p.FiltersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLFilterWriteOne(writer, p.filter, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FiltersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FiltersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FiltersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FiltersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FiltersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLFilterReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &FiltersServiceAddResponse{filter: result}, nil
}

func (p *FiltersServiceAddRequest) MustSend() *FiltersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a filter to a specified user defined scheduling policy.
//
type FiltersServiceAddResponse struct {
	filter *Filter
}

func (p *FiltersServiceAddResponse) Filter() (*Filter, bool) {
	if p.filter != nil {
		return p.filter, true
	}
	return nil, false
}

func (p *FiltersServiceAddResponse) MustFilter() *Filter {
	if p.filter == nil {
		panic("filter in response does not exist")
	}
	return p.filter
}

//
// Add a filter to a specified user defined scheduling policy.
//
func (p *FiltersService) Add() *FiltersServiceAddRequest {
	return &FiltersServiceAddRequest{FiltersService: p}
}

//
// Returns the list of filters used by the scheduling policy.
// The order of the returned list of filters isn't guaranteed.
//
type FiltersServiceListRequest struct {
	FiltersService *FiltersService
	header         map[string]string
	query          map[string]string
	filter         *bool
	follow         *string
	max            *int64
}

func (p *FiltersServiceListRequest) Header(key, value string) *FiltersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FiltersServiceListRequest) Query(key, value string) *FiltersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FiltersServiceListRequest) Filter(filter bool) *FiltersServiceListRequest {
	p.filter = &filter
	return p
}

func (p *FiltersServiceListRequest) Follow(follow string) *FiltersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *FiltersServiceListRequest) Max(max int64) *FiltersServiceListRequest {
	p.max = &max
	return p
}

func (p *FiltersServiceListRequest) Send() (*FiltersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FiltersService.connection.URL(), p.FiltersService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FiltersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FiltersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FiltersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FiltersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FiltersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLFilterReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &FiltersServiceListResponse{filters: result}, nil
}

func (p *FiltersServiceListRequest) MustSend() *FiltersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of filters used by the scheduling policy.
// The order of the returned list of filters isn't guaranteed.
//
type FiltersServiceListResponse struct {
	filters *FilterSlice
}

func (p *FiltersServiceListResponse) Filters() (*FilterSlice, bool) {
	if p.filters != nil {
		return p.filters, true
	}
	return nil, false
}

func (p *FiltersServiceListResponse) MustFilters() *FilterSlice {
	if p.filters == nil {
		panic("filters in response does not exist")
	}
	return p.filters
}

//
// Returns the list of filters used by the scheduling policy.
// The order of the returned list of filters isn't guaranteed.
//
func (p *FiltersService) List() *FiltersServiceListRequest {
	return &FiltersServiceListRequest{FiltersService: p}
}

//
//
func (op *FiltersService) FilterService(id string) *FilterService {
	return NewFilterService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FiltersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.FilterService(path), nil
	}
	return op.FilterService(path[:index]).Service(path[index+1:])
}

func (op *FiltersService) String() string {
	return fmt.Sprintf("FiltersService:%s", op.path)
}

//
// This service manages a collection of parameters for network filters.
//
type NicNetworkFilterParametersService struct {
	BaseService
}

func NewNicNetworkFilterParametersService(connection *Connection, path string) *NicNetworkFilterParametersService {
	var result NicNetworkFilterParametersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a network filter parameter.
// For example, to add the parameter for the network filter on NIC `456` of
// virtual machine `789` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/vms/789/nics/456/networkfilterparameters
// ----
// With a request body like this:
// [source,xml]
// ----
// <network_filter_parameter>
//   <name>IP</name>
//   <value>10.0.1.2</value>
// </network_filter_parameter>
// ----
//
type NicNetworkFilterParametersServiceAddRequest struct {
	NicNetworkFilterParametersService *NicNetworkFilterParametersService
	header                            map[string]string
	query                             map[string]string
	parameter                         *NetworkFilterParameter
}

func (p *NicNetworkFilterParametersServiceAddRequest) Header(key, value string) *NicNetworkFilterParametersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NicNetworkFilterParametersServiceAddRequest) Query(key, value string) *NicNetworkFilterParametersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NicNetworkFilterParametersServiceAddRequest) Parameter(parameter *NetworkFilterParameter) *NicNetworkFilterParametersServiceAddRequest {
	p.parameter = parameter
	return p
}

func (p *NicNetworkFilterParametersServiceAddRequest) Send() (*NicNetworkFilterParametersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NicNetworkFilterParametersService.connection.URL(), p.NicNetworkFilterParametersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNetworkFilterParameterWriteOne(writer, p.parameter, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NicNetworkFilterParametersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NicNetworkFilterParametersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NicNetworkFilterParametersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NicNetworkFilterParametersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NicNetworkFilterParametersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkFilterParameterReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &NicNetworkFilterParametersServiceAddResponse{parameter: result}, nil
}

func (p *NicNetworkFilterParametersServiceAddRequest) MustSend() *NicNetworkFilterParametersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a network filter parameter.
// For example, to add the parameter for the network filter on NIC `456` of
// virtual machine `789` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/vms/789/nics/456/networkfilterparameters
// ----
// With a request body like this:
// [source,xml]
// ----
// <network_filter_parameter>
//   <name>IP</name>
//   <value>10.0.1.2</value>
// </network_filter_parameter>
// ----
//
type NicNetworkFilterParametersServiceAddResponse struct {
	parameter *NetworkFilterParameter
}

func (p *NicNetworkFilterParametersServiceAddResponse) Parameter() (*NetworkFilterParameter, bool) {
	if p.parameter != nil {
		return p.parameter, true
	}
	return nil, false
}

func (p *NicNetworkFilterParametersServiceAddResponse) MustParameter() *NetworkFilterParameter {
	if p.parameter == nil {
		panic("parameter in response does not exist")
	}
	return p.parameter
}

//
// Add a network filter parameter.
// For example, to add the parameter for the network filter on NIC `456` of
// virtual machine `789` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/vms/789/nics/456/networkfilterparameters
// ----
// With a request body like this:
// [source,xml]
// ----
// <network_filter_parameter>
//   <name>IP</name>
//   <value>10.0.1.2</value>
// </network_filter_parameter>
// ----
//
func (p *NicNetworkFilterParametersService) Add() *NicNetworkFilterParametersServiceAddRequest {
	return &NicNetworkFilterParametersServiceAddRequest{NicNetworkFilterParametersService: p}
}

//
// Retrieves the representations of the network filter parameters.
// The order of the returned list of network filters isn't guaranteed.
//
type NicNetworkFilterParametersServiceListRequest struct {
	NicNetworkFilterParametersService *NicNetworkFilterParametersService
	header                            map[string]string
	query                             map[string]string
	follow                            *string
}

func (p *NicNetworkFilterParametersServiceListRequest) Header(key, value string) *NicNetworkFilterParametersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NicNetworkFilterParametersServiceListRequest) Query(key, value string) *NicNetworkFilterParametersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NicNetworkFilterParametersServiceListRequest) Follow(follow string) *NicNetworkFilterParametersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *NicNetworkFilterParametersServiceListRequest) Send() (*NicNetworkFilterParametersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NicNetworkFilterParametersService.connection.URL(), p.NicNetworkFilterParametersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NicNetworkFilterParametersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NicNetworkFilterParametersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NicNetworkFilterParametersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NicNetworkFilterParametersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NicNetworkFilterParametersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkFilterParameterReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &NicNetworkFilterParametersServiceListResponse{parameters: result}, nil
}

func (p *NicNetworkFilterParametersServiceListRequest) MustSend() *NicNetworkFilterParametersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the representations of the network filter parameters.
// The order of the returned list of network filters isn't guaranteed.
//
type NicNetworkFilterParametersServiceListResponse struct {
	parameters *NetworkFilterParameterSlice
}

func (p *NicNetworkFilterParametersServiceListResponse) Parameters() (*NetworkFilterParameterSlice, bool) {
	if p.parameters != nil {
		return p.parameters, true
	}
	return nil, false
}

func (p *NicNetworkFilterParametersServiceListResponse) MustParameters() *NetworkFilterParameterSlice {
	if p.parameters == nil {
		panic("parameters in response does not exist")
	}
	return p.parameters
}

//
// Retrieves the representations of the network filter parameters.
// The order of the returned list of network filters isn't guaranteed.
//
func (p *NicNetworkFilterParametersService) List() *NicNetworkFilterParametersServiceListRequest {
	return &NicNetworkFilterParametersServiceListRequest{NicNetworkFilterParametersService: p}
}

//
// Reference to the service that manages a specific network filter parameter.
//
func (op *NicNetworkFilterParametersService) ParameterService(id string) *NicNetworkFilterParameterService {
	return NewNicNetworkFilterParameterService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *NicNetworkFilterParametersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ParameterService(path), nil
	}
	return op.ParameterService(path[:index]).Service(path[index+1:])
}

func (op *NicNetworkFilterParametersService) String() string {
	return fmt.Sprintf("NicNetworkFilterParametersService:%s", op.path)
}

//
// Lists the backups of a virtual machine.
//
type VmBackupsService struct {
	BaseService
}

func NewVmBackupsService(connection *Connection, path string) *VmBackupsService {
	var result VmBackupsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a new backup entity to a virtual machine.
// For example, to start a new incremental backup of a virtual machine
// since checkpoint id `previous-checkpoint-uuid`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/vms/123/backups
// ----
// With a request body like this:
// [source,xml]
// ----
// POST /ovirt-engine/api/vms/123/backups
// <backup>
//   <from_checkpoint_id>previous-checkpoint-uuid</from_checkpoint_id>
//   <disks>
//       <disk id="disk-uuid" />
//       ...
//   </disks>
// </backup>
// ----
// The response body:
// [source,xml]
// ----
// <backup id="backup-uuid">
//     <from_checkpoint_id>previous-checkpoint-uuid</from_checkpoint_id>
//     <to_checkpoint_id>new-checkpoint-uuid</to_checkpoint_id>
//     <disks>
//         <disk id="disk-uuid" />
//         ...
//         ...
//     </disks>
//     <status>initializing</status>
//     <creation_date>
// </backup>
// ----
//
type VmBackupsServiceAddRequest struct {
	VmBackupsService *VmBackupsService
	header           map[string]string
	query            map[string]string
	backup           *Backup
}

func (p *VmBackupsServiceAddRequest) Header(key, value string) *VmBackupsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmBackupsServiceAddRequest) Query(key, value string) *VmBackupsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmBackupsServiceAddRequest) Backup(backup *Backup) *VmBackupsServiceAddRequest {
	p.backup = backup
	return p
}

func (p *VmBackupsServiceAddRequest) Send() (*VmBackupsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmBackupsService.connection.URL(), p.VmBackupsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLBackupWriteOne(writer, p.backup, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmBackupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmBackupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmBackupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmBackupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmBackupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBackupReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmBackupsServiceAddResponse{backup: result}, nil
}

func (p *VmBackupsServiceAddRequest) MustSend() *VmBackupsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a new backup entity to a virtual machine.
// For example, to start a new incremental backup of a virtual machine
// since checkpoint id `previous-checkpoint-uuid`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/vms/123/backups
// ----
// With a request body like this:
// [source,xml]
// ----
// POST /ovirt-engine/api/vms/123/backups
// <backup>
//   <from_checkpoint_id>previous-checkpoint-uuid</from_checkpoint_id>
//   <disks>
//       <disk id="disk-uuid" />
//       ...
//   </disks>
// </backup>
// ----
// The response body:
// [source,xml]
// ----
// <backup id="backup-uuid">
//     <from_checkpoint_id>previous-checkpoint-uuid</from_checkpoint_id>
//     <to_checkpoint_id>new-checkpoint-uuid</to_checkpoint_id>
//     <disks>
//         <disk id="disk-uuid" />
//         ...
//         ...
//     </disks>
//     <status>initializing</status>
//     <creation_date>
// </backup>
// ----
//
type VmBackupsServiceAddResponse struct {
	backup *Backup
}

func (p *VmBackupsServiceAddResponse) Backup() (*Backup, bool) {
	if p.backup != nil {
		return p.backup, true
	}
	return nil, false
}

func (p *VmBackupsServiceAddResponse) MustBackup() *Backup {
	if p.backup == nil {
		panic("backup in response does not exist")
	}
	return p.backup
}

//
// Adds a new backup entity to a virtual machine.
// For example, to start a new incremental backup of a virtual machine
// since checkpoint id `previous-checkpoint-uuid`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/vms/123/backups
// ----
// With a request body like this:
// [source,xml]
// ----
// POST /ovirt-engine/api/vms/123/backups
// <backup>
//   <from_checkpoint_id>previous-checkpoint-uuid</from_checkpoint_id>
//   <disks>
//       <disk id="disk-uuid" />
//       ...
//   </disks>
// </backup>
// ----
// The response body:
// [source,xml]
// ----
// <backup id="backup-uuid">
//     <from_checkpoint_id>previous-checkpoint-uuid</from_checkpoint_id>
//     <to_checkpoint_id>new-checkpoint-uuid</to_checkpoint_id>
//     <disks>
//         <disk id="disk-uuid" />
//         ...
//         ...
//     </disks>
//     <status>initializing</status>
//     <creation_date>
// </backup>
// ----
//
func (p *VmBackupsService) Add() *VmBackupsServiceAddRequest {
	return &VmBackupsServiceAddRequest{VmBackupsService: p}
}

//
// The list of virtual machine backups.
//
type VmBackupsServiceListRequest struct {
	VmBackupsService *VmBackupsService
	header           map[string]string
	query            map[string]string
	follow           *string
	max              *int64
}

func (p *VmBackupsServiceListRequest) Header(key, value string) *VmBackupsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmBackupsServiceListRequest) Query(key, value string) *VmBackupsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmBackupsServiceListRequest) Follow(follow string) *VmBackupsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *VmBackupsServiceListRequest) Max(max int64) *VmBackupsServiceListRequest {
	p.max = &max
	return p
}

func (p *VmBackupsServiceListRequest) Send() (*VmBackupsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmBackupsService.connection.URL(), p.VmBackupsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmBackupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmBackupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmBackupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmBackupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmBackupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBackupReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &VmBackupsServiceListResponse{backups: result}, nil
}

func (p *VmBackupsServiceListRequest) MustSend() *VmBackupsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// The list of virtual machine backups.
//
type VmBackupsServiceListResponse struct {
	backups *BackupSlice
}

func (p *VmBackupsServiceListResponse) Backups() (*BackupSlice, bool) {
	if p.backups != nil {
		return p.backups, true
	}
	return nil, false
}

func (p *VmBackupsServiceListResponse) MustBackups() *BackupSlice {
	if p.backups == nil {
		panic("backups in response does not exist")
	}
	return p.backups
}

//
// The list of virtual machine backups.
//
func (p *VmBackupsService) List() *VmBackupsServiceListRequest {
	return &VmBackupsServiceListRequest{VmBackupsService: p}
}

//
// Returns a reference to the service that manages a specific VM backup.
//
func (op *VmBackupsService) BackupService(id string) *VmBackupService {
	return NewVmBackupService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmBackupsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.BackupService(path), nil
	}
	return op.BackupService(path[:index]).Service(path[index+1:])
}

func (op *VmBackupsService) String() string {
	return fmt.Sprintf("VmBackupsService:%s", op.path)
}

//
// A service to manage fence agent for a specific host.
//
type FenceAgentService struct {
	BaseService
}

func NewFenceAgentService(connection *Connection, path string) *FenceAgentService {
	var result FenceAgentService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets details of this fence agent.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/fenceagents/0
// ----
// And here is sample response:
// [source,xml]
// ----
// <agent id="0">
//   <type>apc</type>
//   <order>1</order>
//   <ip>192.168.1.101</ip>
//   <user>user</user>
//   <password>xxx</password>
//   <port>9</port>
//   <options>name1=value1, name2=value2</options>
// </agent>
// ----
//
type FenceAgentServiceGetRequest struct {
	FenceAgentService *FenceAgentService
	header            map[string]string
	query             map[string]string
	follow            *string
}

func (p *FenceAgentServiceGetRequest) Header(key, value string) *FenceAgentServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FenceAgentServiceGetRequest) Query(key, value string) *FenceAgentServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FenceAgentServiceGetRequest) Follow(follow string) *FenceAgentServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *FenceAgentServiceGetRequest) Send() (*FenceAgentServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FenceAgentService.connection.URL(), p.FenceAgentService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FenceAgentService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FenceAgentService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FenceAgentService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FenceAgentService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FenceAgentService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAgentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &FenceAgentServiceGetResponse{agent: result}, nil
}

func (p *FenceAgentServiceGetRequest) MustSend() *FenceAgentServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets details of this fence agent.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/fenceagents/0
// ----
// And here is sample response:
// [source,xml]
// ----
// <agent id="0">
//   <type>apc</type>
//   <order>1</order>
//   <ip>192.168.1.101</ip>
//   <user>user</user>
//   <password>xxx</password>
//   <port>9</port>
//   <options>name1=value1, name2=value2</options>
// </agent>
// ----
//
type FenceAgentServiceGetResponse struct {
	agent *Agent
}

func (p *FenceAgentServiceGetResponse) Agent() (*Agent, bool) {
	if p.agent != nil {
		return p.agent, true
	}
	return nil, false
}

func (p *FenceAgentServiceGetResponse) MustAgent() *Agent {
	if p.agent == nil {
		panic("agent in response does not exist")
	}
	return p.agent
}

//
// Gets details of this fence agent.
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/fenceagents/0
// ----
// And here is sample response:
// [source,xml]
// ----
// <agent id="0">
//   <type>apc</type>
//   <order>1</order>
//   <ip>192.168.1.101</ip>
//   <user>user</user>
//   <password>xxx</password>
//   <port>9</port>
//   <options>name1=value1, name2=value2</options>
// </agent>
// ----
//
func (p *FenceAgentService) Get() *FenceAgentServiceGetRequest {
	return &FenceAgentServiceGetRequest{FenceAgentService: p}
}

//
// Removes a fence agent for a specific host.
// [source]
// ----
// DELETE /ovirt-engine/api/hosts/123/fenceagents/0
// ----
//
type FenceAgentServiceRemoveRequest struct {
	FenceAgentService *FenceAgentService
	header            map[string]string
	query             map[string]string
	async             *bool
}

func (p *FenceAgentServiceRemoveRequest) Header(key, value string) *FenceAgentServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FenceAgentServiceRemoveRequest) Query(key, value string) *FenceAgentServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FenceAgentServiceRemoveRequest) Async(async bool) *FenceAgentServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *FenceAgentServiceRemoveRequest) Send() (*FenceAgentServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FenceAgentService.connection.URL(), p.FenceAgentService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FenceAgentService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FenceAgentService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FenceAgentService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FenceAgentService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FenceAgentService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(FenceAgentServiceRemoveResponse), nil
}

func (p *FenceAgentServiceRemoveRequest) MustSend() *FenceAgentServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a fence agent for a specific host.
// [source]
// ----
// DELETE /ovirt-engine/api/hosts/123/fenceagents/0
// ----
//
type FenceAgentServiceRemoveResponse struct {
}

//
// Removes a fence agent for a specific host.
// [source]
// ----
// DELETE /ovirt-engine/api/hosts/123/fenceagents/0
// ----
//
func (p *FenceAgentService) Remove() *FenceAgentServiceRemoveRequest {
	return &FenceAgentServiceRemoveRequest{FenceAgentService: p}
}

//
// Update a fencing-agent.
//
type FenceAgentServiceUpdateRequest struct {
	FenceAgentService *FenceAgentService
	header            map[string]string
	query             map[string]string
	agent             *Agent
	async             *bool
}

func (p *FenceAgentServiceUpdateRequest) Header(key, value string) *FenceAgentServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *FenceAgentServiceUpdateRequest) Query(key, value string) *FenceAgentServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *FenceAgentServiceUpdateRequest) Agent(agent *Agent) *FenceAgentServiceUpdateRequest {
	p.agent = agent
	return p
}

func (p *FenceAgentServiceUpdateRequest) Async(async bool) *FenceAgentServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *FenceAgentServiceUpdateRequest) Send() (*FenceAgentServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.FenceAgentService.connection.URL(), p.FenceAgentService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAgentWriteOne(writer, p.agent, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.FenceAgentService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.FenceAgentService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.FenceAgentService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.FenceAgentService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.FenceAgentService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAgentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &FenceAgentServiceUpdateResponse{agent: result}, nil
}

func (p *FenceAgentServiceUpdateRequest) MustSend() *FenceAgentServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update a fencing-agent.
//
type FenceAgentServiceUpdateResponse struct {
	agent *Agent
}

func (p *FenceAgentServiceUpdateResponse) Agent() (*Agent, bool) {
	if p.agent != nil {
		return p.agent, true
	}
	return nil, false
}

func (p *FenceAgentServiceUpdateResponse) MustAgent() *Agent {
	if p.agent == nil {
		panic("agent in response does not exist")
	}
	return p.agent
}

//
// Update a fencing-agent.
//
func (p *FenceAgentService) Update() *FenceAgentServiceUpdateRequest {
	return &FenceAgentServiceUpdateRequest{FenceAgentService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *FenceAgentService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *FenceAgentService) String() string {
	return fmt.Sprintf("FenceAgentService:%s", op.path)
}

//
//
type TemplateDisksService struct {
	BaseService
}

func NewTemplateDisksService(connection *Connection, path string) *TemplateDisksService {
	var result TemplateDisksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of disks of the template.
// The order of the returned list of disks isn't guaranteed.
//
type TemplateDisksServiceListRequest struct {
	TemplateDisksService *TemplateDisksService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *TemplateDisksServiceListRequest) Header(key, value string) *TemplateDisksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TemplateDisksServiceListRequest) Query(key, value string) *TemplateDisksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TemplateDisksServiceListRequest) Follow(follow string) *TemplateDisksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *TemplateDisksServiceListRequest) Max(max int64) *TemplateDisksServiceListRequest {
	p.max = &max
	return p
}

func (p *TemplateDisksServiceListRequest) Send() (*TemplateDisksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TemplateDisksService.connection.URL(), p.TemplateDisksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TemplateDisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TemplateDisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TemplateDisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TemplateDisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TemplateDisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &TemplateDisksServiceListResponse{disks: result}, nil
}

func (p *TemplateDisksServiceListRequest) MustSend() *TemplateDisksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of disks of the template.
// The order of the returned list of disks isn't guaranteed.
//
type TemplateDisksServiceListResponse struct {
	disks *DiskSlice
}

func (p *TemplateDisksServiceListResponse) Disks() (*DiskSlice, bool) {
	if p.disks != nil {
		return p.disks, true
	}
	return nil, false
}

func (p *TemplateDisksServiceListResponse) MustDisks() *DiskSlice {
	if p.disks == nil {
		panic("disks in response does not exist")
	}
	return p.disks
}

//
// Returns the list of disks of the template.
// The order of the returned list of disks isn't guaranteed.
//
func (p *TemplateDisksService) List() *TemplateDisksServiceListRequest {
	return &TemplateDisksServiceListRequest{TemplateDisksService: p}
}

//
//
func (op *TemplateDisksService) DiskService(id string) *TemplateDiskService {
	return NewTemplateDiskService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *TemplateDisksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DiskService(path), nil
	}
	return op.DiskService(path[:index]).Service(path[index+1:])
}

func (op *TemplateDisksService) String() string {
	return fmt.Sprintf("TemplateDisksService:%s", op.path)
}

//
//
type HostHookService struct {
	BaseService
}

func NewHostHookService(connection *Connection, path string) *HostHookService {
	var result HostHookService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type HostHookServiceGetRequest struct {
	HostHookService *HostHookService
	header          map[string]string
	query           map[string]string
	follow          *string
}

func (p *HostHookServiceGetRequest) Header(key, value string) *HostHookServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostHookServiceGetRequest) Query(key, value string) *HostHookServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostHookServiceGetRequest) Follow(follow string) *HostHookServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *HostHookServiceGetRequest) Send() (*HostHookServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostHookService.connection.URL(), p.HostHookService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostHookService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostHookService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostHookService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostHookService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostHookService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHookReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &HostHookServiceGetResponse{hook: result}, nil
}

func (p *HostHookServiceGetRequest) MustSend() *HostHookServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type HostHookServiceGetResponse struct {
	hook *Hook
}

func (p *HostHookServiceGetResponse) Hook() (*Hook, bool) {
	if p.hook != nil {
		return p.hook, true
	}
	return nil, false
}

func (p *HostHookServiceGetResponse) MustHook() *Hook {
	if p.hook == nil {
		panic("hook in response does not exist")
	}
	return p.hook
}

//
//
func (p *HostHookService) Get() *HostHookServiceGetRequest {
	return &HostHookServiceGetRequest{HostHookService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostHookService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *HostHookService) String() string {
	return fmt.Sprintf("HostHookService:%s", op.path)
}

//
// Manages the collection of disks available inside an storage domain that is attached to a data center.
//
type AttachedStorageDomainDisksService struct {
	BaseService
}

func NewAttachedStorageDomainDisksService(connection *Connection, path string) *AttachedStorageDomainDisksService {
	var result AttachedStorageDomainDisksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds or registers a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To add a new disk use the <<services/disks/methods/add, add>>
// operation of the service that manages the disks of the system. To register an unregistered disk use the
// <<services/attached_storage_domain_disk/methods/register, register>> operation of the service that manages
// that disk.
//
type AttachedStorageDomainDisksServiceAddRequest struct {
	AttachedStorageDomainDisksService *AttachedStorageDomainDisksService
	header                            map[string]string
	query                             map[string]string
	disk                              *Disk
	unregistered                      *bool
}

func (p *AttachedStorageDomainDisksServiceAddRequest) Header(key, value string) *AttachedStorageDomainDisksServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDisksServiceAddRequest) Query(key, value string) *AttachedStorageDomainDisksServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDisksServiceAddRequest) Disk(disk *Disk) *AttachedStorageDomainDisksServiceAddRequest {
	p.disk = disk
	return p
}

func (p *AttachedStorageDomainDisksServiceAddRequest) Unregistered(unregistered bool) *AttachedStorageDomainDisksServiceAddRequest {
	p.unregistered = &unregistered
	return p
}

func (p *AttachedStorageDomainDisksServiceAddRequest) Send() (*AttachedStorageDomainDisksServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainDisksService.connection.URL(), p.AttachedStorageDomainDisksService.path)
	values := make(url.Values)
	if p.unregistered != nil {
		values["unregistered"] = []string{fmt.Sprintf("%v", *p.unregistered)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskWriteOne(writer, p.disk, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainDisksServiceAddResponse{disk: result}, nil
}

func (p *AttachedStorageDomainDisksServiceAddRequest) MustSend() *AttachedStorageDomainDisksServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds or registers a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To add a new disk use the <<services/disks/methods/add, add>>
// operation of the service that manages the disks of the system. To register an unregistered disk use the
// <<services/attached_storage_domain_disk/methods/register, register>> operation of the service that manages
// that disk.
//
type AttachedStorageDomainDisksServiceAddResponse struct {
	disk *Disk
}

func (p *AttachedStorageDomainDisksServiceAddResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *AttachedStorageDomainDisksServiceAddResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Adds or registers a disk.
// IMPORTANT: Since version 4.2 of the engine this operation is deprecated, and preserved only for backwards
// compatibility. It will be removed in the future. To add a new disk use the <<services/disks/methods/add, add>>
// operation of the service that manages the disks of the system. To register an unregistered disk use the
// <<services/attached_storage_domain_disk/methods/register, register>> operation of the service that manages
// that disk.
//
func (p *AttachedStorageDomainDisksService) Add() *AttachedStorageDomainDisksServiceAddRequest {
	return &AttachedStorageDomainDisksServiceAddRequest{AttachedStorageDomainDisksService: p}
}

//
// Retrieve the list of disks that are available in the storage domain.
//
type AttachedStorageDomainDisksServiceListRequest struct {
	AttachedStorageDomainDisksService *AttachedStorageDomainDisksService
	header                            map[string]string
	query                             map[string]string
	follow                            *string
	max                               *int64
}

func (p *AttachedStorageDomainDisksServiceListRequest) Header(key, value string) *AttachedStorageDomainDisksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainDisksServiceListRequest) Query(key, value string) *AttachedStorageDomainDisksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainDisksServiceListRequest) Follow(follow string) *AttachedStorageDomainDisksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AttachedStorageDomainDisksServiceListRequest) Max(max int64) *AttachedStorageDomainDisksServiceListRequest {
	p.max = &max
	return p
}

func (p *AttachedStorageDomainDisksServiceListRequest) Send() (*AttachedStorageDomainDisksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainDisksService.connection.URL(), p.AttachedStorageDomainDisksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainDisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainDisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainDisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainDisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainDisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainDisksServiceListResponse{disks: result}, nil
}

func (p *AttachedStorageDomainDisksServiceListRequest) MustSend() *AttachedStorageDomainDisksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieve the list of disks that are available in the storage domain.
//
type AttachedStorageDomainDisksServiceListResponse struct {
	disks *DiskSlice
}

func (p *AttachedStorageDomainDisksServiceListResponse) Disks() (*DiskSlice, bool) {
	if p.disks != nil {
		return p.disks, true
	}
	return nil, false
}

func (p *AttachedStorageDomainDisksServiceListResponse) MustDisks() *DiskSlice {
	if p.disks == nil {
		panic("disks in response does not exist")
	}
	return p.disks
}

//
// Retrieve the list of disks that are available in the storage domain.
//
func (p *AttachedStorageDomainDisksService) List() *AttachedStorageDomainDisksServiceListRequest {
	return &AttachedStorageDomainDisksServiceListRequest{AttachedStorageDomainDisksService: p}
}

//
// Reference to the service that manages a specific disk.
//
func (op *AttachedStorageDomainDisksService) DiskService(id string) *AttachedStorageDomainDiskService {
	return NewAttachedStorageDomainDiskService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AttachedStorageDomainDisksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DiskService(path), nil
	}
	return op.DiskService(path[:index]).Service(path[index+1:])
}

func (op *AttachedStorageDomainDisksService) String() string {
	return fmt.Sprintf("AttachedStorageDomainDisksService:%s", op.path)
}

//
// A service to manage clusters.
//
type ClustersService struct {
	BaseService
}

func NewClustersService(connection *Connection, path string) *ClustersService {
	var result ClustersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new cluster.
// This requires the `name`, `cpu.type`, and `data_center` attributes. Identify the data center with either the `id`
// or `name` attribute.
// [source]
// ----
// POST /ovirt-engine/api/clusters
// ----
// With a request body like this:
// [source,xml]
// ----
// <cluster>
//   <name>mycluster</name>
//   <cpu>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <data_center id="123"/>
// </cluster>
// ----
// To create a cluster with an external network provider to be deployed on
// every host that is added to the cluster, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters
// ----
// With a request body containing a reference to the desired provider:
// [source,xml]
// ----
// <cluster>
//   <name>mycluster</name>
//   <cpu>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <data_center id="123"/>
//   <external_network_providers>
//     <external_provider name="ovirt-provider-ovn"/>
//   </external_network_providers>
// </cluster>
// ----
//
type ClustersServiceAddRequest struct {
	ClustersService *ClustersService
	header          map[string]string
	query           map[string]string
	cluster         *Cluster
}

func (p *ClustersServiceAddRequest) Header(key, value string) *ClustersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClustersServiceAddRequest) Query(key, value string) *ClustersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClustersServiceAddRequest) Cluster(cluster *Cluster) *ClustersServiceAddRequest {
	p.cluster = cluster
	return p
}

func (p *ClustersServiceAddRequest) Send() (*ClustersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClustersService.connection.URL(), p.ClustersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLClusterWriteOne(writer, p.cluster, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClustersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClustersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClustersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClustersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClustersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ClustersServiceAddResponse{cluster: result}, nil
}

func (p *ClustersServiceAddRequest) MustSend() *ClustersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new cluster.
// This requires the `name`, `cpu.type`, and `data_center` attributes. Identify the data center with either the `id`
// or `name` attribute.
// [source]
// ----
// POST /ovirt-engine/api/clusters
// ----
// With a request body like this:
// [source,xml]
// ----
// <cluster>
//   <name>mycluster</name>
//   <cpu>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <data_center id="123"/>
// </cluster>
// ----
// To create a cluster with an external network provider to be deployed on
// every host that is added to the cluster, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters
// ----
// With a request body containing a reference to the desired provider:
// [source,xml]
// ----
// <cluster>
//   <name>mycluster</name>
//   <cpu>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <data_center id="123"/>
//   <external_network_providers>
//     <external_provider name="ovirt-provider-ovn"/>
//   </external_network_providers>
// </cluster>
// ----
//
type ClustersServiceAddResponse struct {
	cluster *Cluster
}

func (p *ClustersServiceAddResponse) Cluster() (*Cluster, bool) {
	if p.cluster != nil {
		return p.cluster, true
	}
	return nil, false
}

func (p *ClustersServiceAddResponse) MustCluster() *Cluster {
	if p.cluster == nil {
		panic("cluster in response does not exist")
	}
	return p.cluster
}

//
// Creates a new cluster.
// This requires the `name`, `cpu.type`, and `data_center` attributes. Identify the data center with either the `id`
// or `name` attribute.
// [source]
// ----
// POST /ovirt-engine/api/clusters
// ----
// With a request body like this:
// [source,xml]
// ----
// <cluster>
//   <name>mycluster</name>
//   <cpu>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <data_center id="123"/>
// </cluster>
// ----
// To create a cluster with an external network provider to be deployed on
// every host that is added to the cluster, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters
// ----
// With a request body containing a reference to the desired provider:
// [source,xml]
// ----
// <cluster>
//   <name>mycluster</name>
//   <cpu>
//     <type>Intel Nehalem Family</type>
//   </cpu>
//   <data_center id="123"/>
//   <external_network_providers>
//     <external_provider name="ovirt-provider-ovn"/>
//   </external_network_providers>
// </cluster>
// ----
//
func (p *ClustersService) Add() *ClustersServiceAddRequest {
	return &ClustersServiceAddRequest{ClustersService: p}
}

//
// Returns the list of clusters of the system.
// The order of the returned clusters is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type ClustersServiceListRequest struct {
	ClustersService *ClustersService
	header          map[string]string
	query           map[string]string
	caseSensitive   *bool
	filter          *bool
	follow          *string
	max             *int64
	search          *string
}

func (p *ClustersServiceListRequest) Header(key, value string) *ClustersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ClustersServiceListRequest) Query(key, value string) *ClustersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ClustersServiceListRequest) CaseSensitive(caseSensitive bool) *ClustersServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *ClustersServiceListRequest) Filter(filter bool) *ClustersServiceListRequest {
	p.filter = &filter
	return p
}

func (p *ClustersServiceListRequest) Follow(follow string) *ClustersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ClustersServiceListRequest) Max(max int64) *ClustersServiceListRequest {
	p.max = &max
	return p
}

func (p *ClustersServiceListRequest) Search(search string) *ClustersServiceListRequest {
	p.search = &search
	return p
}

func (p *ClustersServiceListRequest) Send() (*ClustersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ClustersService.connection.URL(), p.ClustersService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ClustersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ClustersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ClustersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ClustersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ClustersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLClusterReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ClustersServiceListResponse{clusters: result}, nil
}

func (p *ClustersServiceListRequest) MustSend() *ClustersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of clusters of the system.
// The order of the returned clusters is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type ClustersServiceListResponse struct {
	clusters *ClusterSlice
}

func (p *ClustersServiceListResponse) Clusters() (*ClusterSlice, bool) {
	if p.clusters != nil {
		return p.clusters, true
	}
	return nil, false
}

func (p *ClustersServiceListResponse) MustClusters() *ClusterSlice {
	if p.clusters == nil {
		panic("clusters in response does not exist")
	}
	return p.clusters
}

//
// Returns the list of clusters of the system.
// The order of the returned clusters is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
func (p *ClustersService) List() *ClustersServiceListRequest {
	return &ClustersServiceListRequest{ClustersService: p}
}

//
// A reference to the service that manages a specific cluster.
//
func (op *ClustersService) ClusterService(id string) *ClusterService {
	return NewClusterService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ClustersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ClusterService(path), nil
	}
	return op.ClusterService(path[:index]).Service(path[index+1:])
}

func (op *ClustersService) String() string {
	return fmt.Sprintf("ClustersService:%s", op.path)
}

//
// A service to manage a specific event-subscription in the system.
//
type EventSubscriptionService struct {
	BaseService
}

func NewEventSubscriptionService(connection *Connection, path string) *EventSubscriptionService {
	var result EventSubscriptionService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets the information about the event-subscription.
// For example to retrieve the information about the subscription of user '123' to
// the event 'vm_console_detected':
// ....
// GET /ovirt-engine/api/users/123/vm_console_detected
// ....
// [source,xml]
// ----
// <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/vm_console_detected">
//   <event>vm_console_detected</event>
//   <notification_method>smtp</notification_method>
//   <user href="/ovirt-engine/api/users/123" id="123"/>
//   <address>a@b.com</address>
// </event-subscription>
// ----
//
type EventSubscriptionServiceGetRequest struct {
	EventSubscriptionService *EventSubscriptionService
	header                   map[string]string
	query                    map[string]string
}

func (p *EventSubscriptionServiceGetRequest) Header(key, value string) *EventSubscriptionServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EventSubscriptionServiceGetRequest) Query(key, value string) *EventSubscriptionServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EventSubscriptionServiceGetRequest) Send() (*EventSubscriptionServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EventSubscriptionService.connection.URL(), p.EventSubscriptionService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EventSubscriptionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EventSubscriptionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EventSubscriptionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EventSubscriptionService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EventSubscriptionService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLEventSubscriptionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &EventSubscriptionServiceGetResponse{eventSubscription: result}, nil
}

func (p *EventSubscriptionServiceGetRequest) MustSend() *EventSubscriptionServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the information about the event-subscription.
// For example to retrieve the information about the subscription of user '123' to
// the event 'vm_console_detected':
// ....
// GET /ovirt-engine/api/users/123/vm_console_detected
// ....
// [source,xml]
// ----
// <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/vm_console_detected">
//   <event>vm_console_detected</event>
//   <notification_method>smtp</notification_method>
//   <user href="/ovirt-engine/api/users/123" id="123"/>
//   <address>a@b.com</address>
// </event-subscription>
// ----
//
type EventSubscriptionServiceGetResponse struct {
	eventSubscription *EventSubscription
}

func (p *EventSubscriptionServiceGetResponse) EventSubscription() (*EventSubscription, bool) {
	if p.eventSubscription != nil {
		return p.eventSubscription, true
	}
	return nil, false
}

func (p *EventSubscriptionServiceGetResponse) MustEventSubscription() *EventSubscription {
	if p.eventSubscription == nil {
		panic("eventSubscription in response does not exist")
	}
	return p.eventSubscription
}

//
// Gets the information about the event-subscription.
// For example to retrieve the information about the subscription of user '123' to
// the event 'vm_console_detected':
// ....
// GET /ovirt-engine/api/users/123/vm_console_detected
// ....
// [source,xml]
// ----
// <event-subscription href="/ovirt-engine/api/users/123/event-subscriptions/vm_console_detected">
//   <event>vm_console_detected</event>
//   <notification_method>smtp</notification_method>
//   <user href="/ovirt-engine/api/users/123" id="123"/>
//   <address>a@b.com</address>
// </event-subscription>
// ----
//
func (p *EventSubscriptionService) Get() *EventSubscriptionServiceGetRequest {
	return &EventSubscriptionServiceGetRequest{EventSubscriptionService: p}
}

//
// Removes the event-subscription from the system.
// For example to remove user 123's subscription to `vm_console_detected` event:
// ....
// DELETE /ovirt-engine/api/users/123/vm_console_detected
// ....
//
type EventSubscriptionServiceRemoveRequest struct {
	EventSubscriptionService *EventSubscriptionService
	header                   map[string]string
	query                    map[string]string
	async                    *bool
}

func (p *EventSubscriptionServiceRemoveRequest) Header(key, value string) *EventSubscriptionServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EventSubscriptionServiceRemoveRequest) Query(key, value string) *EventSubscriptionServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EventSubscriptionServiceRemoveRequest) Async(async bool) *EventSubscriptionServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *EventSubscriptionServiceRemoveRequest) Send() (*EventSubscriptionServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EventSubscriptionService.connection.URL(), p.EventSubscriptionService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EventSubscriptionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EventSubscriptionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EventSubscriptionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EventSubscriptionService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EventSubscriptionService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(EventSubscriptionServiceRemoveResponse), nil
}

func (p *EventSubscriptionServiceRemoveRequest) MustSend() *EventSubscriptionServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the event-subscription from the system.
// For example to remove user 123's subscription to `vm_console_detected` event:
// ....
// DELETE /ovirt-engine/api/users/123/vm_console_detected
// ....
//
type EventSubscriptionServiceRemoveResponse struct {
}

//
// Removes the event-subscription from the system.
// For example to remove user 123's subscription to `vm_console_detected` event:
// ....
// DELETE /ovirt-engine/api/users/123/vm_console_detected
// ....
//
func (p *EventSubscriptionService) Remove() *EventSubscriptionServiceRemoveRequest {
	return &EventSubscriptionServiceRemoveRequest{EventSubscriptionService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *EventSubscriptionService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *EventSubscriptionService) String() string {
	return fmt.Sprintf("EventSubscriptionService:%s", op.path)
}

//
//
type WeightsService struct {
	BaseService
}

func NewWeightsService(connection *Connection, path string) *WeightsService {
	var result WeightsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a weight to a specified user defined scheduling policy.
//
type WeightsServiceAddRequest struct {
	WeightsService *WeightsService
	header         map[string]string
	query          map[string]string
	weight         *Weight
}

func (p *WeightsServiceAddRequest) Header(key, value string) *WeightsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *WeightsServiceAddRequest) Query(key, value string) *WeightsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *WeightsServiceAddRequest) Weight(weight *Weight) *WeightsServiceAddRequest {
	p.weight = weight
	return p
}

func (p *WeightsServiceAddRequest) Send() (*WeightsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.WeightsService.connection.URL(), p.WeightsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLWeightWriteOne(writer, p.weight, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.WeightsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.WeightsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.WeightsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.WeightsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.WeightsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLWeightReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &WeightsServiceAddResponse{weight: result}, nil
}

func (p *WeightsServiceAddRequest) MustSend() *WeightsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a weight to a specified user defined scheduling policy.
//
type WeightsServiceAddResponse struct {
	weight *Weight
}

func (p *WeightsServiceAddResponse) Weight() (*Weight, bool) {
	if p.weight != nil {
		return p.weight, true
	}
	return nil, false
}

func (p *WeightsServiceAddResponse) MustWeight() *Weight {
	if p.weight == nil {
		panic("weight in response does not exist")
	}
	return p.weight
}

//
// Add a weight to a specified user defined scheduling policy.
//
func (p *WeightsService) Add() *WeightsServiceAddRequest {
	return &WeightsServiceAddRequest{WeightsService: p}
}

//
// Returns the list of weights.
// The order of the returned list of weights isn't guaranteed.
//
type WeightsServiceListRequest struct {
	WeightsService *WeightsService
	header         map[string]string
	query          map[string]string
	filter         *bool
	follow         *string
	max            *int64
}

func (p *WeightsServiceListRequest) Header(key, value string) *WeightsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *WeightsServiceListRequest) Query(key, value string) *WeightsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *WeightsServiceListRequest) Filter(filter bool) *WeightsServiceListRequest {
	p.filter = &filter
	return p
}

func (p *WeightsServiceListRequest) Follow(follow string) *WeightsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *WeightsServiceListRequest) Max(max int64) *WeightsServiceListRequest {
	p.max = &max
	return p
}

func (p *WeightsServiceListRequest) Send() (*WeightsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.WeightsService.connection.URL(), p.WeightsService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.WeightsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.WeightsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.WeightsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.WeightsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.WeightsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLWeightReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &WeightsServiceListResponse{weights: result}, nil
}

func (p *WeightsServiceListRequest) MustSend() *WeightsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of weights.
// The order of the returned list of weights isn't guaranteed.
//
type WeightsServiceListResponse struct {
	weights *WeightSlice
}

func (p *WeightsServiceListResponse) Weights() (*WeightSlice, bool) {
	if p.weights != nil {
		return p.weights, true
	}
	return nil, false
}

func (p *WeightsServiceListResponse) MustWeights() *WeightSlice {
	if p.weights == nil {
		panic("weights in response does not exist")
	}
	return p.weights
}

//
// Returns the list of weights.
// The order of the returned list of weights isn't guaranteed.
//
func (p *WeightsService) List() *WeightsServiceListRequest {
	return &WeightsServiceListRequest{WeightsService: p}
}

//
//
func (op *WeightsService) WeightService(id string) *WeightService {
	return NewWeightService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *WeightsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.WeightService(path), nil
	}
	return op.WeightService(path[:index]).Service(path[index+1:])
}

func (op *WeightsService) String() string {
	return fmt.Sprintf("WeightsService:%s", op.path)
}

//
//
type OpenstackImageService struct {
	BaseService
}

func NewOpenstackImageService(connection *Connection, path string) *OpenstackImageService {
	var result OpenstackImageService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackImageServiceGetRequest struct {
	OpenstackImageService *OpenstackImageService
	header                map[string]string
	query                 map[string]string
	follow                *string
}

func (p *OpenstackImageServiceGetRequest) Header(key, value string) *OpenstackImageServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageServiceGetRequest) Query(key, value string) *OpenstackImageServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageServiceGetRequest) Follow(follow string) *OpenstackImageServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackImageServiceGetRequest) Send() (*OpenstackImageServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImageService.connection.URL(), p.OpenstackImageService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackImageReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackImageServiceGetResponse{image: result}, nil
}

func (p *OpenstackImageServiceGetRequest) MustSend() *OpenstackImageServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackImageServiceGetResponse struct {
	image *OpenStackImage
}

func (p *OpenstackImageServiceGetResponse) Image() (*OpenStackImage, bool) {
	if p.image != nil {
		return p.image, true
	}
	return nil, false
}

func (p *OpenstackImageServiceGetResponse) MustImage() *OpenStackImage {
	if p.image == nil {
		panic("image in response does not exist")
	}
	return p.image
}

//
//
func (p *OpenstackImageService) Get() *OpenstackImageServiceGetRequest {
	return &OpenstackImageServiceGetRequest{OpenstackImageService: p}
}

//
// Imports a virtual machine from a Glance image storage domain.
// For example, to import the image with identifier `456` from the
// storage domain with identifier `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/openstackimageproviders/123/images/456/import
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain>
//     <name>images0</name>
//   </storage_domain>
//   <cluster>
//     <name>images0</name>
//   </cluster>
// </action>
// ----
//
type OpenstackImageServiceImportRequest struct {
	OpenstackImageService *OpenstackImageService
	header                map[string]string
	query                 map[string]string
	async                 *bool
	cluster               *Cluster
	disk                  *Disk
	importAsTemplate      *bool
	storageDomain         *StorageDomain
	template              *Template
}

func (p *OpenstackImageServiceImportRequest) Header(key, value string) *OpenstackImageServiceImportRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageServiceImportRequest) Query(key, value string) *OpenstackImageServiceImportRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageServiceImportRequest) Async(async bool) *OpenstackImageServiceImportRequest {
	p.async = &async
	return p
}

func (p *OpenstackImageServiceImportRequest) Cluster(cluster *Cluster) *OpenstackImageServiceImportRequest {
	p.cluster = cluster
	return p
}

func (p *OpenstackImageServiceImportRequest) Disk(disk *Disk) *OpenstackImageServiceImportRequest {
	p.disk = disk
	return p
}

func (p *OpenstackImageServiceImportRequest) ImportAsTemplate(importAsTemplate bool) *OpenstackImageServiceImportRequest {
	p.importAsTemplate = &importAsTemplate
	return p
}

func (p *OpenstackImageServiceImportRequest) StorageDomain(storageDomain *StorageDomain) *OpenstackImageServiceImportRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *OpenstackImageServiceImportRequest) Template(template *Template) *OpenstackImageServiceImportRequest {
	p.template = template
	return p
}

func (p *OpenstackImageServiceImportRequest) Send() (*OpenstackImageServiceImportResponse, error) {
	rawURL := fmt.Sprintf("%s%s/import", p.OpenstackImageService.connection.URL(), p.OpenstackImageService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Cluster(p.cluster)
	actionBuilder.Disk(p.disk)
	if p.importAsTemplate != nil {
		actionBuilder.ImportAsTemplate(*p.importAsTemplate)
	}
	actionBuilder.StorageDomain(p.storageDomain)
	actionBuilder.Template(p.template)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackImageServiceImportResponse), nil
}

func (p *OpenstackImageServiceImportRequest) MustSend() *OpenstackImageServiceImportResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Imports a virtual machine from a Glance image storage domain.
// For example, to import the image with identifier `456` from the
// storage domain with identifier `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/openstackimageproviders/123/images/456/import
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain>
//     <name>images0</name>
//   </storage_domain>
//   <cluster>
//     <name>images0</name>
//   </cluster>
// </action>
// ----
//
type OpenstackImageServiceImportResponse struct {
}

//
// Imports a virtual machine from a Glance image storage domain.
// For example, to import the image with identifier `456` from the
// storage domain with identifier `123` send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/openstackimageproviders/123/images/456/import
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <storage_domain>
//     <name>images0</name>
//   </storage_domain>
//   <cluster>
//     <name>images0</name>
//   </cluster>
// </action>
// ----
//
func (p *OpenstackImageService) Import() *OpenstackImageServiceImportRequest {
	return &OpenstackImageServiceImportRequest{OpenstackImageService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackImageService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackImageService) String() string {
	return fmt.Sprintf("OpenstackImageService:%s", op.path)
}

//
// This service manages the OpenStack network provider.
//
type OpenstackNetworkProviderService struct {
	BaseService
}

func NewOpenstackNetworkProviderService(connection *Connection, path string) *OpenstackNetworkProviderService {
	var result OpenstackNetworkProviderService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the representation of the object managed by this service.
// For example, to get the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/openstacknetworkproviders/1234
// ----
//
type OpenstackNetworkProviderServiceGetRequest struct {
	OpenstackNetworkProviderService *OpenstackNetworkProviderService
	header                          map[string]string
	query                           map[string]string
	follow                          *string
}

func (p *OpenstackNetworkProviderServiceGetRequest) Header(key, value string) *OpenstackNetworkProviderServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceGetRequest) Query(key, value string) *OpenstackNetworkProviderServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceGetRequest) Follow(follow string) *OpenstackNetworkProviderServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackNetworkProviderServiceGetRequest) Send() (*OpenstackNetworkProviderServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworkProviderService.connection.URL(), p.OpenstackNetworkProviderService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackNetworkProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackNetworkProviderServiceGetResponse{provider: result}, nil
}

func (p *OpenstackNetworkProviderServiceGetRequest) MustSend() *OpenstackNetworkProviderServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the representation of the object managed by this service.
// For example, to get the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/openstacknetworkproviders/1234
// ----
//
type OpenstackNetworkProviderServiceGetResponse struct {
	provider *OpenStackNetworkProvider
}

func (p *OpenstackNetworkProviderServiceGetResponse) Provider() (*OpenStackNetworkProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackNetworkProviderServiceGetResponse) MustProvider() *OpenStackNetworkProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Returns the representation of the object managed by this service.
// For example, to get the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/openstacknetworkproviders/1234
// ----
//
func (p *OpenstackNetworkProviderService) Get() *OpenstackNetworkProviderServiceGetRequest {
	return &OpenstackNetworkProviderServiceGetRequest{OpenstackNetworkProviderService: p}
}

//
// Import the SSL certificates of the external host provider.
//
type OpenstackNetworkProviderServiceImportCertificatesRequest struct {
	OpenstackNetworkProviderService *OpenstackNetworkProviderService
	header                          map[string]string
	query                           map[string]string
	certificates                    *CertificateSlice
}

func (p *OpenstackNetworkProviderServiceImportCertificatesRequest) Header(key, value string) *OpenstackNetworkProviderServiceImportCertificatesRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceImportCertificatesRequest) Query(key, value string) *OpenstackNetworkProviderServiceImportCertificatesRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceImportCertificatesRequest) Certificates(certificates *CertificateSlice) *OpenstackNetworkProviderServiceImportCertificatesRequest {
	p.certificates = certificates
	return p
}

func (p *OpenstackNetworkProviderServiceImportCertificatesRequest) CertificatesOfAny(anys ...*Certificate) *OpenstackNetworkProviderServiceImportCertificatesRequest {
	if p.certificates == nil {
		p.certificates = new(CertificateSlice)
	}
	p.certificates.slice = append(p.certificates.slice, anys...)
	return p
}

func (p *OpenstackNetworkProviderServiceImportCertificatesRequest) Send() (*OpenstackNetworkProviderServiceImportCertificatesResponse, error) {
	rawURL := fmt.Sprintf("%s%s/importcertificates", p.OpenstackNetworkProviderService.connection.URL(), p.OpenstackNetworkProviderService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Certificates(p.certificates)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackNetworkProviderServiceImportCertificatesResponse), nil
}

func (p *OpenstackNetworkProviderServiceImportCertificatesRequest) MustSend() *OpenstackNetworkProviderServiceImportCertificatesResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Import the SSL certificates of the external host provider.
//
type OpenstackNetworkProviderServiceImportCertificatesResponse struct {
}

//
// Import the SSL certificates of the external host provider.
//
func (p *OpenstackNetworkProviderService) ImportCertificates() *OpenstackNetworkProviderServiceImportCertificatesRequest {
	return &OpenstackNetworkProviderServiceImportCertificatesRequest{OpenstackNetworkProviderService: p}
}

//
// Removes the provider.
// For example, to remove the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/openstacknetworkproviders/1234
// ----
//
type OpenstackNetworkProviderServiceRemoveRequest struct {
	OpenstackNetworkProviderService *OpenstackNetworkProviderService
	header                          map[string]string
	query                           map[string]string
	async                           *bool
}

func (p *OpenstackNetworkProviderServiceRemoveRequest) Header(key, value string) *OpenstackNetworkProviderServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceRemoveRequest) Query(key, value string) *OpenstackNetworkProviderServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceRemoveRequest) Async(async bool) *OpenstackNetworkProviderServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *OpenstackNetworkProviderServiceRemoveRequest) Send() (*OpenstackNetworkProviderServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworkProviderService.connection.URL(), p.OpenstackNetworkProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(OpenstackNetworkProviderServiceRemoveResponse), nil
}

func (p *OpenstackNetworkProviderServiceRemoveRequest) MustSend() *OpenstackNetworkProviderServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the provider.
// For example, to remove the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/openstacknetworkproviders/1234
// ----
//
type OpenstackNetworkProviderServiceRemoveResponse struct {
}

//
// Removes the provider.
// For example, to remove the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/openstacknetworkproviders/1234
// ----
//
func (p *OpenstackNetworkProviderService) Remove() *OpenstackNetworkProviderServiceRemoveRequest {
	return &OpenstackNetworkProviderServiceRemoveRequest{OpenstackNetworkProviderService: p}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type OpenstackNetworkProviderServiceTestConnectivityRequest struct {
	OpenstackNetworkProviderService *OpenstackNetworkProviderService
	header                          map[string]string
	query                           map[string]string
	async                           *bool
}

func (p *OpenstackNetworkProviderServiceTestConnectivityRequest) Header(key, value string) *OpenstackNetworkProviderServiceTestConnectivityRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceTestConnectivityRequest) Query(key, value string) *OpenstackNetworkProviderServiceTestConnectivityRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceTestConnectivityRequest) Async(async bool) *OpenstackNetworkProviderServiceTestConnectivityRequest {
	p.async = &async
	return p
}

func (p *OpenstackNetworkProviderServiceTestConnectivityRequest) Send() (*OpenstackNetworkProviderServiceTestConnectivityResponse, error) {
	rawURL := fmt.Sprintf("%s%s/testconnectivity", p.OpenstackNetworkProviderService.connection.URL(), p.OpenstackNetworkProviderService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackNetworkProviderServiceTestConnectivityResponse), nil
}

func (p *OpenstackNetworkProviderServiceTestConnectivityRequest) MustSend() *OpenstackNetworkProviderServiceTestConnectivityResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type OpenstackNetworkProviderServiceTestConnectivityResponse struct {
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
func (p *OpenstackNetworkProviderService) TestConnectivity() *OpenstackNetworkProviderServiceTestConnectivityRequest {
	return &OpenstackNetworkProviderServiceTestConnectivityRequest{OpenstackNetworkProviderService: p}
}

//
// Updates the provider.
// For example, to update `provider_name`, `requires_authentication`, `url`, `tenant_name` and `type` properties,
// for the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/openstacknetworkproviders/1234
// ----
// With a request body like this:
// [source,xml]
// ----
// <openstack_network_provider>
//   <name>ovn-network-provider</name>
//   <requires_authentication>false</requires_authentication>
//   <url>http://some_server_url.domain.com:9696</url>
//   <tenant_name>oVirt</tenant_name>
//   <type>external</type>
// </openstack_network_provider>
// ----
//
type OpenstackNetworkProviderServiceUpdateRequest struct {
	OpenstackNetworkProviderService *OpenstackNetworkProviderService
	header                          map[string]string
	query                           map[string]string
	async                           *bool
	provider                        *OpenStackNetworkProvider
}

func (p *OpenstackNetworkProviderServiceUpdateRequest) Header(key, value string) *OpenstackNetworkProviderServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceUpdateRequest) Query(key, value string) *OpenstackNetworkProviderServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProviderServiceUpdateRequest) Async(async bool) *OpenstackNetworkProviderServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *OpenstackNetworkProviderServiceUpdateRequest) Provider(provider *OpenStackNetworkProvider) *OpenstackNetworkProviderServiceUpdateRequest {
	p.provider = provider
	return p
}

func (p *OpenstackNetworkProviderServiceUpdateRequest) Send() (*OpenstackNetworkProviderServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworkProviderService.connection.URL(), p.OpenstackNetworkProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackNetworkProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackNetworkProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackNetworkProviderServiceUpdateResponse{provider: result}, nil
}

func (p *OpenstackNetworkProviderServiceUpdateRequest) MustSend() *OpenstackNetworkProviderServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the provider.
// For example, to update `provider_name`, `requires_authentication`, `url`, `tenant_name` and `type` properties,
// for the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/openstacknetworkproviders/1234
// ----
// With a request body like this:
// [source,xml]
// ----
// <openstack_network_provider>
//   <name>ovn-network-provider</name>
//   <requires_authentication>false</requires_authentication>
//   <url>http://some_server_url.domain.com:9696</url>
//   <tenant_name>oVirt</tenant_name>
//   <type>external</type>
// </openstack_network_provider>
// ----
//
type OpenstackNetworkProviderServiceUpdateResponse struct {
	provider *OpenStackNetworkProvider
}

func (p *OpenstackNetworkProviderServiceUpdateResponse) Provider() (*OpenStackNetworkProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackNetworkProviderServiceUpdateResponse) MustProvider() *OpenStackNetworkProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Updates the provider.
// For example, to update `provider_name`, `requires_authentication`, `url`, `tenant_name` and `type` properties,
// for the OpenStack network provider with identifier `1234`, send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/openstacknetworkproviders/1234
// ----
// With a request body like this:
// [source,xml]
// ----
// <openstack_network_provider>
//   <name>ovn-network-provider</name>
//   <requires_authentication>false</requires_authentication>
//   <url>http://some_server_url.domain.com:9696</url>
//   <tenant_name>oVirt</tenant_name>
//   <type>external</type>
// </openstack_network_provider>
// ----
//
func (p *OpenstackNetworkProviderService) Update() *OpenstackNetworkProviderServiceUpdateRequest {
	return &OpenstackNetworkProviderServiceUpdateRequest{OpenstackNetworkProviderService: p}
}

//
// A service to view certificates for this external provider.
//
func (op *OpenstackNetworkProviderService) CertificatesService() *ExternalProviderCertificatesService {
	return NewExternalProviderCertificatesService(op.connection, fmt.Sprintf("%s/certificates", op.path))
}

//
// Reference to OpenStack networks service.
//
func (op *OpenstackNetworkProviderService) NetworksService() *OpenstackNetworksService {
	return NewOpenstackNetworksService(op.connection, fmt.Sprintf("%s/networks", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackNetworkProviderService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "certificates" {
		return op.CertificatesService(), nil
	}
	if strings.HasPrefix(path, "certificates/") {
		return op.CertificatesService().Service(path[13:])
	}
	if path == "networks" {
		return op.NetworksService(), nil
	}
	if strings.HasPrefix(path, "networks/") {
		return op.NetworksService().Service(path[9:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackNetworkProviderService) String() string {
	return fmt.Sprintf("OpenstackNetworkProviderService:%s", op.path)
}

//
//
type OpenstackVolumeProvidersService struct {
	BaseService
}

func NewOpenstackVolumeProvidersService(connection *Connection, path string) *OpenstackVolumeProvidersService {
	var result OpenstackVolumeProvidersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a new volume provider.
// For example:
// [source]
// ----
// POST /ovirt-engine/api/openstackvolumeproviders
// ----
// With a request body like this:
// [source,xml]
// ----
// <openstack_volume_provider>
//   <name>mycinder</name>
//   <url>https://mycinder.example.com:8776</url>
//   <data_center>
//     <name>mydc</name>
//   </data_center>
//   <requires_authentication>true</requires_authentication>
//   <username>admin</username>
//   <password>mypassword</password>
//   <tenant_name>mytenant</tenant_name>
// </openstack_volume_provider>
// ----
//
type OpenstackVolumeProvidersServiceAddRequest struct {
	OpenstackVolumeProvidersService *OpenstackVolumeProvidersService
	header                          map[string]string
	query                           map[string]string
	provider                        *OpenStackVolumeProvider
}

func (p *OpenstackVolumeProvidersServiceAddRequest) Header(key, value string) *OpenstackVolumeProvidersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProvidersServiceAddRequest) Query(key, value string) *OpenstackVolumeProvidersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProvidersServiceAddRequest) Provider(provider *OpenStackVolumeProvider) *OpenstackVolumeProvidersServiceAddRequest {
	p.provider = provider
	return p
}

func (p *OpenstackVolumeProvidersServiceAddRequest) Send() (*OpenstackVolumeProvidersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeProvidersService.connection.URL(), p.OpenstackVolumeProvidersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackVolumeProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackVolumeProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeProvidersServiceAddResponse{provider: result}, nil
}

func (p *OpenstackVolumeProvidersServiceAddRequest) MustSend() *OpenstackVolumeProvidersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a new volume provider.
// For example:
// [source]
// ----
// POST /ovirt-engine/api/openstackvolumeproviders
// ----
// With a request body like this:
// [source,xml]
// ----
// <openstack_volume_provider>
//   <name>mycinder</name>
//   <url>https://mycinder.example.com:8776</url>
//   <data_center>
//     <name>mydc</name>
//   </data_center>
//   <requires_authentication>true</requires_authentication>
//   <username>admin</username>
//   <password>mypassword</password>
//   <tenant_name>mytenant</tenant_name>
// </openstack_volume_provider>
// ----
//
type OpenstackVolumeProvidersServiceAddResponse struct {
	provider *OpenStackVolumeProvider
}

func (p *OpenstackVolumeProvidersServiceAddResponse) Provider() (*OpenStackVolumeProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackVolumeProvidersServiceAddResponse) MustProvider() *OpenStackVolumeProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Adds a new volume provider.
// For example:
// [source]
// ----
// POST /ovirt-engine/api/openstackvolumeproviders
// ----
// With a request body like this:
// [source,xml]
// ----
// <openstack_volume_provider>
//   <name>mycinder</name>
//   <url>https://mycinder.example.com:8776</url>
//   <data_center>
//     <name>mydc</name>
//   </data_center>
//   <requires_authentication>true</requires_authentication>
//   <username>admin</username>
//   <password>mypassword</password>
//   <tenant_name>mytenant</tenant_name>
// </openstack_volume_provider>
// ----
//
func (p *OpenstackVolumeProvidersService) Add() *OpenstackVolumeProvidersServiceAddRequest {
	return &OpenstackVolumeProvidersServiceAddRequest{OpenstackVolumeProvidersService: p}
}

//
// Retrieves the list of volume providers.
// The order of the returned list of volume providers isn't guaranteed.
//
type OpenstackVolumeProvidersServiceListRequest struct {
	OpenstackVolumeProvidersService *OpenstackVolumeProvidersService
	header                          map[string]string
	query                           map[string]string
	follow                          *string
	max                             *int64
	search                          *string
}

func (p *OpenstackVolumeProvidersServiceListRequest) Header(key, value string) *OpenstackVolumeProvidersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProvidersServiceListRequest) Query(key, value string) *OpenstackVolumeProvidersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProvidersServiceListRequest) Follow(follow string) *OpenstackVolumeProvidersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackVolumeProvidersServiceListRequest) Max(max int64) *OpenstackVolumeProvidersServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackVolumeProvidersServiceListRequest) Search(search string) *OpenstackVolumeProvidersServiceListRequest {
	p.search = &search
	return p
}

func (p *OpenstackVolumeProvidersServiceListRequest) Send() (*OpenstackVolumeProvidersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeProvidersService.connection.URL(), p.OpenstackVolumeProvidersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackVolumeProviderReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeProvidersServiceListResponse{providers: result}, nil
}

func (p *OpenstackVolumeProvidersServiceListRequest) MustSend() *OpenstackVolumeProvidersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the list of volume providers.
// The order of the returned list of volume providers isn't guaranteed.
//
type OpenstackVolumeProvidersServiceListResponse struct {
	providers *OpenStackVolumeProviderSlice
}

func (p *OpenstackVolumeProvidersServiceListResponse) Providers() (*OpenStackVolumeProviderSlice, bool) {
	if p.providers != nil {
		return p.providers, true
	}
	return nil, false
}

func (p *OpenstackVolumeProvidersServiceListResponse) MustProviders() *OpenStackVolumeProviderSlice {
	if p.providers == nil {
		panic("providers in response does not exist")
	}
	return p.providers
}

//
// Retrieves the list of volume providers.
// The order of the returned list of volume providers isn't guaranteed.
//
func (p *OpenstackVolumeProvidersService) List() *OpenstackVolumeProvidersServiceListRequest {
	return &OpenstackVolumeProvidersServiceListRequest{OpenstackVolumeProvidersService: p}
}

//
//
func (op *OpenstackVolumeProvidersService) ProviderService(id string) *OpenstackVolumeProviderService {
	return NewOpenstackVolumeProviderService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackVolumeProvidersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProviderService(path), nil
	}
	return op.ProviderService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackVolumeProvidersService) String() string {
	return fmt.Sprintf("OpenstackVolumeProvidersService:%s", op.path)
}

//
// This service manages OpenStack network providers.
//
type OpenstackNetworkProvidersService struct {
	BaseService
}

func NewOpenstackNetworkProvidersService(connection *Connection, path string) *OpenstackNetworkProvidersService {
	var result OpenstackNetworkProvidersService
	result.connection = connection
	result.path = path
	return &result
}

//
// The operation adds a new network provider to the system.
// If the `type` property is not present, a default value of `NEUTRON` will be used.
//
type OpenstackNetworkProvidersServiceAddRequest struct {
	OpenstackNetworkProvidersService *OpenstackNetworkProvidersService
	header                           map[string]string
	query                            map[string]string
	provider                         *OpenStackNetworkProvider
}

func (p *OpenstackNetworkProvidersServiceAddRequest) Header(key, value string) *OpenstackNetworkProvidersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProvidersServiceAddRequest) Query(key, value string) *OpenstackNetworkProvidersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProvidersServiceAddRequest) Provider(provider *OpenStackNetworkProvider) *OpenstackNetworkProvidersServiceAddRequest {
	p.provider = provider
	return p
}

func (p *OpenstackNetworkProvidersServiceAddRequest) Send() (*OpenstackNetworkProvidersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworkProvidersService.connection.URL(), p.OpenstackNetworkProvidersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackNetworkProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackNetworkProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackNetworkProvidersServiceAddResponse{provider: result}, nil
}

func (p *OpenstackNetworkProvidersServiceAddRequest) MustSend() *OpenstackNetworkProvidersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// The operation adds a new network provider to the system.
// If the `type` property is not present, a default value of `NEUTRON` will be used.
//
type OpenstackNetworkProvidersServiceAddResponse struct {
	provider *OpenStackNetworkProvider
}

func (p *OpenstackNetworkProvidersServiceAddResponse) Provider() (*OpenStackNetworkProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackNetworkProvidersServiceAddResponse) MustProvider() *OpenStackNetworkProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// The operation adds a new network provider to the system.
// If the `type` property is not present, a default value of `NEUTRON` will be used.
//
func (p *OpenstackNetworkProvidersService) Add() *OpenstackNetworkProvidersServiceAddRequest {
	return &OpenstackNetworkProvidersServiceAddRequest{OpenstackNetworkProvidersService: p}
}

//
// Returns the list of providers.
// The order of the returned list of providers isn't guaranteed.
//
type OpenstackNetworkProvidersServiceListRequest struct {
	OpenstackNetworkProvidersService *OpenstackNetworkProvidersService
	header                           map[string]string
	query                            map[string]string
	follow                           *string
	max                              *int64
	search                           *string
}

func (p *OpenstackNetworkProvidersServiceListRequest) Header(key, value string) *OpenstackNetworkProvidersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkProvidersServiceListRequest) Query(key, value string) *OpenstackNetworkProvidersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkProvidersServiceListRequest) Follow(follow string) *OpenstackNetworkProvidersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackNetworkProvidersServiceListRequest) Max(max int64) *OpenstackNetworkProvidersServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackNetworkProvidersServiceListRequest) Search(search string) *OpenstackNetworkProvidersServiceListRequest {
	p.search = &search
	return p
}

func (p *OpenstackNetworkProvidersServiceListRequest) Send() (*OpenstackNetworkProvidersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworkProvidersService.connection.URL(), p.OpenstackNetworkProvidersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackNetworkProviderReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackNetworkProvidersServiceListResponse{providers: result}, nil
}

func (p *OpenstackNetworkProvidersServiceListRequest) MustSend() *OpenstackNetworkProvidersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of providers.
// The order of the returned list of providers isn't guaranteed.
//
type OpenstackNetworkProvidersServiceListResponse struct {
	providers *OpenStackNetworkProviderSlice
}

func (p *OpenstackNetworkProvidersServiceListResponse) Providers() (*OpenStackNetworkProviderSlice, bool) {
	if p.providers != nil {
		return p.providers, true
	}
	return nil, false
}

func (p *OpenstackNetworkProvidersServiceListResponse) MustProviders() *OpenStackNetworkProviderSlice {
	if p.providers == nil {
		panic("providers in response does not exist")
	}
	return p.providers
}

//
// Returns the list of providers.
// The order of the returned list of providers isn't guaranteed.
//
func (p *OpenstackNetworkProvidersService) List() *OpenstackNetworkProvidersServiceListRequest {
	return &OpenstackNetworkProvidersServiceListRequest{OpenstackNetworkProvidersService: p}
}

//
// Reference to OpenStack network provider service.
//
func (op *OpenstackNetworkProvidersService) ProviderService(id string) *OpenstackNetworkProviderService {
	return NewOpenstackNetworkProviderService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackNetworkProvidersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProviderService(path), nil
	}
	return op.ProviderService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackNetworkProvidersService) String() string {
	return fmt.Sprintf("OpenstackNetworkProvidersService:%s", op.path)
}

//
//
type OpenstackNetworksService struct {
	BaseService
}

func NewOpenstackNetworksService(connection *Connection, path string) *OpenstackNetworksService {
	var result OpenstackNetworksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of networks.
// The order of the returned list of networks isn't guaranteed.
//
type OpenstackNetworksServiceListRequest struct {
	OpenstackNetworksService *OpenstackNetworksService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
	max                      *int64
}

func (p *OpenstackNetworksServiceListRequest) Header(key, value string) *OpenstackNetworksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworksServiceListRequest) Query(key, value string) *OpenstackNetworksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworksServiceListRequest) Follow(follow string) *OpenstackNetworksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackNetworksServiceListRequest) Max(max int64) *OpenstackNetworksServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackNetworksServiceListRequest) Send() (*OpenstackNetworksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworksService.connection.URL(), p.OpenstackNetworksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackNetworkReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackNetworksServiceListResponse{networks: result}, nil
}

func (p *OpenstackNetworksServiceListRequest) MustSend() *OpenstackNetworksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of networks.
// The order of the returned list of networks isn't guaranteed.
//
type OpenstackNetworksServiceListResponse struct {
	networks *OpenStackNetworkSlice
}

func (p *OpenstackNetworksServiceListResponse) Networks() (*OpenStackNetworkSlice, bool) {
	if p.networks != nil {
		return p.networks, true
	}
	return nil, false
}

func (p *OpenstackNetworksServiceListResponse) MustNetworks() *OpenStackNetworkSlice {
	if p.networks == nil {
		panic("networks in response does not exist")
	}
	return p.networks
}

//
// Returns the list of networks.
// The order of the returned list of networks isn't guaranteed.
//
func (p *OpenstackNetworksService) List() *OpenstackNetworksServiceListRequest {
	return &OpenstackNetworksServiceListRequest{OpenstackNetworksService: p}
}

//
//
func (op *OpenstackNetworksService) NetworkService(id string) *OpenstackNetworkService {
	return NewOpenstackNetworkService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackNetworksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NetworkService(path), nil
	}
	return op.NetworkService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackNetworksService) String() string {
	return fmt.Sprintf("OpenstackNetworksService:%s", op.path)
}

//
//
type OpenstackVolumeProviderService struct {
	BaseService
}

func NewOpenstackVolumeProviderService(connection *Connection, path string) *OpenstackVolumeProviderService {
	var result OpenstackVolumeProviderService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackVolumeProviderServiceGetRequest struct {
	OpenstackVolumeProviderService *OpenstackVolumeProviderService
	header                         map[string]string
	query                          map[string]string
	follow                         *string
}

func (p *OpenstackVolumeProviderServiceGetRequest) Header(key, value string) *OpenstackVolumeProviderServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceGetRequest) Query(key, value string) *OpenstackVolumeProviderServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceGetRequest) Follow(follow string) *OpenstackVolumeProviderServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackVolumeProviderServiceGetRequest) Send() (*OpenstackVolumeProviderServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeProviderService.connection.URL(), p.OpenstackVolumeProviderService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackVolumeProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeProviderServiceGetResponse{provider: result}, nil
}

func (p *OpenstackVolumeProviderServiceGetRequest) MustSend() *OpenstackVolumeProviderServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackVolumeProviderServiceGetResponse struct {
	provider *OpenStackVolumeProvider
}

func (p *OpenstackVolumeProviderServiceGetResponse) Provider() (*OpenStackVolumeProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackVolumeProviderServiceGetResponse) MustProvider() *OpenStackVolumeProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
//
func (p *OpenstackVolumeProviderService) Get() *OpenstackVolumeProviderServiceGetRequest {
	return &OpenstackVolumeProviderServiceGetRequest{OpenstackVolumeProviderService: p}
}

//
// Import the SSL certificates of the external host provider.
//
type OpenstackVolumeProviderServiceImportCertificatesRequest struct {
	OpenstackVolumeProviderService *OpenstackVolumeProviderService
	header                         map[string]string
	query                          map[string]string
	certificates                   *CertificateSlice
}

func (p *OpenstackVolumeProviderServiceImportCertificatesRequest) Header(key, value string) *OpenstackVolumeProviderServiceImportCertificatesRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceImportCertificatesRequest) Query(key, value string) *OpenstackVolumeProviderServiceImportCertificatesRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceImportCertificatesRequest) Certificates(certificates *CertificateSlice) *OpenstackVolumeProviderServiceImportCertificatesRequest {
	p.certificates = certificates
	return p
}

func (p *OpenstackVolumeProviderServiceImportCertificatesRequest) CertificatesOfAny(anys ...*Certificate) *OpenstackVolumeProviderServiceImportCertificatesRequest {
	if p.certificates == nil {
		p.certificates = new(CertificateSlice)
	}
	p.certificates.slice = append(p.certificates.slice, anys...)
	return p
}

func (p *OpenstackVolumeProviderServiceImportCertificatesRequest) Send() (*OpenstackVolumeProviderServiceImportCertificatesResponse, error) {
	rawURL := fmt.Sprintf("%s%s/importcertificates", p.OpenstackVolumeProviderService.connection.URL(), p.OpenstackVolumeProviderService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Certificates(p.certificates)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackVolumeProviderServiceImportCertificatesResponse), nil
}

func (p *OpenstackVolumeProviderServiceImportCertificatesRequest) MustSend() *OpenstackVolumeProviderServiceImportCertificatesResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Import the SSL certificates of the external host provider.
//
type OpenstackVolumeProviderServiceImportCertificatesResponse struct {
}

//
// Import the SSL certificates of the external host provider.
//
func (p *OpenstackVolumeProviderService) ImportCertificates() *OpenstackVolumeProviderServiceImportCertificatesRequest {
	return &OpenstackVolumeProviderServiceImportCertificatesRequest{OpenstackVolumeProviderService: p}
}

//
//
type OpenstackVolumeProviderServiceRemoveRequest struct {
	OpenstackVolumeProviderService *OpenstackVolumeProviderService
	header                         map[string]string
	query                          map[string]string
	async                          *bool
	force                          *bool
}

func (p *OpenstackVolumeProviderServiceRemoveRequest) Header(key, value string) *OpenstackVolumeProviderServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceRemoveRequest) Query(key, value string) *OpenstackVolumeProviderServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceRemoveRequest) Async(async bool) *OpenstackVolumeProviderServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *OpenstackVolumeProviderServiceRemoveRequest) Force(force bool) *OpenstackVolumeProviderServiceRemoveRequest {
	p.force = &force
	return p
}

func (p *OpenstackVolumeProviderServiceRemoveRequest) Send() (*OpenstackVolumeProviderServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeProviderService.connection.URL(), p.OpenstackVolumeProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.force != nil {
		values["force"] = []string{fmt.Sprintf("%v", *p.force)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(OpenstackVolumeProviderServiceRemoveResponse), nil
}

func (p *OpenstackVolumeProviderServiceRemoveRequest) MustSend() *OpenstackVolumeProviderServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackVolumeProviderServiceRemoveResponse struct {
}

//
//
func (p *OpenstackVolumeProviderService) Remove() *OpenstackVolumeProviderServiceRemoveRequest {
	return &OpenstackVolumeProviderServiceRemoveRequest{OpenstackVolumeProviderService: p}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type OpenstackVolumeProviderServiceTestConnectivityRequest struct {
	OpenstackVolumeProviderService *OpenstackVolumeProviderService
	header                         map[string]string
	query                          map[string]string
	async                          *bool
}

func (p *OpenstackVolumeProviderServiceTestConnectivityRequest) Header(key, value string) *OpenstackVolumeProviderServiceTestConnectivityRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceTestConnectivityRequest) Query(key, value string) *OpenstackVolumeProviderServiceTestConnectivityRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceTestConnectivityRequest) Async(async bool) *OpenstackVolumeProviderServiceTestConnectivityRequest {
	p.async = &async
	return p
}

func (p *OpenstackVolumeProviderServiceTestConnectivityRequest) Send() (*OpenstackVolumeProviderServiceTestConnectivityResponse, error) {
	rawURL := fmt.Sprintf("%s%s/testconnectivity", p.OpenstackVolumeProviderService.connection.URL(), p.OpenstackVolumeProviderService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackVolumeProviderServiceTestConnectivityResponse), nil
}

func (p *OpenstackVolumeProviderServiceTestConnectivityRequest) MustSend() *OpenstackVolumeProviderServiceTestConnectivityResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type OpenstackVolumeProviderServiceTestConnectivityResponse struct {
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
func (p *OpenstackVolumeProviderService) TestConnectivity() *OpenstackVolumeProviderServiceTestConnectivityRequest {
	return &OpenstackVolumeProviderServiceTestConnectivityRequest{OpenstackVolumeProviderService: p}
}

//
// Update the specified OpenStack volume provider in the system.
//
type OpenstackVolumeProviderServiceUpdateRequest struct {
	OpenstackVolumeProviderService *OpenstackVolumeProviderService
	header                         map[string]string
	query                          map[string]string
	async                          *bool
	provider                       *OpenStackVolumeProvider
}

func (p *OpenstackVolumeProviderServiceUpdateRequest) Header(key, value string) *OpenstackVolumeProviderServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceUpdateRequest) Query(key, value string) *OpenstackVolumeProviderServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeProviderServiceUpdateRequest) Async(async bool) *OpenstackVolumeProviderServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *OpenstackVolumeProviderServiceUpdateRequest) Provider(provider *OpenStackVolumeProvider) *OpenstackVolumeProviderServiceUpdateRequest {
	p.provider = provider
	return p
}

func (p *OpenstackVolumeProviderServiceUpdateRequest) Send() (*OpenstackVolumeProviderServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeProviderService.connection.URL(), p.OpenstackVolumeProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackVolumeProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackVolumeProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeProviderServiceUpdateResponse{provider: result}, nil
}

func (p *OpenstackVolumeProviderServiceUpdateRequest) MustSend() *OpenstackVolumeProviderServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified OpenStack volume provider in the system.
//
type OpenstackVolumeProviderServiceUpdateResponse struct {
	provider *OpenStackVolumeProvider
}

func (p *OpenstackVolumeProviderServiceUpdateResponse) Provider() (*OpenStackVolumeProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackVolumeProviderServiceUpdateResponse) MustProvider() *OpenStackVolumeProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Update the specified OpenStack volume provider in the system.
//
func (p *OpenstackVolumeProviderService) Update() *OpenstackVolumeProviderServiceUpdateRequest {
	return &OpenstackVolumeProviderServiceUpdateRequest{OpenstackVolumeProviderService: p}
}

//
//
func (op *OpenstackVolumeProviderService) AuthenticationKeysService() *OpenstackVolumeAuthenticationKeysService {
	return NewOpenstackVolumeAuthenticationKeysService(op.connection, fmt.Sprintf("%s/authenticationkeys", op.path))
}

//
// A service to view certificates for this external provider.
//
func (op *OpenstackVolumeProviderService) CertificatesService() *ExternalProviderCertificatesService {
	return NewExternalProviderCertificatesService(op.connection, fmt.Sprintf("%s/certificates", op.path))
}

//
//
func (op *OpenstackVolumeProviderService) VolumeTypesService() *OpenstackVolumeTypesService {
	return NewOpenstackVolumeTypesService(op.connection, fmt.Sprintf("%s/volumetypes", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackVolumeProviderService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "authenticationkeys" {
		return op.AuthenticationKeysService(), nil
	}
	if strings.HasPrefix(path, "authenticationkeys/") {
		return op.AuthenticationKeysService().Service(path[19:])
	}
	if path == "certificates" {
		return op.CertificatesService(), nil
	}
	if strings.HasPrefix(path, "certificates/") {
		return op.CertificatesService().Service(path[13:])
	}
	if path == "volumetypes" {
		return op.VolumeTypesService(), nil
	}
	if strings.HasPrefix(path, "volumetypes/") {
		return op.VolumeTypesService().Service(path[12:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackVolumeProviderService) String() string {
	return fmt.Sprintf("OpenstackVolumeProviderService:%s", op.path)
}

//
//
type OpenstackImageProvidersService struct {
	BaseService
}

func NewOpenstackImageProvidersService(connection *Connection, path string) *OpenstackImageProvidersService {
	var result OpenstackImageProvidersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new OpenStack image provider to the system.
//
type OpenstackImageProvidersServiceAddRequest struct {
	OpenstackImageProvidersService *OpenstackImageProvidersService
	header                         map[string]string
	query                          map[string]string
	provider                       *OpenStackImageProvider
}

func (p *OpenstackImageProvidersServiceAddRequest) Header(key, value string) *OpenstackImageProvidersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProvidersServiceAddRequest) Query(key, value string) *OpenstackImageProvidersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProvidersServiceAddRequest) Provider(provider *OpenStackImageProvider) *OpenstackImageProvidersServiceAddRequest {
	p.provider = provider
	return p
}

func (p *OpenstackImageProvidersServiceAddRequest) Send() (*OpenstackImageProvidersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImageProvidersService.connection.URL(), p.OpenstackImageProvidersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackImageProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackImageProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackImageProvidersServiceAddResponse{provider: result}, nil
}

func (p *OpenstackImageProvidersServiceAddRequest) MustSend() *OpenstackImageProvidersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new OpenStack image provider to the system.
//
type OpenstackImageProvidersServiceAddResponse struct {
	provider *OpenStackImageProvider
}

func (p *OpenstackImageProvidersServiceAddResponse) Provider() (*OpenStackImageProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackImageProvidersServiceAddResponse) MustProvider() *OpenStackImageProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Add a new OpenStack image provider to the system.
//
func (p *OpenstackImageProvidersService) Add() *OpenstackImageProvidersServiceAddRequest {
	return &OpenstackImageProvidersServiceAddRequest{OpenstackImageProvidersService: p}
}

//
// Returns the list of providers.
// The order of the returned list of providers isn't guaranteed.
//
type OpenstackImageProvidersServiceListRequest struct {
	OpenstackImageProvidersService *OpenstackImageProvidersService
	header                         map[string]string
	query                          map[string]string
	follow                         *string
	max                            *int64
	search                         *string
}

func (p *OpenstackImageProvidersServiceListRequest) Header(key, value string) *OpenstackImageProvidersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProvidersServiceListRequest) Query(key, value string) *OpenstackImageProvidersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProvidersServiceListRequest) Follow(follow string) *OpenstackImageProvidersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackImageProvidersServiceListRequest) Max(max int64) *OpenstackImageProvidersServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackImageProvidersServiceListRequest) Search(search string) *OpenstackImageProvidersServiceListRequest {
	p.search = &search
	return p
}

func (p *OpenstackImageProvidersServiceListRequest) Send() (*OpenstackImageProvidersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImageProvidersService.connection.URL(), p.OpenstackImageProvidersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackImageProviderReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackImageProvidersServiceListResponse{providers: result}, nil
}

func (p *OpenstackImageProvidersServiceListRequest) MustSend() *OpenstackImageProvidersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of providers.
// The order of the returned list of providers isn't guaranteed.
//
type OpenstackImageProvidersServiceListResponse struct {
	providers *OpenStackImageProviderSlice
}

func (p *OpenstackImageProvidersServiceListResponse) Providers() (*OpenStackImageProviderSlice, bool) {
	if p.providers != nil {
		return p.providers, true
	}
	return nil, false
}

func (p *OpenstackImageProvidersServiceListResponse) MustProviders() *OpenStackImageProviderSlice {
	if p.providers == nil {
		panic("providers in response does not exist")
	}
	return p.providers
}

//
// Returns the list of providers.
// The order of the returned list of providers isn't guaranteed.
//
func (p *OpenstackImageProvidersService) List() *OpenstackImageProvidersServiceListRequest {
	return &OpenstackImageProvidersServiceListRequest{OpenstackImageProvidersService: p}
}

//
//
func (op *OpenstackImageProvidersService) ProviderService(id string) *OpenstackImageProviderService {
	return NewOpenstackImageProviderService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackImageProvidersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProviderService(path), nil
	}
	return op.ProviderService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackImageProvidersService) String() string {
	return fmt.Sprintf("OpenstackImageProvidersService:%s", op.path)
}

//
//
type OpenstackNetworkService struct {
	BaseService
}

func NewOpenstackNetworkService(connection *Connection, path string) *OpenstackNetworkService {
	var result OpenstackNetworkService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackNetworkServiceGetRequest struct {
	OpenstackNetworkService *OpenstackNetworkService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
}

func (p *OpenstackNetworkServiceGetRequest) Header(key, value string) *OpenstackNetworkServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkServiceGetRequest) Query(key, value string) *OpenstackNetworkServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkServiceGetRequest) Follow(follow string) *OpenstackNetworkServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackNetworkServiceGetRequest) Send() (*OpenstackNetworkServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackNetworkService.connection.URL(), p.OpenstackNetworkService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackNetworkReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackNetworkServiceGetResponse{network: result}, nil
}

func (p *OpenstackNetworkServiceGetRequest) MustSend() *OpenstackNetworkServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackNetworkServiceGetResponse struct {
	network *OpenStackNetwork
}

func (p *OpenstackNetworkServiceGetResponse) Network() (*OpenStackNetwork, bool) {
	if p.network != nil {
		return p.network, true
	}
	return nil, false
}

func (p *OpenstackNetworkServiceGetResponse) MustNetwork() *OpenStackNetwork {
	if p.network == nil {
		panic("network in response does not exist")
	}
	return p.network
}

//
//
func (p *OpenstackNetworkService) Get() *OpenstackNetworkServiceGetRequest {
	return &OpenstackNetworkServiceGetRequest{OpenstackNetworkService: p}
}

//
// This operation imports an external network into {product-name}.
// The network will be added to the specified data center.
//
type OpenstackNetworkServiceImportRequest struct {
	OpenstackNetworkService *OpenstackNetworkService
	header                  map[string]string
	query                   map[string]string
	async                   *bool
	dataCenter              *DataCenter
}

func (p *OpenstackNetworkServiceImportRequest) Header(key, value string) *OpenstackNetworkServiceImportRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackNetworkServiceImportRequest) Query(key, value string) *OpenstackNetworkServiceImportRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackNetworkServiceImportRequest) Async(async bool) *OpenstackNetworkServiceImportRequest {
	p.async = &async
	return p
}

func (p *OpenstackNetworkServiceImportRequest) DataCenter(dataCenter *DataCenter) *OpenstackNetworkServiceImportRequest {
	p.dataCenter = dataCenter
	return p
}

func (p *OpenstackNetworkServiceImportRequest) Send() (*OpenstackNetworkServiceImportResponse, error) {
	rawURL := fmt.Sprintf("%s%s/import", p.OpenstackNetworkService.connection.URL(), p.OpenstackNetworkService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.DataCenter(p.dataCenter)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackNetworkService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackNetworkService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackNetworkService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackNetworkService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackNetworkService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackNetworkServiceImportResponse), nil
}

func (p *OpenstackNetworkServiceImportRequest) MustSend() *OpenstackNetworkServiceImportResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation imports an external network into {product-name}.
// The network will be added to the specified data center.
//
type OpenstackNetworkServiceImportResponse struct {
}

//
// This operation imports an external network into {product-name}.
// The network will be added to the specified data center.
//
func (p *OpenstackNetworkService) Import() *OpenstackNetworkServiceImportRequest {
	return &OpenstackNetworkServiceImportRequest{OpenstackNetworkService: p}
}

//
//
func (op *OpenstackNetworkService) SubnetsService() *OpenstackSubnetsService {
	return NewOpenstackSubnetsService(op.connection, fmt.Sprintf("%s/subnets", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackNetworkService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "subnets" {
		return op.SubnetsService(), nil
	}
	if strings.HasPrefix(path, "subnets/") {
		return op.SubnetsService().Service(path[8:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackNetworkService) String() string {
	return fmt.Sprintf("OpenstackNetworkService:%s", op.path)
}

//
//
type OpenstackVolumeTypesService struct {
	BaseService
}

func NewOpenstackVolumeTypesService(connection *Connection, path string) *OpenstackVolumeTypesService {
	var result OpenstackVolumeTypesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of volume types.
// The order of the returned list of volume types isn't guaranteed.
//
type OpenstackVolumeTypesServiceListRequest struct {
	OpenstackVolumeTypesService *OpenstackVolumeTypesService
	header                      map[string]string
	query                       map[string]string
	follow                      *string
	max                         *int64
}

func (p *OpenstackVolumeTypesServiceListRequest) Header(key, value string) *OpenstackVolumeTypesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeTypesServiceListRequest) Query(key, value string) *OpenstackVolumeTypesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeTypesServiceListRequest) Follow(follow string) *OpenstackVolumeTypesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackVolumeTypesServiceListRequest) Max(max int64) *OpenstackVolumeTypesServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackVolumeTypesServiceListRequest) Send() (*OpenstackVolumeTypesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeTypesService.connection.URL(), p.OpenstackVolumeTypesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeTypesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeTypesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeTypesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeTypesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeTypesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackVolumeTypeReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeTypesServiceListResponse{types: result}, nil
}

func (p *OpenstackVolumeTypesServiceListRequest) MustSend() *OpenstackVolumeTypesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of volume types.
// The order of the returned list of volume types isn't guaranteed.
//
type OpenstackVolumeTypesServiceListResponse struct {
	types *OpenStackVolumeTypeSlice
}

func (p *OpenstackVolumeTypesServiceListResponse) Types() (*OpenStackVolumeTypeSlice, bool) {
	if p.types != nil {
		return p.types, true
	}
	return nil, false
}

func (p *OpenstackVolumeTypesServiceListResponse) MustTypes() *OpenStackVolumeTypeSlice {
	if p.types == nil {
		panic("types in response does not exist")
	}
	return p.types
}

//
// Returns the list of volume types.
// The order of the returned list of volume types isn't guaranteed.
//
func (p *OpenstackVolumeTypesService) List() *OpenstackVolumeTypesServiceListRequest {
	return &OpenstackVolumeTypesServiceListRequest{OpenstackVolumeTypesService: p}
}

//
//
func (op *OpenstackVolumeTypesService) TypeService(id string) *OpenstackVolumeTypeService {
	return NewOpenstackVolumeTypeService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackVolumeTypesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.TypeService(path), nil
	}
	return op.TypeService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackVolumeTypesService) String() string {
	return fmt.Sprintf("OpenstackVolumeTypesService:%s", op.path)
}

//
//
type OpenstackImagesService struct {
	BaseService
}

func NewOpenstackImagesService(connection *Connection, path string) *OpenstackImagesService {
	var result OpenstackImagesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Lists the images of a Glance image storage domain.
// The order of the returned list of images isn't guaranteed.
//
type OpenstackImagesServiceListRequest struct {
	OpenstackImagesService *OpenstackImagesService
	header                 map[string]string
	query                  map[string]string
	follow                 *string
	max                    *int64
}

func (p *OpenstackImagesServiceListRequest) Header(key, value string) *OpenstackImagesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImagesServiceListRequest) Query(key, value string) *OpenstackImagesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImagesServiceListRequest) Follow(follow string) *OpenstackImagesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackImagesServiceListRequest) Max(max int64) *OpenstackImagesServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackImagesServiceListRequest) Send() (*OpenstackImagesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImagesService.connection.URL(), p.OpenstackImagesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImagesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImagesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImagesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImagesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImagesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackImageReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackImagesServiceListResponse{images: result}, nil
}

func (p *OpenstackImagesServiceListRequest) MustSend() *OpenstackImagesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists the images of a Glance image storage domain.
// The order of the returned list of images isn't guaranteed.
//
type OpenstackImagesServiceListResponse struct {
	images *OpenStackImageSlice
}

func (p *OpenstackImagesServiceListResponse) Images() (*OpenStackImageSlice, bool) {
	if p.images != nil {
		return p.images, true
	}
	return nil, false
}

func (p *OpenstackImagesServiceListResponse) MustImages() *OpenStackImageSlice {
	if p.images == nil {
		panic("images in response does not exist")
	}
	return p.images
}

//
// Lists the images of a Glance image storage domain.
// The order of the returned list of images isn't guaranteed.
//
func (p *OpenstackImagesService) List() *OpenstackImagesServiceListRequest {
	return &OpenstackImagesServiceListRequest{OpenstackImagesService: p}
}

//
// Returns a reference to the service that manages a specific image.
//
func (op *OpenstackImagesService) ImageService(id string) *OpenstackImageService {
	return NewOpenstackImageService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackImagesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ImageService(path), nil
	}
	return op.ImageService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackImagesService) String() string {
	return fmt.Sprintf("OpenstackImagesService:%s", op.path)
}

//
//
type OpenstackImageProviderService struct {
	BaseService
}

func NewOpenstackImageProviderService(connection *Connection, path string) *OpenstackImageProviderService {
	var result OpenstackImageProviderService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackImageProviderServiceGetRequest struct {
	OpenstackImageProviderService *OpenstackImageProviderService
	header                        map[string]string
	query                         map[string]string
	follow                        *string
}

func (p *OpenstackImageProviderServiceGetRequest) Header(key, value string) *OpenstackImageProviderServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProviderServiceGetRequest) Query(key, value string) *OpenstackImageProviderServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProviderServiceGetRequest) Follow(follow string) *OpenstackImageProviderServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackImageProviderServiceGetRequest) Send() (*OpenstackImageProviderServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImageProviderService.connection.URL(), p.OpenstackImageProviderService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackImageProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackImageProviderServiceGetResponse{provider: result}, nil
}

func (p *OpenstackImageProviderServiceGetRequest) MustSend() *OpenstackImageProviderServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackImageProviderServiceGetResponse struct {
	provider *OpenStackImageProvider
}

func (p *OpenstackImageProviderServiceGetResponse) Provider() (*OpenStackImageProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackImageProviderServiceGetResponse) MustProvider() *OpenStackImageProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
//
func (p *OpenstackImageProviderService) Get() *OpenstackImageProviderServiceGetRequest {
	return &OpenstackImageProviderServiceGetRequest{OpenstackImageProviderService: p}
}

//
// Import the SSL certificates of the external host provider.
//
type OpenstackImageProviderServiceImportCertificatesRequest struct {
	OpenstackImageProviderService *OpenstackImageProviderService
	header                        map[string]string
	query                         map[string]string
	certificates                  *CertificateSlice
}

func (p *OpenstackImageProviderServiceImportCertificatesRequest) Header(key, value string) *OpenstackImageProviderServiceImportCertificatesRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProviderServiceImportCertificatesRequest) Query(key, value string) *OpenstackImageProviderServiceImportCertificatesRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProviderServiceImportCertificatesRequest) Certificates(certificates *CertificateSlice) *OpenstackImageProviderServiceImportCertificatesRequest {
	p.certificates = certificates
	return p
}

func (p *OpenstackImageProviderServiceImportCertificatesRequest) CertificatesOfAny(anys ...*Certificate) *OpenstackImageProviderServiceImportCertificatesRequest {
	if p.certificates == nil {
		p.certificates = new(CertificateSlice)
	}
	p.certificates.slice = append(p.certificates.slice, anys...)
	return p
}

func (p *OpenstackImageProviderServiceImportCertificatesRequest) Send() (*OpenstackImageProviderServiceImportCertificatesResponse, error) {
	rawURL := fmt.Sprintf("%s%s/importcertificates", p.OpenstackImageProviderService.connection.URL(), p.OpenstackImageProviderService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Certificates(p.certificates)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackImageProviderServiceImportCertificatesResponse), nil
}

func (p *OpenstackImageProviderServiceImportCertificatesRequest) MustSend() *OpenstackImageProviderServiceImportCertificatesResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Import the SSL certificates of the external host provider.
//
type OpenstackImageProviderServiceImportCertificatesResponse struct {
}

//
// Import the SSL certificates of the external host provider.
//
func (p *OpenstackImageProviderService) ImportCertificates() *OpenstackImageProviderServiceImportCertificatesRequest {
	return &OpenstackImageProviderServiceImportCertificatesRequest{OpenstackImageProviderService: p}
}

//
//
type OpenstackImageProviderServiceRemoveRequest struct {
	OpenstackImageProviderService *OpenstackImageProviderService
	header                        map[string]string
	query                         map[string]string
	async                         *bool
}

func (p *OpenstackImageProviderServiceRemoveRequest) Header(key, value string) *OpenstackImageProviderServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProviderServiceRemoveRequest) Query(key, value string) *OpenstackImageProviderServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProviderServiceRemoveRequest) Async(async bool) *OpenstackImageProviderServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *OpenstackImageProviderServiceRemoveRequest) Send() (*OpenstackImageProviderServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImageProviderService.connection.URL(), p.OpenstackImageProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(OpenstackImageProviderServiceRemoveResponse), nil
}

func (p *OpenstackImageProviderServiceRemoveRequest) MustSend() *OpenstackImageProviderServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackImageProviderServiceRemoveResponse struct {
}

//
//
func (p *OpenstackImageProviderService) Remove() *OpenstackImageProviderServiceRemoveRequest {
	return &OpenstackImageProviderServiceRemoveRequest{OpenstackImageProviderService: p}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type OpenstackImageProviderServiceTestConnectivityRequest struct {
	OpenstackImageProviderService *OpenstackImageProviderService
	header                        map[string]string
	query                         map[string]string
	async                         *bool
}

func (p *OpenstackImageProviderServiceTestConnectivityRequest) Header(key, value string) *OpenstackImageProviderServiceTestConnectivityRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProviderServiceTestConnectivityRequest) Query(key, value string) *OpenstackImageProviderServiceTestConnectivityRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProviderServiceTestConnectivityRequest) Async(async bool) *OpenstackImageProviderServiceTestConnectivityRequest {
	p.async = &async
	return p
}

func (p *OpenstackImageProviderServiceTestConnectivityRequest) Send() (*OpenstackImageProviderServiceTestConnectivityResponse, error) {
	rawURL := fmt.Sprintf("%s%s/testconnectivity", p.OpenstackImageProviderService.connection.URL(), p.OpenstackImageProviderService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(OpenstackImageProviderServiceTestConnectivityResponse), nil
}

func (p *OpenstackImageProviderServiceTestConnectivityRequest) MustSend() *OpenstackImageProviderServiceTestConnectivityResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type OpenstackImageProviderServiceTestConnectivityResponse struct {
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
func (p *OpenstackImageProviderService) TestConnectivity() *OpenstackImageProviderServiceTestConnectivityRequest {
	return &OpenstackImageProviderServiceTestConnectivityRequest{OpenstackImageProviderService: p}
}

//
// Update the specified OpenStack image provider in the system.
//
type OpenstackImageProviderServiceUpdateRequest struct {
	OpenstackImageProviderService *OpenstackImageProviderService
	header                        map[string]string
	query                         map[string]string
	async                         *bool
	provider                      *OpenStackImageProvider
}

func (p *OpenstackImageProviderServiceUpdateRequest) Header(key, value string) *OpenstackImageProviderServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackImageProviderServiceUpdateRequest) Query(key, value string) *OpenstackImageProviderServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackImageProviderServiceUpdateRequest) Async(async bool) *OpenstackImageProviderServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *OpenstackImageProviderServiceUpdateRequest) Provider(provider *OpenStackImageProvider) *OpenstackImageProviderServiceUpdateRequest {
	p.provider = provider
	return p
}

func (p *OpenstackImageProviderServiceUpdateRequest) Send() (*OpenstackImageProviderServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackImageProviderService.connection.URL(), p.OpenstackImageProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackImageProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackImageProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackImageProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackImageProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackImageProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackImageProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackImageProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackImageProviderServiceUpdateResponse{provider: result}, nil
}

func (p *OpenstackImageProviderServiceUpdateRequest) MustSend() *OpenstackImageProviderServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified OpenStack image provider in the system.
//
type OpenstackImageProviderServiceUpdateResponse struct {
	provider *OpenStackImageProvider
}

func (p *OpenstackImageProviderServiceUpdateResponse) Provider() (*OpenStackImageProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *OpenstackImageProviderServiceUpdateResponse) MustProvider() *OpenStackImageProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Update the specified OpenStack image provider in the system.
//
func (p *OpenstackImageProviderService) Update() *OpenstackImageProviderServiceUpdateRequest {
	return &OpenstackImageProviderServiceUpdateRequest{OpenstackImageProviderService: p}
}

//
// A service to view certificates for this external provider.
//
func (op *OpenstackImageProviderService) CertificatesService() *ExternalProviderCertificatesService {
	return NewExternalProviderCertificatesService(op.connection, fmt.Sprintf("%s/certificates", op.path))
}

//
//
func (op *OpenstackImageProviderService) ImagesService() *OpenstackImagesService {
	return NewOpenstackImagesService(op.connection, fmt.Sprintf("%s/images", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackImageProviderService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "certificates" {
		return op.CertificatesService(), nil
	}
	if strings.HasPrefix(path, "certificates/") {
		return op.CertificatesService().Service(path[13:])
	}
	if path == "images" {
		return op.ImagesService(), nil
	}
	if strings.HasPrefix(path, "images/") {
		return op.ImagesService().Service(path[7:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackImageProviderService) String() string {
	return fmt.Sprintf("OpenstackImageProviderService:%s", op.path)
}

//
//
type OpenstackSubnetsService struct {
	BaseService
}

func NewOpenstackSubnetsService(connection *Connection, path string) *OpenstackSubnetsService {
	var result OpenstackSubnetsService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackSubnetsServiceAddRequest struct {
	OpenstackSubnetsService *OpenstackSubnetsService
	header                  map[string]string
	query                   map[string]string
	subnet                  *OpenStackSubnet
}

func (p *OpenstackSubnetsServiceAddRequest) Header(key, value string) *OpenstackSubnetsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackSubnetsServiceAddRequest) Query(key, value string) *OpenstackSubnetsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackSubnetsServiceAddRequest) Subnet(subnet *OpenStackSubnet) *OpenstackSubnetsServiceAddRequest {
	p.subnet = subnet
	return p
}

func (p *OpenstackSubnetsServiceAddRequest) Send() (*OpenstackSubnetsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackSubnetsService.connection.URL(), p.OpenstackSubnetsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenStackSubnetWriteOne(writer, p.subnet, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackSubnetsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackSubnetsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackSubnetsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackSubnetsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackSubnetsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackSubnetReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackSubnetsServiceAddResponse{subnet: result}, nil
}

func (p *OpenstackSubnetsServiceAddRequest) MustSend() *OpenstackSubnetsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackSubnetsServiceAddResponse struct {
	subnet *OpenStackSubnet
}

func (p *OpenstackSubnetsServiceAddResponse) Subnet() (*OpenStackSubnet, bool) {
	if p.subnet != nil {
		return p.subnet, true
	}
	return nil, false
}

func (p *OpenstackSubnetsServiceAddResponse) MustSubnet() *OpenStackSubnet {
	if p.subnet == nil {
		panic("subnet in response does not exist")
	}
	return p.subnet
}

//
//
func (p *OpenstackSubnetsService) Add() *OpenstackSubnetsServiceAddRequest {
	return &OpenstackSubnetsServiceAddRequest{OpenstackSubnetsService: p}
}

//
// Returns the list of sub-networks.
// The order of the returned list of sub-networks isn't guaranteed.
//
type OpenstackSubnetsServiceListRequest struct {
	OpenstackSubnetsService *OpenstackSubnetsService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
	max                     *int64
}

func (p *OpenstackSubnetsServiceListRequest) Header(key, value string) *OpenstackSubnetsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackSubnetsServiceListRequest) Query(key, value string) *OpenstackSubnetsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackSubnetsServiceListRequest) Follow(follow string) *OpenstackSubnetsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackSubnetsServiceListRequest) Max(max int64) *OpenstackSubnetsServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackSubnetsServiceListRequest) Send() (*OpenstackSubnetsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackSubnetsService.connection.URL(), p.OpenstackSubnetsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackSubnetsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackSubnetsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackSubnetsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackSubnetsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackSubnetsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackSubnetReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackSubnetsServiceListResponse{subnets: result}, nil
}

func (p *OpenstackSubnetsServiceListRequest) MustSend() *OpenstackSubnetsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of sub-networks.
// The order of the returned list of sub-networks isn't guaranteed.
//
type OpenstackSubnetsServiceListResponse struct {
	subnets *OpenStackSubnetSlice
}

func (p *OpenstackSubnetsServiceListResponse) Subnets() (*OpenStackSubnetSlice, bool) {
	if p.subnets != nil {
		return p.subnets, true
	}
	return nil, false
}

func (p *OpenstackSubnetsServiceListResponse) MustSubnets() *OpenStackSubnetSlice {
	if p.subnets == nil {
		panic("subnets in response does not exist")
	}
	return p.subnets
}

//
// Returns the list of sub-networks.
// The order of the returned list of sub-networks isn't guaranteed.
//
func (p *OpenstackSubnetsService) List() *OpenstackSubnetsServiceListRequest {
	return &OpenstackSubnetsServiceListRequest{OpenstackSubnetsService: p}
}

//
//
func (op *OpenstackSubnetsService) SubnetService(id string) *OpenstackSubnetService {
	return NewOpenstackSubnetService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackSubnetsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.SubnetService(path), nil
	}
	return op.SubnetService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackSubnetsService) String() string {
	return fmt.Sprintf("OpenstackSubnetsService:%s", op.path)
}

//
//
type OpenstackVolumeAuthenticationKeyService struct {
	BaseService
}

func NewOpenstackVolumeAuthenticationKeyService(connection *Connection, path string) *OpenstackVolumeAuthenticationKeyService {
	var result OpenstackVolumeAuthenticationKeyService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackVolumeAuthenticationKeyServiceGetRequest struct {
	OpenstackVolumeAuthenticationKeyService *OpenstackVolumeAuthenticationKeyService
	header                                  map[string]string
	query                                   map[string]string
	follow                                  *string
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetRequest) Header(key, value string) *OpenstackVolumeAuthenticationKeyServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetRequest) Query(key, value string) *OpenstackVolumeAuthenticationKeyServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetRequest) Follow(follow string) *OpenstackVolumeAuthenticationKeyServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetRequest) Send() (*OpenstackVolumeAuthenticationKeyServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeAuthenticationKeyService.connection.URL(), p.OpenstackVolumeAuthenticationKeyService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeAuthenticationKeyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeAuthenticationKeyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeAuthenticationKeyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeAuthenticationKeyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeAuthenticationKeyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenstackVolumeAuthenticationKeyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeAuthenticationKeyServiceGetResponse{key: result}, nil
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetRequest) MustSend() *OpenstackVolumeAuthenticationKeyServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackVolumeAuthenticationKeyServiceGetResponse struct {
	key *OpenstackVolumeAuthenticationKey
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetResponse) Key() (*OpenstackVolumeAuthenticationKey, bool) {
	if p.key != nil {
		return p.key, true
	}
	return nil, false
}

func (p *OpenstackVolumeAuthenticationKeyServiceGetResponse) MustKey() *OpenstackVolumeAuthenticationKey {
	if p.key == nil {
		panic("key in response does not exist")
	}
	return p.key
}

//
//
func (p *OpenstackVolumeAuthenticationKeyService) Get() *OpenstackVolumeAuthenticationKeyServiceGetRequest {
	return &OpenstackVolumeAuthenticationKeyServiceGetRequest{OpenstackVolumeAuthenticationKeyService: p}
}

//
//
type OpenstackVolumeAuthenticationKeyServiceRemoveRequest struct {
	OpenstackVolumeAuthenticationKeyService *OpenstackVolumeAuthenticationKeyService
	header                                  map[string]string
	query                                   map[string]string
	async                                   *bool
}

func (p *OpenstackVolumeAuthenticationKeyServiceRemoveRequest) Header(key, value string) *OpenstackVolumeAuthenticationKeyServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceRemoveRequest) Query(key, value string) *OpenstackVolumeAuthenticationKeyServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceRemoveRequest) Async(async bool) *OpenstackVolumeAuthenticationKeyServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceRemoveRequest) Send() (*OpenstackVolumeAuthenticationKeyServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeAuthenticationKeyService.connection.URL(), p.OpenstackVolumeAuthenticationKeyService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeAuthenticationKeyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeAuthenticationKeyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeAuthenticationKeyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeAuthenticationKeyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeAuthenticationKeyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(OpenstackVolumeAuthenticationKeyServiceRemoveResponse), nil
}

func (p *OpenstackVolumeAuthenticationKeyServiceRemoveRequest) MustSend() *OpenstackVolumeAuthenticationKeyServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackVolumeAuthenticationKeyServiceRemoveResponse struct {
}

//
//
func (p *OpenstackVolumeAuthenticationKeyService) Remove() *OpenstackVolumeAuthenticationKeyServiceRemoveRequest {
	return &OpenstackVolumeAuthenticationKeyServiceRemoveRequest{OpenstackVolumeAuthenticationKeyService: p}
}

//
// Update the specified authentication key.
//
type OpenstackVolumeAuthenticationKeyServiceUpdateRequest struct {
	OpenstackVolumeAuthenticationKeyService *OpenstackVolumeAuthenticationKeyService
	header                                  map[string]string
	query                                   map[string]string
	key                                     *OpenstackVolumeAuthenticationKey
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateRequest) Header(key, value string) *OpenstackVolumeAuthenticationKeyServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateRequest) Query(key, value string) *OpenstackVolumeAuthenticationKeyServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateRequest) Key(key *OpenstackVolumeAuthenticationKey) *OpenstackVolumeAuthenticationKeyServiceUpdateRequest {
	p.key = key
	return p
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateRequest) Send() (*OpenstackVolumeAuthenticationKeyServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeAuthenticationKeyService.connection.URL(), p.OpenstackVolumeAuthenticationKeyService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenstackVolumeAuthenticationKeyWriteOne(writer, p.key, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeAuthenticationKeyService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeAuthenticationKeyService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeAuthenticationKeyService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeAuthenticationKeyService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeAuthenticationKeyService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenstackVolumeAuthenticationKeyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeAuthenticationKeyServiceUpdateResponse{key: result}, nil
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateRequest) MustSend() *OpenstackVolumeAuthenticationKeyServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified authentication key.
//
type OpenstackVolumeAuthenticationKeyServiceUpdateResponse struct {
	key *OpenstackVolumeAuthenticationKey
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateResponse) Key() (*OpenstackVolumeAuthenticationKey, bool) {
	if p.key != nil {
		return p.key, true
	}
	return nil, false
}

func (p *OpenstackVolumeAuthenticationKeyServiceUpdateResponse) MustKey() *OpenstackVolumeAuthenticationKey {
	if p.key == nil {
		panic("key in response does not exist")
	}
	return p.key
}

//
// Update the specified authentication key.
//
func (p *OpenstackVolumeAuthenticationKeyService) Update() *OpenstackVolumeAuthenticationKeyServiceUpdateRequest {
	return &OpenstackVolumeAuthenticationKeyServiceUpdateRequest{OpenstackVolumeAuthenticationKeyService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackVolumeAuthenticationKeyService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackVolumeAuthenticationKeyService) String() string {
	return fmt.Sprintf("OpenstackVolumeAuthenticationKeyService:%s", op.path)
}

//
//
type OpenstackSubnetService struct {
	BaseService
}

func NewOpenstackSubnetService(connection *Connection, path string) *OpenstackSubnetService {
	var result OpenstackSubnetService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackSubnetServiceGetRequest struct {
	OpenstackSubnetService *OpenstackSubnetService
	header                 map[string]string
	query                  map[string]string
	follow                 *string
}

func (p *OpenstackSubnetServiceGetRequest) Header(key, value string) *OpenstackSubnetServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackSubnetServiceGetRequest) Query(key, value string) *OpenstackSubnetServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackSubnetServiceGetRequest) Follow(follow string) *OpenstackSubnetServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackSubnetServiceGetRequest) Send() (*OpenstackSubnetServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackSubnetService.connection.URL(), p.OpenstackSubnetService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackSubnetService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackSubnetService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackSubnetService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackSubnetService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackSubnetService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackSubnetReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackSubnetServiceGetResponse{subnet: result}, nil
}

func (p *OpenstackSubnetServiceGetRequest) MustSend() *OpenstackSubnetServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackSubnetServiceGetResponse struct {
	subnet *OpenStackSubnet
}

func (p *OpenstackSubnetServiceGetResponse) Subnet() (*OpenStackSubnet, bool) {
	if p.subnet != nil {
		return p.subnet, true
	}
	return nil, false
}

func (p *OpenstackSubnetServiceGetResponse) MustSubnet() *OpenStackSubnet {
	if p.subnet == nil {
		panic("subnet in response does not exist")
	}
	return p.subnet
}

//
//
func (p *OpenstackSubnetService) Get() *OpenstackSubnetServiceGetRequest {
	return &OpenstackSubnetServiceGetRequest{OpenstackSubnetService: p}
}

//
//
type OpenstackSubnetServiceRemoveRequest struct {
	OpenstackSubnetService *OpenstackSubnetService
	header                 map[string]string
	query                  map[string]string
	async                  *bool
}

func (p *OpenstackSubnetServiceRemoveRequest) Header(key, value string) *OpenstackSubnetServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackSubnetServiceRemoveRequest) Query(key, value string) *OpenstackSubnetServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackSubnetServiceRemoveRequest) Async(async bool) *OpenstackSubnetServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *OpenstackSubnetServiceRemoveRequest) Send() (*OpenstackSubnetServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackSubnetService.connection.URL(), p.OpenstackSubnetService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackSubnetService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackSubnetService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackSubnetService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackSubnetService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackSubnetService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(OpenstackSubnetServiceRemoveResponse), nil
}

func (p *OpenstackSubnetServiceRemoveRequest) MustSend() *OpenstackSubnetServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackSubnetServiceRemoveResponse struct {
}

//
//
func (p *OpenstackSubnetService) Remove() *OpenstackSubnetServiceRemoveRequest {
	return &OpenstackSubnetServiceRemoveRequest{OpenstackSubnetService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackSubnetService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackSubnetService) String() string {
	return fmt.Sprintf("OpenstackSubnetService:%s", op.path)
}

//
//
type OpenstackVolumeTypeService struct {
	BaseService
}

func NewOpenstackVolumeTypeService(connection *Connection, path string) *OpenstackVolumeTypeService {
	var result OpenstackVolumeTypeService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type OpenstackVolumeTypeServiceGetRequest struct {
	OpenstackVolumeTypeService *OpenstackVolumeTypeService
	header                     map[string]string
	query                      map[string]string
	follow                     *string
}

func (p *OpenstackVolumeTypeServiceGetRequest) Header(key, value string) *OpenstackVolumeTypeServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeTypeServiceGetRequest) Query(key, value string) *OpenstackVolumeTypeServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeTypeServiceGetRequest) Follow(follow string) *OpenstackVolumeTypeServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackVolumeTypeServiceGetRequest) Send() (*OpenstackVolumeTypeServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeTypeService.connection.URL(), p.OpenstackVolumeTypeService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeTypeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeTypeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeTypeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeTypeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeTypeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenStackVolumeTypeReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeTypeServiceGetResponse{type_: result}, nil
}

func (p *OpenstackVolumeTypeServiceGetRequest) MustSend() *OpenstackVolumeTypeServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type OpenstackVolumeTypeServiceGetResponse struct {
	type_ *OpenStackVolumeType
}

func (p *OpenstackVolumeTypeServiceGetResponse) Type() (*OpenStackVolumeType, bool) {
	if p.type_ != nil {
		return p.type_, true
	}
	return nil, false
}

func (p *OpenstackVolumeTypeServiceGetResponse) MustType() *OpenStackVolumeType {
	if p.type_ == nil {
		panic("type_ in response does not exist")
	}
	return p.type_
}

//
//
func (p *OpenstackVolumeTypeService) Get() *OpenstackVolumeTypeServiceGetRequest {
	return &OpenstackVolumeTypeServiceGetRequest{OpenstackVolumeTypeService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackVolumeTypeService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *OpenstackVolumeTypeService) String() string {
	return fmt.Sprintf("OpenstackVolumeTypeService:%s", op.path)
}

//
//
type OpenstackVolumeAuthenticationKeysService struct {
	BaseService
}

func NewOpenstackVolumeAuthenticationKeysService(connection *Connection, path string) *OpenstackVolumeAuthenticationKeysService {
	var result OpenstackVolumeAuthenticationKeysService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new authentication key to the OpenStack volume provider.
//
type OpenstackVolumeAuthenticationKeysServiceAddRequest struct {
	OpenstackVolumeAuthenticationKeysService *OpenstackVolumeAuthenticationKeysService
	header                                   map[string]string
	query                                    map[string]string
	key                                      *OpenstackVolumeAuthenticationKey
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddRequest) Header(key, value string) *OpenstackVolumeAuthenticationKeysServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddRequest) Query(key, value string) *OpenstackVolumeAuthenticationKeysServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddRequest) Key(key *OpenstackVolumeAuthenticationKey) *OpenstackVolumeAuthenticationKeysServiceAddRequest {
	p.key = key
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddRequest) Send() (*OpenstackVolumeAuthenticationKeysServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeAuthenticationKeysService.connection.URL(), p.OpenstackVolumeAuthenticationKeysService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLOpenstackVolumeAuthenticationKeyWriteOne(writer, p.key, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeAuthenticationKeysService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeAuthenticationKeysService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeAuthenticationKeysService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeAuthenticationKeysService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeAuthenticationKeysService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenstackVolumeAuthenticationKeyReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeAuthenticationKeysServiceAddResponse{key: result}, nil
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddRequest) MustSend() *OpenstackVolumeAuthenticationKeysServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new authentication key to the OpenStack volume provider.
//
type OpenstackVolumeAuthenticationKeysServiceAddResponse struct {
	key *OpenstackVolumeAuthenticationKey
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddResponse) Key() (*OpenstackVolumeAuthenticationKey, bool) {
	if p.key != nil {
		return p.key, true
	}
	return nil, false
}

func (p *OpenstackVolumeAuthenticationKeysServiceAddResponse) MustKey() *OpenstackVolumeAuthenticationKey {
	if p.key == nil {
		panic("key in response does not exist")
	}
	return p.key
}

//
// Add a new authentication key to the OpenStack volume provider.
//
func (p *OpenstackVolumeAuthenticationKeysService) Add() *OpenstackVolumeAuthenticationKeysServiceAddRequest {
	return &OpenstackVolumeAuthenticationKeysServiceAddRequest{OpenstackVolumeAuthenticationKeysService: p}
}

//
// Returns the list of authentication keys.
// The order of the returned list of authentication keys isn't guaranteed.
//
type OpenstackVolumeAuthenticationKeysServiceListRequest struct {
	OpenstackVolumeAuthenticationKeysService *OpenstackVolumeAuthenticationKeysService
	header                                   map[string]string
	query                                    map[string]string
	follow                                   *string
	max                                      *int64
}

func (p *OpenstackVolumeAuthenticationKeysServiceListRequest) Header(key, value string) *OpenstackVolumeAuthenticationKeysServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceListRequest) Query(key, value string) *OpenstackVolumeAuthenticationKeysServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceListRequest) Follow(follow string) *OpenstackVolumeAuthenticationKeysServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceListRequest) Max(max int64) *OpenstackVolumeAuthenticationKeysServiceListRequest {
	p.max = &max
	return p
}

func (p *OpenstackVolumeAuthenticationKeysServiceListRequest) Send() (*OpenstackVolumeAuthenticationKeysServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OpenstackVolumeAuthenticationKeysService.connection.URL(), p.OpenstackVolumeAuthenticationKeysService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OpenstackVolumeAuthenticationKeysService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OpenstackVolumeAuthenticationKeysService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OpenstackVolumeAuthenticationKeysService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OpenstackVolumeAuthenticationKeysService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OpenstackVolumeAuthenticationKeysService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOpenstackVolumeAuthenticationKeyReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OpenstackVolumeAuthenticationKeysServiceListResponse{keys: result}, nil
}

func (p *OpenstackVolumeAuthenticationKeysServiceListRequest) MustSend() *OpenstackVolumeAuthenticationKeysServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of authentication keys.
// The order of the returned list of authentication keys isn't guaranteed.
//
type OpenstackVolumeAuthenticationKeysServiceListResponse struct {
	keys *OpenstackVolumeAuthenticationKeySlice
}

func (p *OpenstackVolumeAuthenticationKeysServiceListResponse) Keys() (*OpenstackVolumeAuthenticationKeySlice, bool) {
	if p.keys != nil {
		return p.keys, true
	}
	return nil, false
}

func (p *OpenstackVolumeAuthenticationKeysServiceListResponse) MustKeys() *OpenstackVolumeAuthenticationKeySlice {
	if p.keys == nil {
		panic("keys in response does not exist")
	}
	return p.keys
}

//
// Returns the list of authentication keys.
// The order of the returned list of authentication keys isn't guaranteed.
//
func (p *OpenstackVolumeAuthenticationKeysService) List() *OpenstackVolumeAuthenticationKeysServiceListRequest {
	return &OpenstackVolumeAuthenticationKeysServiceListRequest{OpenstackVolumeAuthenticationKeysService: p}
}

//
//
func (op *OpenstackVolumeAuthenticationKeysService) KeyService(id string) *OpenstackVolumeAuthenticationKeyService {
	return NewOpenstackVolumeAuthenticationKeyService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OpenstackVolumeAuthenticationKeysService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.KeyService(path), nil
	}
	return op.KeyService(path[:index]).Service(path[index+1:])
}

func (op *OpenstackVolumeAuthenticationKeysService) String() string {
	return fmt.Sprintf("OpenstackVolumeAuthenticationKeysService:%s", op.path)
}

//
// Returns the details of the disks attached to a virtual machine in the export domain.
//
type StorageDomainVmDiskAttachmentService struct {
	BaseService
}

func NewStorageDomainVmDiskAttachmentService(connection *Connection, path string) *StorageDomainVmDiskAttachmentService {
	var result StorageDomainVmDiskAttachmentService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the details of the attachment with all its properties and a link to the disk.
//
type StorageDomainVmDiskAttachmentServiceGetRequest struct {
	StorageDomainVmDiskAttachmentService *StorageDomainVmDiskAttachmentService
	header                               map[string]string
	query                                map[string]string
	follow                               *string
}

func (p *StorageDomainVmDiskAttachmentServiceGetRequest) Header(key, value string) *StorageDomainVmDiskAttachmentServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainVmDiskAttachmentServiceGetRequest) Query(key, value string) *StorageDomainVmDiskAttachmentServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainVmDiskAttachmentServiceGetRequest) Follow(follow string) *StorageDomainVmDiskAttachmentServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *StorageDomainVmDiskAttachmentServiceGetRequest) Send() (*StorageDomainVmDiskAttachmentServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainVmDiskAttachmentService.connection.URL(), p.StorageDomainVmDiskAttachmentService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainVmDiskAttachmentService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainVmDiskAttachmentService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainVmDiskAttachmentService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainVmDiskAttachmentService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainVmDiskAttachmentService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskAttachmentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageDomainVmDiskAttachmentServiceGetResponse{attachment: result}, nil
}

func (p *StorageDomainVmDiskAttachmentServiceGetRequest) MustSend() *StorageDomainVmDiskAttachmentServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the details of the attachment with all its properties and a link to the disk.
//
type StorageDomainVmDiskAttachmentServiceGetResponse struct {
	attachment *DiskAttachment
}

func (p *StorageDomainVmDiskAttachmentServiceGetResponse) Attachment() (*DiskAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *StorageDomainVmDiskAttachmentServiceGetResponse) MustAttachment() *DiskAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
// Returns the details of the attachment with all its properties and a link to the disk.
//
func (p *StorageDomainVmDiskAttachmentService) Get() *StorageDomainVmDiskAttachmentServiceGetRequest {
	return &StorageDomainVmDiskAttachmentServiceGetRequest{StorageDomainVmDiskAttachmentService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageDomainVmDiskAttachmentService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *StorageDomainVmDiskAttachmentService) String() string {
	return fmt.Sprintf("StorageDomainVmDiskAttachmentService:%s", op.path)
}

//
// This service represents a host that has a specific
// label when accessed through the affinitylabels/hosts
// subcollection.
//
type AffinityLabelHostService struct {
	BaseService
}

func NewAffinityLabelHostService(connection *Connection, path string) *AffinityLabelHostService {
	var result AffinityLabelHostService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves details about a host that has this label assigned.
//
type AffinityLabelHostServiceGetRequest struct {
	AffinityLabelHostService *AffinityLabelHostService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
}

func (p *AffinityLabelHostServiceGetRequest) Header(key, value string) *AffinityLabelHostServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelHostServiceGetRequest) Query(key, value string) *AffinityLabelHostServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelHostServiceGetRequest) Follow(follow string) *AffinityLabelHostServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AffinityLabelHostServiceGetRequest) Send() (*AffinityLabelHostServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelHostService.connection.URL(), p.AffinityLabelHostService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelHostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelHostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelHostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelHostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelHostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelHostServiceGetResponse{host: result}, nil
}

func (p *AffinityLabelHostServiceGetRequest) MustSend() *AffinityLabelHostServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves details about a host that has this label assigned.
//
type AffinityLabelHostServiceGetResponse struct {
	host *Host
}

func (p *AffinityLabelHostServiceGetResponse) Host() (*Host, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

func (p *AffinityLabelHostServiceGetResponse) MustHost() *Host {
	if p.host == nil {
		panic("host in response does not exist")
	}
	return p.host
}

//
// Retrieves details about a host that has this label assigned.
//
func (p *AffinityLabelHostService) Get() *AffinityLabelHostServiceGetRequest {
	return &AffinityLabelHostServiceGetRequest{AffinityLabelHostService: p}
}

//
// Remove a label from a host.
//
type AffinityLabelHostServiceRemoveRequest struct {
	AffinityLabelHostService *AffinityLabelHostService
	header                   map[string]string
	query                    map[string]string
}

func (p *AffinityLabelHostServiceRemoveRequest) Header(key, value string) *AffinityLabelHostServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelHostServiceRemoveRequest) Query(key, value string) *AffinityLabelHostServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelHostServiceRemoveRequest) Send() (*AffinityLabelHostServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelHostService.connection.URL(), p.AffinityLabelHostService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelHostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelHostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelHostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelHostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelHostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AffinityLabelHostServiceRemoveResponse), nil
}

func (p *AffinityLabelHostServiceRemoveRequest) MustSend() *AffinityLabelHostServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Remove a label from a host.
//
type AffinityLabelHostServiceRemoveResponse struct {
}

//
// Remove a label from a host.
//
func (p *AffinityLabelHostService) Remove() *AffinityLabelHostServiceRemoveRequest {
	return &AffinityLabelHostServiceRemoveRequest{AffinityLabelHostService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityLabelHostService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AffinityLabelHostService) String() string {
	return fmt.Sprintf("AffinityLabelHostService:%s", op.path)
}

//
// This service manages the set of disks attached to a virtual machine. Each attached disk is represented by a
// <<types/disk_attachment,DiskAttachment>>, containing the bootable flag, the disk interface and the reference to
// the disk.
//
type DiskAttachmentsService struct {
	BaseService
}

func NewDiskAttachmentsService(connection *Connection, path string) *DiskAttachmentsService {
	var result DiskAttachmentsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a new disk attachment to the virtual machine. The `attachment` parameter can contain just a reference, if
// the disk already exists:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk id="123"/>
// </disk_attachment>
// ----
// Or it can contain the complete representation of the disk, if the disk doesn't exist yet:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk>
//     <name>mydisk</name>
//     <provisioned_size>1024</provisioned_size>
//     ...
//   </disk>
// </disk_attachment>
// ----
// In this case the disk will be created and then attached to the virtual machine.
// In both cases, use the following URL for a virtual machine with an id `345`:
// [source]
// ----
// POST /ovirt-engine/api/vms/345/diskattachments
// ----
// IMPORTANT: The server accepts requests that don't contain the `active` attribute, but the effect is
// undefined. In some cases the disk will be automatically activated and in other cases it won't. To
// avoid issues it is strongly recommended to always include the `active` attribute with the desired
// value.
//
type DiskAttachmentsServiceAddRequest struct {
	DiskAttachmentsService *DiskAttachmentsService
	header                 map[string]string
	query                  map[string]string
	attachment             *DiskAttachment
}

func (p *DiskAttachmentsServiceAddRequest) Header(key, value string) *DiskAttachmentsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddRequest) Query(key, value string) *DiskAttachmentsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddRequest) Attachment(attachment *DiskAttachment) *DiskAttachmentsServiceAddRequest {
	p.attachment = attachment
	return p
}

func (p *DiskAttachmentsServiceAddRequest) Send() (*DiskAttachmentsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskAttachmentsService.connection.URL(), p.DiskAttachmentsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskAttachmentWriteOne(writer, p.attachment, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskAttachmentReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskAttachmentsServiceAddResponse{attachment: result}, nil
}

func (p *DiskAttachmentsServiceAddRequest) MustSend() *DiskAttachmentsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a new disk attachment to the virtual machine. The `attachment` parameter can contain just a reference, if
// the disk already exists:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk id="123"/>
// </disk_attachment>
// ----
// Or it can contain the complete representation of the disk, if the disk doesn't exist yet:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk>
//     <name>mydisk</name>
//     <provisioned_size>1024</provisioned_size>
//     ...
//   </disk>
// </disk_attachment>
// ----
// In this case the disk will be created and then attached to the virtual machine.
// In both cases, use the following URL for a virtual machine with an id `345`:
// [source]
// ----
// POST /ovirt-engine/api/vms/345/diskattachments
// ----
// IMPORTANT: The server accepts requests that don't contain the `active` attribute, but the effect is
// undefined. In some cases the disk will be automatically activated and in other cases it won't. To
// avoid issues it is strongly recommended to always include the `active` attribute with the desired
// value.
//
type DiskAttachmentsServiceAddResponse struct {
	attachment *DiskAttachment
}

func (p *DiskAttachmentsServiceAddResponse) Attachment() (*DiskAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *DiskAttachmentsServiceAddResponse) MustAttachment() *DiskAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
// Adds a new disk attachment to the virtual machine. The `attachment` parameter can contain just a reference, if
// the disk already exists:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk id="123"/>
// </disk_attachment>
// ----
// Or it can contain the complete representation of the disk, if the disk doesn't exist yet:
// [source,xml]
// ----
// <disk_attachment>
//   <bootable>true</bootable>
//   <pass_discard>true</pass_discard>
//   <interface>ide</interface>
//   <active>true</active>
//   <disk>
//     <name>mydisk</name>
//     <provisioned_size>1024</provisioned_size>
//     ...
//   </disk>
// </disk_attachment>
// ----
// In this case the disk will be created and then attached to the virtual machine.
// In both cases, use the following URL for a virtual machine with an id `345`:
// [source]
// ----
// POST /ovirt-engine/api/vms/345/diskattachments
// ----
// IMPORTANT: The server accepts requests that don't contain the `active` attribute, but the effect is
// undefined. In some cases the disk will be automatically activated and in other cases it won't. To
// avoid issues it is strongly recommended to always include the `active` attribute with the desired
// value.
//
func (p *DiskAttachmentsService) Add() *DiskAttachmentsServiceAddRequest {
	return &DiskAttachmentsServiceAddRequest{DiskAttachmentsService: p}
}

//
// List the disk that are attached to the virtual machine.
// The order of the returned list of disks attachments isn't guaranteed.
//
type DiskAttachmentsServiceListRequest struct {
	DiskAttachmentsService *DiskAttachmentsService
	header                 map[string]string
	query                  map[string]string
	follow                 *string
}

func (p *DiskAttachmentsServiceListRequest) Header(key, value string) *DiskAttachmentsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentsServiceListRequest) Query(key, value string) *DiskAttachmentsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentsServiceListRequest) Follow(follow string) *DiskAttachmentsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DiskAttachmentsServiceListRequest) Send() (*DiskAttachmentsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskAttachmentsService.connection.URL(), p.DiskAttachmentsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskAttachmentReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DiskAttachmentsServiceListResponse{attachments: result}, nil
}

func (p *DiskAttachmentsServiceListRequest) MustSend() *DiskAttachmentsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List the disk that are attached to the virtual machine.
// The order of the returned list of disks attachments isn't guaranteed.
//
type DiskAttachmentsServiceListResponse struct {
	attachments *DiskAttachmentSlice
}

func (p *DiskAttachmentsServiceListResponse) Attachments() (*DiskAttachmentSlice, bool) {
	if p.attachments != nil {
		return p.attachments, true
	}
	return nil, false
}

func (p *DiskAttachmentsServiceListResponse) MustAttachments() *DiskAttachmentSlice {
	if p.attachments == nil {
		panic("attachments in response does not exist")
	}
	return p.attachments
}

//
// List the disk that are attached to the virtual machine.
// The order of the returned list of disks attachments isn't guaranteed.
//
func (p *DiskAttachmentsService) List() *DiskAttachmentsServiceListRequest {
	return &DiskAttachmentsServiceListRequest{DiskAttachmentsService: p}
}

//
//
type DiskAttachmentsServiceAddProvidingDiskIdRequest struct {
	DiskAttachmentsService *DiskAttachmentsService
	header                 map[string]string
	query                  map[string]string
	attachment             *DiskAttachment
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) Header(key, value string) *DiskAttachmentsServiceAddProvidingDiskIdRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) Query(key, value string) *DiskAttachmentsServiceAddProvidingDiskIdRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) Attachment(attachment *DiskAttachment) *DiskAttachmentsServiceAddProvidingDiskIdRequest {
	p.attachment = attachment
	return p
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) Send() (*DiskAttachmentsServiceAddProvidingDiskIdResponse, error) {
	rawURL := fmt.Sprintf("%s%s/providingdiskid", p.DiskAttachmentsService.connection.URL(), p.DiskAttachmentsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Attachment(p.attachment)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustAttachment()
	return &DiskAttachmentsServiceAddProvidingDiskIdResponse{attachment: result}, nil
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdRequest) MustSend() *DiskAttachmentsServiceAddProvidingDiskIdResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskAttachmentsServiceAddProvidingDiskIdResponse struct {
	attachment *DiskAttachment
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdResponse) Attachment() (*DiskAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *DiskAttachmentsServiceAddProvidingDiskIdResponse) MustAttachment() *DiskAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
//
func (p *DiskAttachmentsService) AddProvidingDiskId() *DiskAttachmentsServiceAddProvidingDiskIdRequest {
	return &DiskAttachmentsServiceAddProvidingDiskIdRequest{DiskAttachmentsService: p}
}

//
//
type DiskAttachmentsServiceAddSignature1Request struct {
	DiskAttachmentsService *DiskAttachmentsService
	header                 map[string]string
	query                  map[string]string
	attachment             *DiskAttachment
}

func (p *DiskAttachmentsServiceAddSignature1Request) Header(key, value string) *DiskAttachmentsServiceAddSignature1Request {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddSignature1Request) Query(key, value string) *DiskAttachmentsServiceAddSignature1Request {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskAttachmentsServiceAddSignature1Request) Attachment(attachment *DiskAttachment) *DiskAttachmentsServiceAddSignature1Request {
	p.attachment = attachment
	return p
}

func (p *DiskAttachmentsServiceAddSignature1Request) Send() (*DiskAttachmentsServiceAddSignature1Response, error) {
	rawURL := fmt.Sprintf("%s%s/signature1", p.DiskAttachmentsService.connection.URL(), p.DiskAttachmentsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Attachment(p.attachment)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskAttachmentsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskAttachmentsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskAttachmentsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskAttachmentsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskAttachmentsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustAttachment()
	return &DiskAttachmentsServiceAddSignature1Response{attachment: result}, nil
}

func (p *DiskAttachmentsServiceAddSignature1Request) MustSend() *DiskAttachmentsServiceAddSignature1Response {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskAttachmentsServiceAddSignature1Response struct {
	attachment *DiskAttachment
}

func (p *DiskAttachmentsServiceAddSignature1Response) Attachment() (*DiskAttachment, bool) {
	if p.attachment != nil {
		return p.attachment, true
	}
	return nil, false
}

func (p *DiskAttachmentsServiceAddSignature1Response) MustAttachment() *DiskAttachment {
	if p.attachment == nil {
		panic("attachment in response does not exist")
	}
	return p.attachment
}

//
//
func (p *DiskAttachmentsService) AddSignature1() *DiskAttachmentsServiceAddSignature1Request {
	return &DiskAttachmentsServiceAddSignature1Request{DiskAttachmentsService: p}
}

//
// Reference to the service that manages a specific attachment.
//
func (op *DiskAttachmentsService) AttachmentService(id string) *DiskAttachmentService {
	return NewDiskAttachmentService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DiskAttachmentsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.AttachmentService(path), nil
	}
	return op.AttachmentService(path[:index]).Service(path[index+1:])
}

func (op *DiskAttachmentsService) String() string {
	return fmt.Sprintf("DiskAttachmentsService:%s", op.path)
}

//
//
type SnapshotCdromService struct {
	BaseService
}

func NewSnapshotCdromService(connection *Connection, path string) *SnapshotCdromService {
	var result SnapshotCdromService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type SnapshotCdromServiceGetRequest struct {
	SnapshotCdromService *SnapshotCdromService
	header               map[string]string
	query                map[string]string
	follow               *string
}

func (p *SnapshotCdromServiceGetRequest) Header(key, value string) *SnapshotCdromServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SnapshotCdromServiceGetRequest) Query(key, value string) *SnapshotCdromServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SnapshotCdromServiceGetRequest) Follow(follow string) *SnapshotCdromServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *SnapshotCdromServiceGetRequest) Send() (*SnapshotCdromServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SnapshotCdromService.connection.URL(), p.SnapshotCdromService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SnapshotCdromService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SnapshotCdromService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SnapshotCdromService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SnapshotCdromService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SnapshotCdromService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCdromReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SnapshotCdromServiceGetResponse{cdrom: result}, nil
}

func (p *SnapshotCdromServiceGetRequest) MustSend() *SnapshotCdromServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SnapshotCdromServiceGetResponse struct {
	cdrom *Cdrom
}

func (p *SnapshotCdromServiceGetResponse) Cdrom() (*Cdrom, bool) {
	if p.cdrom != nil {
		return p.cdrom, true
	}
	return nil, false
}

func (p *SnapshotCdromServiceGetResponse) MustCdrom() *Cdrom {
	if p.cdrom == nil {
		panic("cdrom in response does not exist")
	}
	return p.cdrom
}

//
//
func (p *SnapshotCdromService) Get() *SnapshotCdromServiceGetRequest {
	return &SnapshotCdromServiceGetRequest{SnapshotCdromService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SnapshotCdromService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *SnapshotCdromService) String() string {
	return fmt.Sprintf("SnapshotCdromService:%s", op.path)
}

//
//
type AssignedCpuProfilesService struct {
	BaseService
}

func NewAssignedCpuProfilesService(connection *Connection, path string) *AssignedCpuProfilesService {
	var result AssignedCpuProfilesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new cpu profile for the cluster.
//
type AssignedCpuProfilesServiceAddRequest struct {
	AssignedCpuProfilesService *AssignedCpuProfilesService
	header                     map[string]string
	query                      map[string]string
	profile                    *CpuProfile
}

func (p *AssignedCpuProfilesServiceAddRequest) Header(key, value string) *AssignedCpuProfilesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedCpuProfilesServiceAddRequest) Query(key, value string) *AssignedCpuProfilesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedCpuProfilesServiceAddRequest) Profile(profile *CpuProfile) *AssignedCpuProfilesServiceAddRequest {
	p.profile = profile
	return p
}

func (p *AssignedCpuProfilesServiceAddRequest) Send() (*AssignedCpuProfilesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedCpuProfilesService.connection.URL(), p.AssignedCpuProfilesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLCpuProfileWriteOne(writer, p.profile, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedCpuProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedCpuProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedCpuProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedCpuProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedCpuProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCpuProfileReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedCpuProfilesServiceAddResponse{profile: result}, nil
}

func (p *AssignedCpuProfilesServiceAddRequest) MustSend() *AssignedCpuProfilesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new cpu profile for the cluster.
//
type AssignedCpuProfilesServiceAddResponse struct {
	profile *CpuProfile
}

func (p *AssignedCpuProfilesServiceAddResponse) Profile() (*CpuProfile, bool) {
	if p.profile != nil {
		return p.profile, true
	}
	return nil, false
}

func (p *AssignedCpuProfilesServiceAddResponse) MustProfile() *CpuProfile {
	if p.profile == nil {
		panic("profile in response does not exist")
	}
	return p.profile
}

//
// Add a new cpu profile for the cluster.
//
func (p *AssignedCpuProfilesService) Add() *AssignedCpuProfilesServiceAddRequest {
	return &AssignedCpuProfilesServiceAddRequest{AssignedCpuProfilesService: p}
}

//
// List the CPU profiles assigned to the cluster.
// The order of the returned CPU profiles isn't guaranteed.
//
type AssignedCpuProfilesServiceListRequest struct {
	AssignedCpuProfilesService *AssignedCpuProfilesService
	header                     map[string]string
	query                      map[string]string
	follow                     *string
	max                        *int64
}

func (p *AssignedCpuProfilesServiceListRequest) Header(key, value string) *AssignedCpuProfilesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedCpuProfilesServiceListRequest) Query(key, value string) *AssignedCpuProfilesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedCpuProfilesServiceListRequest) Follow(follow string) *AssignedCpuProfilesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AssignedCpuProfilesServiceListRequest) Max(max int64) *AssignedCpuProfilesServiceListRequest {
	p.max = &max
	return p
}

func (p *AssignedCpuProfilesServiceListRequest) Send() (*AssignedCpuProfilesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedCpuProfilesService.connection.URL(), p.AssignedCpuProfilesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedCpuProfilesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedCpuProfilesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedCpuProfilesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedCpuProfilesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedCpuProfilesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLCpuProfileReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AssignedCpuProfilesServiceListResponse{profiles: result}, nil
}

func (p *AssignedCpuProfilesServiceListRequest) MustSend() *AssignedCpuProfilesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List the CPU profiles assigned to the cluster.
// The order of the returned CPU profiles isn't guaranteed.
//
type AssignedCpuProfilesServiceListResponse struct {
	profiles *CpuProfileSlice
}

func (p *AssignedCpuProfilesServiceListResponse) Profiles() (*CpuProfileSlice, bool) {
	if p.profiles != nil {
		return p.profiles, true
	}
	return nil, false
}

func (p *AssignedCpuProfilesServiceListResponse) MustProfiles() *CpuProfileSlice {
	if p.profiles == nil {
		panic("profiles in response does not exist")
	}
	return p.profiles
}

//
// List the CPU profiles assigned to the cluster.
// The order of the returned CPU profiles isn't guaranteed.
//
func (p *AssignedCpuProfilesService) List() *AssignedCpuProfilesServiceListRequest {
	return &AssignedCpuProfilesServiceListRequest{AssignedCpuProfilesService: p}
}

//
//
func (op *AssignedCpuProfilesService) ProfileService(id string) *AssignedCpuProfileService {
	return NewAssignedCpuProfileService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedCpuProfilesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProfileService(path), nil
	}
	return op.ProfileService(path[:index]).Service(path[index+1:])
}

func (op *AssignedCpuProfilesService) String() string {
	return fmt.Sprintf("AssignedCpuProfilesService:%s", op.path)
}

//
//
type QuotaService struct {
	BaseService
}

func NewQuotaService(connection *Connection, path string) *QuotaService {
	var result QuotaService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves a quota.
// An example of retrieving a quota:
// [source]
// ----
// GET /ovirt-engine/api/datacenters/123/quotas/456
// ----
// [source,xml]
// ----
// <quota id="456">
//   <name>myquota</name>
//   <description>My new quota for virtual machines</description>
//   <cluster_hard_limit_pct>20</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>80</cluster_soft_limit_pct>
//   <storage_hard_limit_pct>20</storage_hard_limit_pct>
//   <storage_soft_limit_pct>80</storage_soft_limit_pct>
// </quota>
// ----
//
type QuotaServiceGetRequest struct {
	QuotaService *QuotaService
	header       map[string]string
	query        map[string]string
	follow       *string
}

func (p *QuotaServiceGetRequest) Header(key, value string) *QuotaServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *QuotaServiceGetRequest) Query(key, value string) *QuotaServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *QuotaServiceGetRequest) Follow(follow string) *QuotaServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *QuotaServiceGetRequest) Send() (*QuotaServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.QuotaService.connection.URL(), p.QuotaService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.QuotaService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.QuotaService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.QuotaService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.QuotaService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.QuotaService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLQuotaReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &QuotaServiceGetResponse{quota: result}, nil
}

func (p *QuotaServiceGetRequest) MustSend() *QuotaServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves a quota.
// An example of retrieving a quota:
// [source]
// ----
// GET /ovirt-engine/api/datacenters/123/quotas/456
// ----
// [source,xml]
// ----
// <quota id="456">
//   <name>myquota</name>
//   <description>My new quota for virtual machines</description>
//   <cluster_hard_limit_pct>20</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>80</cluster_soft_limit_pct>
//   <storage_hard_limit_pct>20</storage_hard_limit_pct>
//   <storage_soft_limit_pct>80</storage_soft_limit_pct>
// </quota>
// ----
//
type QuotaServiceGetResponse struct {
	quota *Quota
}

func (p *QuotaServiceGetResponse) Quota() (*Quota, bool) {
	if p.quota != nil {
		return p.quota, true
	}
	return nil, false
}

func (p *QuotaServiceGetResponse) MustQuota() *Quota {
	if p.quota == nil {
		panic("quota in response does not exist")
	}
	return p.quota
}

//
// Retrieves a quota.
// An example of retrieving a quota:
// [source]
// ----
// GET /ovirt-engine/api/datacenters/123/quotas/456
// ----
// [source,xml]
// ----
// <quota id="456">
//   <name>myquota</name>
//   <description>My new quota for virtual machines</description>
//   <cluster_hard_limit_pct>20</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>80</cluster_soft_limit_pct>
//   <storage_hard_limit_pct>20</storage_hard_limit_pct>
//   <storage_soft_limit_pct>80</storage_soft_limit_pct>
// </quota>
// ----
//
func (p *QuotaService) Get() *QuotaServiceGetRequest {
	return &QuotaServiceGetRequest{QuotaService: p}
}

//
// Delete a quota.
// An example of deleting a quota:
// [source]
// ----
// DELETE /ovirt-engine/api/datacenters/123-456/quotas/654-321
// -0472718ab224 HTTP/1.1
// Accept: application/xml
// Content-type: application/xml
// ----
//
type QuotaServiceRemoveRequest struct {
	QuotaService *QuotaService
	header       map[string]string
	query        map[string]string
	async        *bool
}

func (p *QuotaServiceRemoveRequest) Header(key, value string) *QuotaServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *QuotaServiceRemoveRequest) Query(key, value string) *QuotaServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *QuotaServiceRemoveRequest) Async(async bool) *QuotaServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *QuotaServiceRemoveRequest) Send() (*QuotaServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.QuotaService.connection.URL(), p.QuotaService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.QuotaService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.QuotaService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.QuotaService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.QuotaService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.QuotaService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(QuotaServiceRemoveResponse), nil
}

func (p *QuotaServiceRemoveRequest) MustSend() *QuotaServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Delete a quota.
// An example of deleting a quota:
// [source]
// ----
// DELETE /ovirt-engine/api/datacenters/123-456/quotas/654-321
// -0472718ab224 HTTP/1.1
// Accept: application/xml
// Content-type: application/xml
// ----
//
type QuotaServiceRemoveResponse struct {
}

//
// Delete a quota.
// An example of deleting a quota:
// [source]
// ----
// DELETE /ovirt-engine/api/datacenters/123-456/quotas/654-321
// -0472718ab224 HTTP/1.1
// Accept: application/xml
// Content-type: application/xml
// ----
//
func (p *QuotaService) Remove() *QuotaServiceRemoveRequest {
	return &QuotaServiceRemoveRequest{QuotaService: p}
}

//
// Updates a quota.
// An example of updating a quota:
// [source]
// ----
// PUT /ovirt-engine/api/datacenters/123/quotas/456
// ----
// [source,xml]
// ----
// <quota>
//   <cluster_hard_limit_pct>30</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>70</cluster_soft_limit_pct>
//   <storage_hard_limit_pct>20</storage_hard_limit_pct>
//   <storage_soft_limit_pct>80</storage_soft_limit_pct>
// </quota>
// ----
//
type QuotaServiceUpdateRequest struct {
	QuotaService *QuotaService
	header       map[string]string
	query        map[string]string
	async        *bool
	quota        *Quota
}

func (p *QuotaServiceUpdateRequest) Header(key, value string) *QuotaServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *QuotaServiceUpdateRequest) Query(key, value string) *QuotaServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *QuotaServiceUpdateRequest) Async(async bool) *QuotaServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *QuotaServiceUpdateRequest) Quota(quota *Quota) *QuotaServiceUpdateRequest {
	p.quota = quota
	return p
}

func (p *QuotaServiceUpdateRequest) Send() (*QuotaServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.QuotaService.connection.URL(), p.QuotaService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLQuotaWriteOne(writer, p.quota, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.QuotaService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.QuotaService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.QuotaService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.QuotaService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.QuotaService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLQuotaReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &QuotaServiceUpdateResponse{quota: result}, nil
}

func (p *QuotaServiceUpdateRequest) MustSend() *QuotaServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates a quota.
// An example of updating a quota:
// [source]
// ----
// PUT /ovirt-engine/api/datacenters/123/quotas/456
// ----
// [source,xml]
// ----
// <quota>
//   <cluster_hard_limit_pct>30</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>70</cluster_soft_limit_pct>
//   <storage_hard_limit_pct>20</storage_hard_limit_pct>
//   <storage_soft_limit_pct>80</storage_soft_limit_pct>
// </quota>
// ----
//
type QuotaServiceUpdateResponse struct {
	quota *Quota
}

func (p *QuotaServiceUpdateResponse) Quota() (*Quota, bool) {
	if p.quota != nil {
		return p.quota, true
	}
	return nil, false
}

func (p *QuotaServiceUpdateResponse) MustQuota() *Quota {
	if p.quota == nil {
		panic("quota in response does not exist")
	}
	return p.quota
}

//
// Updates a quota.
// An example of updating a quota:
// [source]
// ----
// PUT /ovirt-engine/api/datacenters/123/quotas/456
// ----
// [source,xml]
// ----
// <quota>
//   <cluster_hard_limit_pct>30</cluster_hard_limit_pct>
//   <cluster_soft_limit_pct>70</cluster_soft_limit_pct>
//   <storage_hard_limit_pct>20</storage_hard_limit_pct>
//   <storage_soft_limit_pct>80</storage_soft_limit_pct>
// </quota>
// ----
//
func (p *QuotaService) Update() *QuotaServiceUpdateRequest {
	return &QuotaServiceUpdateRequest{QuotaService: p}
}

//
//
func (op *QuotaService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
//
func (op *QuotaService) QuotaClusterLimitsService() *QuotaClusterLimitsService {
	return NewQuotaClusterLimitsService(op.connection, fmt.Sprintf("%s/quotaclusterlimits", op.path))
}

//
//
func (op *QuotaService) QuotaStorageLimitsService() *QuotaStorageLimitsService {
	return NewQuotaStorageLimitsService(op.connection, fmt.Sprintf("%s/quotastoragelimits", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *QuotaService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "quotaclusterlimits" {
		return op.QuotaClusterLimitsService(), nil
	}
	if strings.HasPrefix(path, "quotaclusterlimits/") {
		return op.QuotaClusterLimitsService().Service(path[19:])
	}
	if path == "quotastoragelimits" {
		return op.QuotaStorageLimitsService(), nil
	}
	if strings.HasPrefix(path, "quotastoragelimits/") {
		return op.QuotaStorageLimitsService().Service(path[19:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *QuotaService) String() string {
	return fmt.Sprintf("QuotaService:%s", op.path)
}

//
//
type BalanceService struct {
	BaseService
}

func NewBalanceService(connection *Connection, path string) *BalanceService {
	var result BalanceService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type BalanceServiceGetRequest struct {
	BalanceService *BalanceService
	header         map[string]string
	query          map[string]string
	filter         *bool
	follow         *string
}

func (p *BalanceServiceGetRequest) Header(key, value string) *BalanceServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BalanceServiceGetRequest) Query(key, value string) *BalanceServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BalanceServiceGetRequest) Filter(filter bool) *BalanceServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *BalanceServiceGetRequest) Follow(follow string) *BalanceServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *BalanceServiceGetRequest) Send() (*BalanceServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BalanceService.connection.URL(), p.BalanceService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BalanceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BalanceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BalanceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BalanceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BalanceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBalanceReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &BalanceServiceGetResponse{balance: result}, nil
}

func (p *BalanceServiceGetRequest) MustSend() *BalanceServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type BalanceServiceGetResponse struct {
	balance *Balance
}

func (p *BalanceServiceGetResponse) Balance() (*Balance, bool) {
	if p.balance != nil {
		return p.balance, true
	}
	return nil, false
}

func (p *BalanceServiceGetResponse) MustBalance() *Balance {
	if p.balance == nil {
		panic("balance in response does not exist")
	}
	return p.balance
}

//
//
func (p *BalanceService) Get() *BalanceServiceGetRequest {
	return &BalanceServiceGetRequest{BalanceService: p}
}

//
//
type BalanceServiceRemoveRequest struct {
	BalanceService *BalanceService
	header         map[string]string
	query          map[string]string
	async          *bool
}

func (p *BalanceServiceRemoveRequest) Header(key, value string) *BalanceServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BalanceServiceRemoveRequest) Query(key, value string) *BalanceServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BalanceServiceRemoveRequest) Async(async bool) *BalanceServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *BalanceServiceRemoveRequest) Send() (*BalanceServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BalanceService.connection.URL(), p.BalanceService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BalanceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BalanceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BalanceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BalanceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BalanceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(BalanceServiceRemoveResponse), nil
}

func (p *BalanceServiceRemoveRequest) MustSend() *BalanceServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type BalanceServiceRemoveResponse struct {
}

//
//
func (p *BalanceService) Remove() *BalanceServiceRemoveRequest {
	return &BalanceServiceRemoveRequest{BalanceService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *BalanceService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *BalanceService) String() string {
	return fmt.Sprintf("BalanceService:%s", op.path)
}

//
// This service manages a collection of all the virtual machines assigned to an affinity group.
//
type AffinityGroupVmsService struct {
	BaseService
}

func NewAffinityGroupVmsService(connection *Connection, path string) *AffinityGroupVmsService {
	var result AffinityGroupVmsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a virtual machine to the affinity group.
// For example, to add the virtual machine `789` to the affinity group `456` of cluster `123`, send a request like
// this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/vms
// ....
// With the following body:
// [source,xml]
// ----
// <vm id="789"/>
// ----
//
type AffinityGroupVmsServiceAddRequest struct {
	AffinityGroupVmsService *AffinityGroupVmsService
	header                  map[string]string
	query                   map[string]string
	vm                      *Vm
}

func (p *AffinityGroupVmsServiceAddRequest) Header(key, value string) *AffinityGroupVmsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupVmsServiceAddRequest) Query(key, value string) *AffinityGroupVmsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupVmsServiceAddRequest) Vm(vm *Vm) *AffinityGroupVmsServiceAddRequest {
	p.vm = vm
	return p
}

func (p *AffinityGroupVmsServiceAddRequest) Send() (*AffinityGroupVmsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupVmsService.connection.URL(), p.AffinityGroupVmsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLVmWriteOne(writer, p.vm, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupVmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupVmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupVmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupVmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupVmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityGroupVmsServiceAddResponse{vm: result}, nil
}

func (p *AffinityGroupVmsServiceAddRequest) MustSend() *AffinityGroupVmsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a virtual machine to the affinity group.
// For example, to add the virtual machine `789` to the affinity group `456` of cluster `123`, send a request like
// this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/vms
// ....
// With the following body:
// [source,xml]
// ----
// <vm id="789"/>
// ----
//
type AffinityGroupVmsServiceAddResponse struct {
	vm *Vm
}

func (p *AffinityGroupVmsServiceAddResponse) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

func (p *AffinityGroupVmsServiceAddResponse) MustVm() *Vm {
	if p.vm == nil {
		panic("vm in response does not exist")
	}
	return p.vm
}

//
// Adds a virtual machine to the affinity group.
// For example, to add the virtual machine `789` to the affinity group `456` of cluster `123`, send a request like
// this:
// ....
// POST /ovirt-engine/api/clusters/123/affinitygroups/456/vms
// ....
// With the following body:
// [source,xml]
// ----
// <vm id="789"/>
// ----
//
func (p *AffinityGroupVmsService) Add() *AffinityGroupVmsServiceAddRequest {
	return &AffinityGroupVmsServiceAddRequest{AffinityGroupVmsService: p}
}

//
// List all virtual machines assigned to this affinity group.
// The order of the returned virtual machines isn't guaranteed.
//
type AffinityGroupVmsServiceListRequest struct {
	AffinityGroupVmsService *AffinityGroupVmsService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
	max                     *int64
}

func (p *AffinityGroupVmsServiceListRequest) Header(key, value string) *AffinityGroupVmsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityGroupVmsServiceListRequest) Query(key, value string) *AffinityGroupVmsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityGroupVmsServiceListRequest) Follow(follow string) *AffinityGroupVmsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AffinityGroupVmsServiceListRequest) Max(max int64) *AffinityGroupVmsServiceListRequest {
	p.max = &max
	return p
}

func (p *AffinityGroupVmsServiceListRequest) Send() (*AffinityGroupVmsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityGroupVmsService.connection.URL(), p.AffinityGroupVmsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityGroupVmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityGroupVmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityGroupVmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityGroupVmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityGroupVmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AffinityGroupVmsServiceListResponse{vms: result}, nil
}

func (p *AffinityGroupVmsServiceListRequest) MustSend() *AffinityGroupVmsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all virtual machines assigned to this affinity group.
// The order of the returned virtual machines isn't guaranteed.
//
type AffinityGroupVmsServiceListResponse struct {
	vms *VmSlice
}

func (p *AffinityGroupVmsServiceListResponse) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *AffinityGroupVmsServiceListResponse) MustVms() *VmSlice {
	if p.vms == nil {
		panic("vms in response does not exist")
	}
	return p.vms
}

//
// List all virtual machines assigned to this affinity group.
// The order of the returned virtual machines isn't guaranteed.
//
func (p *AffinityGroupVmsService) List() *AffinityGroupVmsServiceListRequest {
	return &AffinityGroupVmsServiceListRequest{AffinityGroupVmsService: p}
}

//
// Access the service that manages the virtual machine assignment to this affinity group.
//
func (op *AffinityGroupVmsService) VmService(id string) *AffinityGroupVmService {
	return NewAffinityGroupVmService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityGroupVmsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.VmService(path), nil
	}
	return op.VmService(path[:index]).Service(path[index+1:])
}

func (op *AffinityGroupVmsService) String() string {
	return fmt.Sprintf("AffinityGroupVmsService:%s", op.path)
}

//
// A service to manage a virtual machines pool.
//
type VmPoolService struct {
	BaseService
}

func NewVmPoolService(connection *Connection, path string) *VmPoolService {
	var result VmPoolService
	result.connection = connection
	result.path = path
	return &result
}

//
// This operation allocates a virtual machine in the virtual machine pool.
// [source]
// ----
// POST /ovirt-engine/api/vmpools/123/allocatevm
// ----
// The allocate virtual machine action does not take any action specific parameters, so the request body should
// contain an empty `action`:
// [source,xml]
// ----
// <action/>
// ----
//
type VmPoolServiceAllocateVmRequest struct {
	VmPoolService *VmPoolService
	header        map[string]string
	query         map[string]string
	async         *bool
}

func (p *VmPoolServiceAllocateVmRequest) Header(key, value string) *VmPoolServiceAllocateVmRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmPoolServiceAllocateVmRequest) Query(key, value string) *VmPoolServiceAllocateVmRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmPoolServiceAllocateVmRequest) Async(async bool) *VmPoolServiceAllocateVmRequest {
	p.async = &async
	return p
}

func (p *VmPoolServiceAllocateVmRequest) Send() (*VmPoolServiceAllocateVmResponse, error) {
	rawURL := fmt.Sprintf("%s%s/allocatevm", p.VmPoolService.connection.URL(), p.VmPoolService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmPoolService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmPoolService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmPoolService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmPoolService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmPoolService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(VmPoolServiceAllocateVmResponse), nil
}

func (p *VmPoolServiceAllocateVmRequest) MustSend() *VmPoolServiceAllocateVmResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation allocates a virtual machine in the virtual machine pool.
// [source]
// ----
// POST /ovirt-engine/api/vmpools/123/allocatevm
// ----
// The allocate virtual machine action does not take any action specific parameters, so the request body should
// contain an empty `action`:
// [source,xml]
// ----
// <action/>
// ----
//
type VmPoolServiceAllocateVmResponse struct {
}

//
// This operation allocates a virtual machine in the virtual machine pool.
// [source]
// ----
// POST /ovirt-engine/api/vmpools/123/allocatevm
// ----
// The allocate virtual machine action does not take any action specific parameters, so the request body should
// contain an empty `action`:
// [source,xml]
// ----
// <action/>
// ----
//
func (p *VmPoolService) AllocateVm() *VmPoolServiceAllocateVmRequest {
	return &VmPoolServiceAllocateVmRequest{VmPoolService: p}
}

//
// Get the virtual machine pool.
// [source]
// ----
// GET /ovirt-engine/api/vmpools/123
// ----
// You will get a XML response like that one:
// [source,xml]
// ----
// <vm_pool id="123">
//   <actions>...</actions>
//   <name>MyVmPool</name>
//   <description>MyVmPool description</description>
//   <link href="/ovirt-engine/api/vmpools/123/permissions" rel="permissions"/>
//   <max_user_vms>1</max_user_vms>
//   <prestarted_vms>0</prestarted_vms>
//   <size>100</size>
//   <stateful>false</stateful>
//   <type>automatic</type>
//   <use_latest_template_version>false</use_latest_template_version>
//   <cluster id="123"/>
//   <template id="123"/>
//   <vm id="123">...</vm>
//   ...
// </vm_pool>
// ----
//
type VmPoolServiceGetRequest struct {
	VmPoolService *VmPoolService
	header        map[string]string
	query         map[string]string
	filter        *bool
	follow        *string
}

func (p *VmPoolServiceGetRequest) Header(key, value string) *VmPoolServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmPoolServiceGetRequest) Query(key, value string) *VmPoolServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmPoolServiceGetRequest) Filter(filter bool) *VmPoolServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *VmPoolServiceGetRequest) Follow(follow string) *VmPoolServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *VmPoolServiceGetRequest) Send() (*VmPoolServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmPoolService.connection.URL(), p.VmPoolService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmPoolService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmPoolService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmPoolService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmPoolService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmPoolService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmPoolReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmPoolServiceGetResponse{pool: result}, nil
}

func (p *VmPoolServiceGetRequest) MustSend() *VmPoolServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get the virtual machine pool.
// [source]
// ----
// GET /ovirt-engine/api/vmpools/123
// ----
// You will get a XML response like that one:
// [source,xml]
// ----
// <vm_pool id="123">
//   <actions>...</actions>
//   <name>MyVmPool</name>
//   <description>MyVmPool description</description>
//   <link href="/ovirt-engine/api/vmpools/123/permissions" rel="permissions"/>
//   <max_user_vms>1</max_user_vms>
//   <prestarted_vms>0</prestarted_vms>
//   <size>100</size>
//   <stateful>false</stateful>
//   <type>automatic</type>
//   <use_latest_template_version>false</use_latest_template_version>
//   <cluster id="123"/>
//   <template id="123"/>
//   <vm id="123">...</vm>
//   ...
// </vm_pool>
// ----
//
type VmPoolServiceGetResponse struct {
	pool *VmPool
}

func (p *VmPoolServiceGetResponse) Pool() (*VmPool, bool) {
	if p.pool != nil {
		return p.pool, true
	}
	return nil, false
}

func (p *VmPoolServiceGetResponse) MustPool() *VmPool {
	if p.pool == nil {
		panic("pool in response does not exist")
	}
	return p.pool
}

//
// Get the virtual machine pool.
// [source]
// ----
// GET /ovirt-engine/api/vmpools/123
// ----
// You will get a XML response like that one:
// [source,xml]
// ----
// <vm_pool id="123">
//   <actions>...</actions>
//   <name>MyVmPool</name>
//   <description>MyVmPool description</description>
//   <link href="/ovirt-engine/api/vmpools/123/permissions" rel="permissions"/>
//   <max_user_vms>1</max_user_vms>
//   <prestarted_vms>0</prestarted_vms>
//   <size>100</size>
//   <stateful>false</stateful>
//   <type>automatic</type>
//   <use_latest_template_version>false</use_latest_template_version>
//   <cluster id="123"/>
//   <template id="123"/>
//   <vm id="123">...</vm>
//   ...
// </vm_pool>
// ----
//
func (p *VmPoolService) Get() *VmPoolServiceGetRequest {
	return &VmPoolServiceGetRequest{VmPoolService: p}
}

//
// Removes a virtual machine pool.
// [source]
// ----
// DELETE /ovirt-engine/api/vmpools/123
// ----
//
type VmPoolServiceRemoveRequest struct {
	VmPoolService *VmPoolService
	header        map[string]string
	query         map[string]string
	async         *bool
}

func (p *VmPoolServiceRemoveRequest) Header(key, value string) *VmPoolServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmPoolServiceRemoveRequest) Query(key, value string) *VmPoolServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmPoolServiceRemoveRequest) Async(async bool) *VmPoolServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *VmPoolServiceRemoveRequest) Send() (*VmPoolServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmPoolService.connection.URL(), p.VmPoolService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmPoolService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmPoolService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmPoolService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmPoolService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmPoolService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(VmPoolServiceRemoveResponse), nil
}

func (p *VmPoolServiceRemoveRequest) MustSend() *VmPoolServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a virtual machine pool.
// [source]
// ----
// DELETE /ovirt-engine/api/vmpools/123
// ----
//
type VmPoolServiceRemoveResponse struct {
}

//
// Removes a virtual machine pool.
// [source]
// ----
// DELETE /ovirt-engine/api/vmpools/123
// ----
//
func (p *VmPoolService) Remove() *VmPoolServiceRemoveRequest {
	return &VmPoolServiceRemoveRequest{VmPoolService: p}
}

//
// Update the virtual machine pool.
// [source]
// ----
// PUT /ovirt-engine/api/vmpools/123
// ----
// The `name`, `description`, `size`, `prestarted_vms` and `max_user_vms`
// attributes can be updated after the virtual machine pool has been
// created.
// [source,xml]
// ----
// <vmpool>
//   <name>VM_Pool_B</name>
//   <description>Virtual Machine Pool B</description>
//   <size>3</size>
//   <prestarted_vms>1</size>
//   <max_user_vms>2</size>
// </vmpool>
// ----
//
type VmPoolServiceUpdateRequest struct {
	VmPoolService *VmPoolService
	header        map[string]string
	query         map[string]string
	async         *bool
	pool          *VmPool
}

func (p *VmPoolServiceUpdateRequest) Header(key, value string) *VmPoolServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmPoolServiceUpdateRequest) Query(key, value string) *VmPoolServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmPoolServiceUpdateRequest) Async(async bool) *VmPoolServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *VmPoolServiceUpdateRequest) Pool(pool *VmPool) *VmPoolServiceUpdateRequest {
	p.pool = pool
	return p
}

func (p *VmPoolServiceUpdateRequest) Send() (*VmPoolServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmPoolService.connection.URL(), p.VmPoolService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLVmPoolWriteOne(writer, p.pool, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmPoolService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmPoolService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmPoolService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmPoolService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmPoolService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmPoolReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmPoolServiceUpdateResponse{pool: result}, nil
}

func (p *VmPoolServiceUpdateRequest) MustSend() *VmPoolServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the virtual machine pool.
// [source]
// ----
// PUT /ovirt-engine/api/vmpools/123
// ----
// The `name`, `description`, `size`, `prestarted_vms` and `max_user_vms`
// attributes can be updated after the virtual machine pool has been
// created.
// [source,xml]
// ----
// <vmpool>
//   <name>VM_Pool_B</name>
//   <description>Virtual Machine Pool B</description>
//   <size>3</size>
//   <prestarted_vms>1</size>
//   <max_user_vms>2</size>
// </vmpool>
// ----
//
type VmPoolServiceUpdateResponse struct {
	pool *VmPool
}

func (p *VmPoolServiceUpdateResponse) Pool() (*VmPool, bool) {
	if p.pool != nil {
		return p.pool, true
	}
	return nil, false
}

func (p *VmPoolServiceUpdateResponse) MustPool() *VmPool {
	if p.pool == nil {
		panic("pool in response does not exist")
	}
	return p.pool
}

//
// Update the virtual machine pool.
// [source]
// ----
// PUT /ovirt-engine/api/vmpools/123
// ----
// The `name`, `description`, `size`, `prestarted_vms` and `max_user_vms`
// attributes can be updated after the virtual machine pool has been
// created.
// [source,xml]
// ----
// <vmpool>
//   <name>VM_Pool_B</name>
//   <description>Virtual Machine Pool B</description>
//   <size>3</size>
//   <prestarted_vms>1</size>
//   <max_user_vms>2</size>
// </vmpool>
// ----
//
func (p *VmPoolService) Update() *VmPoolServiceUpdateRequest {
	return &VmPoolServiceUpdateRequest{VmPoolService: p}
}

//
// Reference to a service managing the virtual machine pool assigned permissions.
//
func (op *VmPoolService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmPoolService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *VmPoolService) String() string {
	return fmt.Sprintf("VmPoolService:%s", op.path)
}

//
// Represents a readonly network filters sub-collection.
// The network filter enables to filter packets send to/from the VM's nic according to defined rules.
// For more information please refer to <<services/network_filter,NetworkFilter>> service documentation
// Network filters are supported in different versions, starting from version 3.0.
// A network filter is defined for each vnic profile.
// A vnic profile is defined for a specific network.
// A network can be assigned to several different clusters. In the future, each network will be defined in
// cluster level.
// Currently, each network is being defined at data center level. Potential network filters for each network
// are determined by the network's data center compatibility version V.
// V must be >= the network filter version in order to configure this network filter for a specific network.
// Please note, that if a network is assigned to cluster with a version supporting a network filter, the filter
// may not be available due to the data center version being smaller then the network filter's version.
// Example of listing all of the supported network filters for a specific cluster:
// [source]
// ----
// GET http://localhost:8080/ovirt-engine/api/clusters/{cluster:id}/networkfilters
// ----
// Output:
// [source,xml]
// ----
// <network_filters>
//   <network_filter id="00000019-0019-0019-0019-00000000026c">
//     <name>example-network-filter-a</name>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//       <build>-1</build>
//       <revision>-1</revision>
//     </version>
//   </network_filter>
//   <network_filter id="00000019-0019-0019-0019-00000000026b">
//     <name>example-network-filter-b</name>
//     <version>
//       <major>4</major>
//       <minor>0</minor>
//       <build>-1</build>
//       <revision>-1</revision>
//     </version>
//   </network_filter>
//   <network_filter id="00000019-0019-0019-0019-00000000026a">
//     <name>example-network-filter-a</name>
//     <version>
//       <major>3</major>
//       <minor>0</minor>
//       <build>-1</build>
//       <revision>-1</revision>
//     </version>
//   </network_filter>
// </network_filters>
// ----
//
type NetworkFiltersService struct {
	BaseService
}

func NewNetworkFiltersService(connection *Connection, path string) *NetworkFiltersService {
	var result NetworkFiltersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the representations of the network filters.
// The order of the returned list of network filters isn't guaranteed.
//
type NetworkFiltersServiceListRequest struct {
	NetworkFiltersService *NetworkFiltersService
	header                map[string]string
	query                 map[string]string
	follow                *string
}

func (p *NetworkFiltersServiceListRequest) Header(key, value string) *NetworkFiltersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *NetworkFiltersServiceListRequest) Query(key, value string) *NetworkFiltersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *NetworkFiltersServiceListRequest) Follow(follow string) *NetworkFiltersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *NetworkFiltersServiceListRequest) Send() (*NetworkFiltersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.NetworkFiltersService.connection.URL(), p.NetworkFiltersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.NetworkFiltersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.NetworkFiltersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.NetworkFiltersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.NetworkFiltersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.NetworkFiltersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNetworkFilterReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &NetworkFiltersServiceListResponse{filters: result}, nil
}

func (p *NetworkFiltersServiceListRequest) MustSend() *NetworkFiltersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the representations of the network filters.
// The order of the returned list of network filters isn't guaranteed.
//
type NetworkFiltersServiceListResponse struct {
	filters *NetworkFilterSlice
}

func (p *NetworkFiltersServiceListResponse) Filters() (*NetworkFilterSlice, bool) {
	if p.filters != nil {
		return p.filters, true
	}
	return nil, false
}

func (p *NetworkFiltersServiceListResponse) MustFilters() *NetworkFilterSlice {
	if p.filters == nil {
		panic("filters in response does not exist")
	}
	return p.filters
}

//
// Retrieves the representations of the network filters.
// The order of the returned list of network filters isn't guaranteed.
//
func (p *NetworkFiltersService) List() *NetworkFiltersServiceListRequest {
	return &NetworkFiltersServiceListRequest{NetworkFiltersService: p}
}

//
//
func (op *NetworkFiltersService) NetworkFilterService(id string) *NetworkFilterService {
	return NewNetworkFilterService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *NetworkFiltersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NetworkFilterService(path), nil
	}
	return op.NetworkFilterService(path[:index]).Service(path[index+1:])
}

func (op *NetworkFiltersService) String() string {
	return fmt.Sprintf("NetworkFiltersService:%s", op.path)
}

//
//
type InstanceTypeNicsService struct {
	BaseService
}

func NewInstanceTypeNicsService(connection *Connection, path string) *InstanceTypeNicsService {
	var result InstanceTypeNicsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add new network interface to the instance type.
//
type InstanceTypeNicsServiceAddRequest struct {
	InstanceTypeNicsService *InstanceTypeNicsService
	header                  map[string]string
	query                   map[string]string
	nic                     *Nic
}

func (p *InstanceTypeNicsServiceAddRequest) Header(key, value string) *InstanceTypeNicsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *InstanceTypeNicsServiceAddRequest) Query(key, value string) *InstanceTypeNicsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *InstanceTypeNicsServiceAddRequest) Nic(nic *Nic) *InstanceTypeNicsServiceAddRequest {
	p.nic = nic
	return p
}

func (p *InstanceTypeNicsServiceAddRequest) Send() (*InstanceTypeNicsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.InstanceTypeNicsService.connection.URL(), p.InstanceTypeNicsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNicWriteOne(writer, p.nic, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.InstanceTypeNicsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.InstanceTypeNicsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.InstanceTypeNicsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.InstanceTypeNicsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.InstanceTypeNicsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNicReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &InstanceTypeNicsServiceAddResponse{nic: result}, nil
}

func (p *InstanceTypeNicsServiceAddRequest) MustSend() *InstanceTypeNicsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add new network interface to the instance type.
//
type InstanceTypeNicsServiceAddResponse struct {
	nic *Nic
}

func (p *InstanceTypeNicsServiceAddResponse) Nic() (*Nic, bool) {
	if p.nic != nil {
		return p.nic, true
	}
	return nil, false
}

func (p *InstanceTypeNicsServiceAddResponse) MustNic() *Nic {
	if p.nic == nil {
		panic("nic in response does not exist")
	}
	return p.nic
}

//
// Add new network interface to the instance type.
//
func (p *InstanceTypeNicsService) Add() *InstanceTypeNicsServiceAddRequest {
	return &InstanceTypeNicsServiceAddRequest{InstanceTypeNicsService: p}
}

//
// Lists all the configured network interface of the instance type.
// The order of the returned list of network interfaces isn't guaranteed.
//
type InstanceTypeNicsServiceListRequest struct {
	InstanceTypeNicsService *InstanceTypeNicsService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
	max                     *int64
	search                  *string
}

func (p *InstanceTypeNicsServiceListRequest) Header(key, value string) *InstanceTypeNicsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *InstanceTypeNicsServiceListRequest) Query(key, value string) *InstanceTypeNicsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *InstanceTypeNicsServiceListRequest) Follow(follow string) *InstanceTypeNicsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *InstanceTypeNicsServiceListRequest) Max(max int64) *InstanceTypeNicsServiceListRequest {
	p.max = &max
	return p
}

func (p *InstanceTypeNicsServiceListRequest) Search(search string) *InstanceTypeNicsServiceListRequest {
	p.search = &search
	return p
}

func (p *InstanceTypeNicsServiceListRequest) Send() (*InstanceTypeNicsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.InstanceTypeNicsService.connection.URL(), p.InstanceTypeNicsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.InstanceTypeNicsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.InstanceTypeNicsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.InstanceTypeNicsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.InstanceTypeNicsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.InstanceTypeNicsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNicReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &InstanceTypeNicsServiceListResponse{nics: result}, nil
}

func (p *InstanceTypeNicsServiceListRequest) MustSend() *InstanceTypeNicsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists all the configured network interface of the instance type.
// The order of the returned list of network interfaces isn't guaranteed.
//
type InstanceTypeNicsServiceListResponse struct {
	nics *NicSlice
}

func (p *InstanceTypeNicsServiceListResponse) Nics() (*NicSlice, bool) {
	if p.nics != nil {
		return p.nics, true
	}
	return nil, false
}

func (p *InstanceTypeNicsServiceListResponse) MustNics() *NicSlice {
	if p.nics == nil {
		panic("nics in response does not exist")
	}
	return p.nics
}

//
// Lists all the configured network interface of the instance type.
// The order of the returned list of network interfaces isn't guaranteed.
//
func (p *InstanceTypeNicsService) List() *InstanceTypeNicsServiceListRequest {
	return &InstanceTypeNicsServiceListRequest{InstanceTypeNicsService: p}
}

//
//
func (op *InstanceTypeNicsService) NicService(id string) *InstanceTypeNicService {
	return NewInstanceTypeNicService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *InstanceTypeNicsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NicService(path), nil
	}
	return op.NicService(path[:index]).Service(path[index+1:])
}

func (op *InstanceTypeNicsService) String() string {
	return fmt.Sprintf("InstanceTypeNicsService:%s", op.path)
}

//
// Manages the set of types of operating systems available in the system.
//
type OperatingSystemsService struct {
	BaseService
}

func NewOperatingSystemsService(connection *Connection, path string) *OperatingSystemsService {
	var result OperatingSystemsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of types of operating system available in the system.
// The order of the returned list of operating systems isn't guaranteed.
//
type OperatingSystemsServiceListRequest struct {
	OperatingSystemsService *OperatingSystemsService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
	max                     *int64
}

func (p *OperatingSystemsServiceListRequest) Header(key, value string) *OperatingSystemsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *OperatingSystemsServiceListRequest) Query(key, value string) *OperatingSystemsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *OperatingSystemsServiceListRequest) Follow(follow string) *OperatingSystemsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *OperatingSystemsServiceListRequest) Max(max int64) *OperatingSystemsServiceListRequest {
	p.max = &max
	return p
}

func (p *OperatingSystemsServiceListRequest) Send() (*OperatingSystemsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.OperatingSystemsService.connection.URL(), p.OperatingSystemsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.OperatingSystemsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.OperatingSystemsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.OperatingSystemsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.OperatingSystemsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.OperatingSystemsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLOperatingSystemInfoReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &OperatingSystemsServiceListResponse{operatingSystem: result}, nil
}

func (p *OperatingSystemsServiceListRequest) MustSend() *OperatingSystemsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of types of operating system available in the system.
// The order of the returned list of operating systems isn't guaranteed.
//
type OperatingSystemsServiceListResponse struct {
	operatingSystem *OperatingSystemInfoSlice
}

func (p *OperatingSystemsServiceListResponse) OperatingSystem() (*OperatingSystemInfoSlice, bool) {
	if p.operatingSystem != nil {
		return p.operatingSystem, true
	}
	return nil, false
}

func (p *OperatingSystemsServiceListResponse) MustOperatingSystem() *OperatingSystemInfoSlice {
	if p.operatingSystem == nil {
		panic("operatingSystem in response does not exist")
	}
	return p.operatingSystem
}

//
// Returns the list of types of operating system available in the system.
// The order of the returned list of operating systems isn't guaranteed.
//
func (p *OperatingSystemsService) List() *OperatingSystemsServiceListRequest {
	return &OperatingSystemsServiceListRequest{OperatingSystemsService: p}
}

//
//
func (op *OperatingSystemsService) OperatingSystemService(id string) *OperatingSystemService {
	return NewOperatingSystemService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *OperatingSystemsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.OperatingSystemService(path), nil
	}
	return op.OperatingSystemService(path[:index]).Service(path[index+1:])
}

func (op *OperatingSystemsService) String() string {
	return fmt.Sprintf("OperatingSystemsService:%s", op.path)
}

//
//
type InstanceTypeGraphicsConsoleService struct {
	BaseService
}

func NewInstanceTypeGraphicsConsoleService(connection *Connection, path string) *InstanceTypeGraphicsConsoleService {
	var result InstanceTypeGraphicsConsoleService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets graphics console configuration of the instance type.
//
type InstanceTypeGraphicsConsoleServiceGetRequest struct {
	InstanceTypeGraphicsConsoleService *InstanceTypeGraphicsConsoleService
	header                             map[string]string
	query                              map[string]string
	follow                             *string
}

func (p *InstanceTypeGraphicsConsoleServiceGetRequest) Header(key, value string) *InstanceTypeGraphicsConsoleServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *InstanceTypeGraphicsConsoleServiceGetRequest) Query(key, value string) *InstanceTypeGraphicsConsoleServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *InstanceTypeGraphicsConsoleServiceGetRequest) Follow(follow string) *InstanceTypeGraphicsConsoleServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *InstanceTypeGraphicsConsoleServiceGetRequest) Send() (*InstanceTypeGraphicsConsoleServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.InstanceTypeGraphicsConsoleService.connection.URL(), p.InstanceTypeGraphicsConsoleService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.InstanceTypeGraphicsConsoleService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.InstanceTypeGraphicsConsoleService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.InstanceTypeGraphicsConsoleService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.InstanceTypeGraphicsConsoleService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.InstanceTypeGraphicsConsoleService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGraphicsConsoleReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &InstanceTypeGraphicsConsoleServiceGetResponse{console: result}, nil
}

func (p *InstanceTypeGraphicsConsoleServiceGetRequest) MustSend() *InstanceTypeGraphicsConsoleServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets graphics console configuration of the instance type.
//
type InstanceTypeGraphicsConsoleServiceGetResponse struct {
	console *GraphicsConsole
}

func (p *InstanceTypeGraphicsConsoleServiceGetResponse) Console() (*GraphicsConsole, bool) {
	if p.console != nil {
		return p.console, true
	}
	return nil, false
}

func (p *InstanceTypeGraphicsConsoleServiceGetResponse) MustConsole() *GraphicsConsole {
	if p.console == nil {
		panic("console in response does not exist")
	}
	return p.console
}

//
// Gets graphics console configuration of the instance type.
//
func (p *InstanceTypeGraphicsConsoleService) Get() *InstanceTypeGraphicsConsoleServiceGetRequest {
	return &InstanceTypeGraphicsConsoleServiceGetRequest{InstanceTypeGraphicsConsoleService: p}
}

//
// Remove the graphics console from the instance type.
//
type InstanceTypeGraphicsConsoleServiceRemoveRequest struct {
	InstanceTypeGraphicsConsoleService *InstanceTypeGraphicsConsoleService
	header                             map[string]string
	query                              map[string]string
	async                              *bool
}

func (p *InstanceTypeGraphicsConsoleServiceRemoveRequest) Header(key, value string) *InstanceTypeGraphicsConsoleServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *InstanceTypeGraphicsConsoleServiceRemoveRequest) Query(key, value string) *InstanceTypeGraphicsConsoleServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *InstanceTypeGraphicsConsoleServiceRemoveRequest) Async(async bool) *InstanceTypeGraphicsConsoleServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *InstanceTypeGraphicsConsoleServiceRemoveRequest) Send() (*InstanceTypeGraphicsConsoleServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.InstanceTypeGraphicsConsoleService.connection.URL(), p.InstanceTypeGraphicsConsoleService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.InstanceTypeGraphicsConsoleService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.InstanceTypeGraphicsConsoleService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.InstanceTypeGraphicsConsoleService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.InstanceTypeGraphicsConsoleService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.InstanceTypeGraphicsConsoleService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(InstanceTypeGraphicsConsoleServiceRemoveResponse), nil
}

func (p *InstanceTypeGraphicsConsoleServiceRemoveRequest) MustSend() *InstanceTypeGraphicsConsoleServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Remove the graphics console from the instance type.
//
type InstanceTypeGraphicsConsoleServiceRemoveResponse struct {
}

//
// Remove the graphics console from the instance type.
//
func (p *InstanceTypeGraphicsConsoleService) Remove() *InstanceTypeGraphicsConsoleServiceRemoveRequest {
	return &InstanceTypeGraphicsConsoleServiceRemoveRequest{InstanceTypeGraphicsConsoleService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *InstanceTypeGraphicsConsoleService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *InstanceTypeGraphicsConsoleService) String() string {
	return fmt.Sprintf("InstanceTypeGraphicsConsoleService:%s", op.path)
}

//
// A service to access a particular device of a host.
//
type HostDeviceService struct {
	BaseService
}

func NewHostDeviceService(connection *Connection, path string) *HostDeviceService {
	var result HostDeviceService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieve information about a particular host's device.
// An example of getting a host device:
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/devices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/123/devices/456" id="456">
//   <name>usb_1_9_1_1_0</name>
//   <capability>usb</capability>
//   <host href="/ovirt-engine/api/hosts/123" id="123"/>
//   <parent_device href="/ovirt-engine/api/hosts/123/devices/789" id="789">
//     <name>usb_1_9_1</name>
//   </parent_device>
// </host_device>
// ----
//
type HostDeviceServiceGetRequest struct {
	HostDeviceService *HostDeviceService
	header            map[string]string
	query             map[string]string
	follow            *string
}

func (p *HostDeviceServiceGetRequest) Header(key, value string) *HostDeviceServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *HostDeviceServiceGetRequest) Query(key, value string) *HostDeviceServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *HostDeviceServiceGetRequest) Follow(follow string) *HostDeviceServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *HostDeviceServiceGetRequest) Send() (*HostDeviceServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.HostDeviceService.connection.URL(), p.HostDeviceService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.HostDeviceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.HostDeviceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.HostDeviceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.HostDeviceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.HostDeviceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLHostDeviceReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &HostDeviceServiceGetResponse{device: result}, nil
}

func (p *HostDeviceServiceGetRequest) MustSend() *HostDeviceServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieve information about a particular host's device.
// An example of getting a host device:
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/devices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/123/devices/456" id="456">
//   <name>usb_1_9_1_1_0</name>
//   <capability>usb</capability>
//   <host href="/ovirt-engine/api/hosts/123" id="123"/>
//   <parent_device href="/ovirt-engine/api/hosts/123/devices/789" id="789">
//     <name>usb_1_9_1</name>
//   </parent_device>
// </host_device>
// ----
//
type HostDeviceServiceGetResponse struct {
	device *HostDevice
}

func (p *HostDeviceServiceGetResponse) Device() (*HostDevice, bool) {
	if p.device != nil {
		return p.device, true
	}
	return nil, false
}

func (p *HostDeviceServiceGetResponse) MustDevice() *HostDevice {
	if p.device == nil {
		panic("device in response does not exist")
	}
	return p.device
}

//
// Retrieve information about a particular host's device.
// An example of getting a host device:
// [source]
// ----
// GET /ovirt-engine/api/hosts/123/devices/456
// ----
// [source,xml]
// ----
// <host_device href="/ovirt-engine/api/hosts/123/devices/456" id="456">
//   <name>usb_1_9_1_1_0</name>
//   <capability>usb</capability>
//   <host href="/ovirt-engine/api/hosts/123" id="123"/>
//   <parent_device href="/ovirt-engine/api/hosts/123/devices/789" id="789">
//     <name>usb_1_9_1</name>
//   </parent_device>
// </host_device>
// ----
//
func (p *HostDeviceService) Get() *HostDeviceServiceGetRequest {
	return &HostDeviceServiceGetRequest{HostDeviceService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *HostDeviceService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *HostDeviceService) String() string {
	return fmt.Sprintf("HostDeviceService:%s", op.path)
}

//
//
type VmNicsService struct {
	BaseService
}

func NewVmNicsService(connection *Connection, path string) *VmNicsService {
	var result VmNicsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a NIC to the virtual machine.
// The following example adds to the virtual machine `123` a network interface named `mynic` using `virtio` and the
// NIC profile `456`.
// [source]
// ----
// POST /ovirt-engine/api/vms/123/nics
// ----
// [source,xml]
// ----
// <nic>
//   <name>mynic</name>
//   <interface>virtio</interface>
//   <vnic_profile id="456"/>
// </nic>
// ----
// The following example sends that request using `curl`:
// [source,bash]
// ----
// curl \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --user "admin@internal:mypassword" \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --data '
// <nic>
//   <name>mynic</name>
//   <interface>virtio</interface>
//   <vnic_profile id="456"/>
// </nic>
// ' \
// https://myengine.example.com/ovirt-engine/api/vms/123/nics
// ----
// [IMPORTANT]
// ====
// The hotplugging feature only supports virtual machine operating systems with hotplugging operations.
// Example operating systems include:
// - Red Hat Enterprise Linux 6
// - Red Hat Enterprise Linux 5
// - Windows Server 2008 and
// - Windows Server 2003
// ====
//
type VmNicsServiceAddRequest struct {
	VmNicsService *VmNicsService
	header        map[string]string
	query         map[string]string
	nic           *Nic
}

func (p *VmNicsServiceAddRequest) Header(key, value string) *VmNicsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmNicsServiceAddRequest) Query(key, value string) *VmNicsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmNicsServiceAddRequest) Nic(nic *Nic) *VmNicsServiceAddRequest {
	p.nic = nic
	return p
}

func (p *VmNicsServiceAddRequest) Send() (*VmNicsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmNicsService.connection.URL(), p.VmNicsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLNicWriteOne(writer, p.nic, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmNicsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmNicsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmNicsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmNicsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmNicsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNicReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmNicsServiceAddResponse{nic: result}, nil
}

func (p *VmNicsServiceAddRequest) MustSend() *VmNicsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a NIC to the virtual machine.
// The following example adds to the virtual machine `123` a network interface named `mynic` using `virtio` and the
// NIC profile `456`.
// [source]
// ----
// POST /ovirt-engine/api/vms/123/nics
// ----
// [source,xml]
// ----
// <nic>
//   <name>mynic</name>
//   <interface>virtio</interface>
//   <vnic_profile id="456"/>
// </nic>
// ----
// The following example sends that request using `curl`:
// [source,bash]
// ----
// curl \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --user "admin@internal:mypassword" \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --data '
// <nic>
//   <name>mynic</name>
//   <interface>virtio</interface>
//   <vnic_profile id="456"/>
// </nic>
// ' \
// https://myengine.example.com/ovirt-engine/api/vms/123/nics
// ----
// [IMPORTANT]
// ====
// The hotplugging feature only supports virtual machine operating systems with hotplugging operations.
// Example operating systems include:
// - Red Hat Enterprise Linux 6
// - Red Hat Enterprise Linux 5
// - Windows Server 2008 and
// - Windows Server 2003
// ====
//
type VmNicsServiceAddResponse struct {
	nic *Nic
}

func (p *VmNicsServiceAddResponse) Nic() (*Nic, bool) {
	if p.nic != nil {
		return p.nic, true
	}
	return nil, false
}

func (p *VmNicsServiceAddResponse) MustNic() *Nic {
	if p.nic == nil {
		panic("nic in response does not exist")
	}
	return p.nic
}

//
// Adds a NIC to the virtual machine.
// The following example adds to the virtual machine `123` a network interface named `mynic` using `virtio` and the
// NIC profile `456`.
// [source]
// ----
// POST /ovirt-engine/api/vms/123/nics
// ----
// [source,xml]
// ----
// <nic>
//   <name>mynic</name>
//   <interface>virtio</interface>
//   <vnic_profile id="456"/>
// </nic>
// ----
// The following example sends that request using `curl`:
// [source,bash]
// ----
// curl \
// --request POST \
// --header "Version: 4" \
// --header "Content-Type: application/xml" \
// --header "Accept: application/xml" \
// --user "admin@internal:mypassword" \
// --cacert /etc/pki/ovirt-engine/ca.pem \
// --data '
// <nic>
//   <name>mynic</name>
//   <interface>virtio</interface>
//   <vnic_profile id="456"/>
// </nic>
// ' \
// https://myengine.example.com/ovirt-engine/api/vms/123/nics
// ----
// [IMPORTANT]
// ====
// The hotplugging feature only supports virtual machine operating systems with hotplugging operations.
// Example operating systems include:
// - Red Hat Enterprise Linux 6
// - Red Hat Enterprise Linux 5
// - Windows Server 2008 and
// - Windows Server 2003
// ====
//
func (p *VmNicsService) Add() *VmNicsServiceAddRequest {
	return &VmNicsServiceAddRequest{VmNicsService: p}
}

//
// Returns the list of NICs of the virtual machine.
// The order of the returned list of NICs isn't guaranteed.
//
type VmNicsServiceListRequest struct {
	VmNicsService *VmNicsService
	header        map[string]string
	query         map[string]string
	follow        *string
	max           *int64
}

func (p *VmNicsServiceListRequest) Header(key, value string) *VmNicsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmNicsServiceListRequest) Query(key, value string) *VmNicsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmNicsServiceListRequest) Follow(follow string) *VmNicsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *VmNicsServiceListRequest) Max(max int64) *VmNicsServiceListRequest {
	p.max = &max
	return p
}

func (p *VmNicsServiceListRequest) Send() (*VmNicsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmNicsService.connection.URL(), p.VmNicsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmNicsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmNicsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmNicsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmNicsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmNicsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLNicReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &VmNicsServiceListResponse{nics: result}, nil
}

func (p *VmNicsServiceListRequest) MustSend() *VmNicsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of NICs of the virtual machine.
// The order of the returned list of NICs isn't guaranteed.
//
type VmNicsServiceListResponse struct {
	nics *NicSlice
}

func (p *VmNicsServiceListResponse) Nics() (*NicSlice, bool) {
	if p.nics != nil {
		return p.nics, true
	}
	return nil, false
}

func (p *VmNicsServiceListResponse) MustNics() *NicSlice {
	if p.nics == nil {
		panic("nics in response does not exist")
	}
	return p.nics
}

//
// Returns the list of NICs of the virtual machine.
// The order of the returned list of NICs isn't guaranteed.
//
func (p *VmNicsService) List() *VmNicsServiceListRequest {
	return &VmNicsServiceListRequest{VmNicsService: p}
}

//
//
func (op *VmNicsService) NicService(id string) *VmNicService {
	return NewVmNicService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmNicsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.NicService(path), nil
	}
	return op.NicService(path[:index]).Service(path[index+1:])
}

func (op *VmNicsService) String() string {
	return fmt.Sprintf("VmNicsService:%s", op.path)
}

//
//
type VmReportedDevicesService struct {
	BaseService
}

func NewVmReportedDevicesService(connection *Connection, path string) *VmReportedDevicesService {
	var result VmReportedDevicesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of reported devices of the virtual machine.
// The order of the returned list of devices isn't guaranteed.
//
type VmReportedDevicesServiceListRequest struct {
	VmReportedDevicesService *VmReportedDevicesService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
	max                      *int64
}

func (p *VmReportedDevicesServiceListRequest) Header(key, value string) *VmReportedDevicesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmReportedDevicesServiceListRequest) Query(key, value string) *VmReportedDevicesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmReportedDevicesServiceListRequest) Follow(follow string) *VmReportedDevicesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *VmReportedDevicesServiceListRequest) Max(max int64) *VmReportedDevicesServiceListRequest {
	p.max = &max
	return p
}

func (p *VmReportedDevicesServiceListRequest) Send() (*VmReportedDevicesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmReportedDevicesService.connection.URL(), p.VmReportedDevicesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmReportedDevicesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmReportedDevicesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmReportedDevicesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmReportedDevicesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmReportedDevicesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLReportedDeviceReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &VmReportedDevicesServiceListResponse{reportedDevice: result}, nil
}

func (p *VmReportedDevicesServiceListRequest) MustSend() *VmReportedDevicesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of reported devices of the virtual machine.
// The order of the returned list of devices isn't guaranteed.
//
type VmReportedDevicesServiceListResponse struct {
	reportedDevice *ReportedDeviceSlice
}

func (p *VmReportedDevicesServiceListResponse) ReportedDevice() (*ReportedDeviceSlice, bool) {
	if p.reportedDevice != nil {
		return p.reportedDevice, true
	}
	return nil, false
}

func (p *VmReportedDevicesServiceListResponse) MustReportedDevice() *ReportedDeviceSlice {
	if p.reportedDevice == nil {
		panic("reportedDevice in response does not exist")
	}
	return p.reportedDevice
}

//
// Returns the list of reported devices of the virtual machine.
// The order of the returned list of devices isn't guaranteed.
//
func (p *VmReportedDevicesService) List() *VmReportedDevicesServiceListRequest {
	return &VmReportedDevicesServiceListRequest{VmReportedDevicesService: p}
}

//
//
func (op *VmReportedDevicesService) ReportedDeviceService(id string) *VmReportedDeviceService {
	return NewVmReportedDeviceService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmReportedDevicesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ReportedDeviceService(path), nil
	}
	return op.ReportedDeviceService(path[:index]).Service(path[index+1:])
}

func (op *VmReportedDevicesService) String() string {
	return fmt.Sprintf("VmReportedDevicesService:%s", op.path)
}

//
// This service represents list of vms that have a specific
// label when accessed through the affinitylabels/vms
// subcollection.
//
type AffinityLabelVmsService struct {
	BaseService
}

func NewAffinityLabelVmsService(connection *Connection, path string) *AffinityLabelVmsService {
	var result AffinityLabelVmsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a label to a vm.
//
type AffinityLabelVmsServiceAddRequest struct {
	AffinityLabelVmsService *AffinityLabelVmsService
	header                  map[string]string
	query                   map[string]string
	vm                      *Vm
}

func (p *AffinityLabelVmsServiceAddRequest) Header(key, value string) *AffinityLabelVmsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelVmsServiceAddRequest) Query(key, value string) *AffinityLabelVmsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelVmsServiceAddRequest) Vm(vm *Vm) *AffinityLabelVmsServiceAddRequest {
	p.vm = vm
	return p
}

func (p *AffinityLabelVmsServiceAddRequest) Send() (*AffinityLabelVmsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelVmsService.connection.URL(), p.AffinityLabelVmsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLVmWriteOne(writer, p.vm, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelVmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelVmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelVmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelVmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelVmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelVmsServiceAddResponse{vm: result}, nil
}

func (p *AffinityLabelVmsServiceAddRequest) MustSend() *AffinityLabelVmsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a label to a vm.
//
type AffinityLabelVmsServiceAddResponse struct {
	vm *Vm
}

func (p *AffinityLabelVmsServiceAddResponse) Vm() (*Vm, bool) {
	if p.vm != nil {
		return p.vm, true
	}
	return nil, false
}

func (p *AffinityLabelVmsServiceAddResponse) MustVm() *Vm {
	if p.vm == nil {
		panic("vm in response does not exist")
	}
	return p.vm
}

//
// Add a label to a vm.
//
func (p *AffinityLabelVmsService) Add() *AffinityLabelVmsServiceAddRequest {
	return &AffinityLabelVmsServiceAddRequest{AffinityLabelVmsService: p}
}

//
// List all virtual machines with the label.
// The order of the returned virtual machines isn't guaranteed.
//
type AffinityLabelVmsServiceListRequest struct {
	AffinityLabelVmsService *AffinityLabelVmsService
	header                  map[string]string
	query                   map[string]string
	follow                  *string
}

func (p *AffinityLabelVmsServiceListRequest) Header(key, value string) *AffinityLabelVmsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelVmsServiceListRequest) Query(key, value string) *AffinityLabelVmsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelVmsServiceListRequest) Follow(follow string) *AffinityLabelVmsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *AffinityLabelVmsServiceListRequest) Send() (*AffinityLabelVmsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelVmsService.connection.URL(), p.AffinityLabelVmsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelVmsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelVmsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelVmsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelVmsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelVmsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLVmReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &AffinityLabelVmsServiceListResponse{vms: result}, nil
}

func (p *AffinityLabelVmsServiceListRequest) MustSend() *AffinityLabelVmsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// List all virtual machines with the label.
// The order of the returned virtual machines isn't guaranteed.
//
type AffinityLabelVmsServiceListResponse struct {
	vms *VmSlice
}

func (p *AffinityLabelVmsServiceListResponse) Vms() (*VmSlice, bool) {
	if p.vms != nil {
		return p.vms, true
	}
	return nil, false
}

func (p *AffinityLabelVmsServiceListResponse) MustVms() *VmSlice {
	if p.vms == nil {
		panic("vms in response does not exist")
	}
	return p.vms
}

//
// List all virtual machines with the label.
// The order of the returned virtual machines isn't guaranteed.
//
func (p *AffinityLabelVmsService) List() *AffinityLabelVmsServiceListRequest {
	return &AffinityLabelVmsServiceListRequest{AffinityLabelVmsService: p}
}

//
// A link to the specific label-vm assignment to
// allow label removal.
//
func (op *AffinityLabelVmsService) VmService(id string) *AffinityLabelVmService {
	return NewAffinityLabelVmService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityLabelVmsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.VmService(path), nil
	}
	return op.VmService(path[:index]).Service(path[index+1:])
}

func (op *AffinityLabelVmsService) String() string {
	return fmt.Sprintf("AffinityLabelVmsService:%s", op.path)
}

//
//
type DiskSnapshotService struct {
	BaseService
}

func NewDiskSnapshotService(connection *Connection, path string) *DiskSnapshotService {
	var result DiskSnapshotService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type DiskSnapshotServiceGetRequest struct {
	DiskSnapshotService *DiskSnapshotService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *DiskSnapshotServiceGetRequest) Header(key, value string) *DiskSnapshotServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskSnapshotServiceGetRequest) Query(key, value string) *DiskSnapshotServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskSnapshotServiceGetRequest) Follow(follow string) *DiskSnapshotServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *DiskSnapshotServiceGetRequest) Send() (*DiskSnapshotServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskSnapshotService.connection.URL(), p.DiskSnapshotService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskSnapshotService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskSnapshotService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskSnapshotService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskSnapshotService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskSnapshotService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskSnapshotReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DiskSnapshotServiceGetResponse{snapshot: result}, nil
}

func (p *DiskSnapshotServiceGetRequest) MustSend() *DiskSnapshotServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskSnapshotServiceGetResponse struct {
	snapshot *DiskSnapshot
}

func (p *DiskSnapshotServiceGetResponse) Snapshot() (*DiskSnapshot, bool) {
	if p.snapshot != nil {
		return p.snapshot, true
	}
	return nil, false
}

func (p *DiskSnapshotServiceGetResponse) MustSnapshot() *DiskSnapshot {
	if p.snapshot == nil {
		panic("snapshot in response does not exist")
	}
	return p.snapshot
}

//
//
func (p *DiskSnapshotService) Get() *DiskSnapshotServiceGetRequest {
	return &DiskSnapshotServiceGetRequest{DiskSnapshotService: p}
}

//
//
type DiskSnapshotServiceRemoveRequest struct {
	DiskSnapshotService *DiskSnapshotService
	header              map[string]string
	query               map[string]string
	async               *bool
}

func (p *DiskSnapshotServiceRemoveRequest) Header(key, value string) *DiskSnapshotServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DiskSnapshotServiceRemoveRequest) Query(key, value string) *DiskSnapshotServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DiskSnapshotServiceRemoveRequest) Async(async bool) *DiskSnapshotServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *DiskSnapshotServiceRemoveRequest) Send() (*DiskSnapshotServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DiskSnapshotService.connection.URL(), p.DiskSnapshotService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DiskSnapshotService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DiskSnapshotService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DiskSnapshotService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DiskSnapshotService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DiskSnapshotService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(DiskSnapshotServiceRemoveResponse), nil
}

func (p *DiskSnapshotServiceRemoveRequest) MustSend() *DiskSnapshotServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type DiskSnapshotServiceRemoveResponse struct {
}

//
//
func (p *DiskSnapshotService) Remove() *DiskSnapshotServiceRemoveRequest {
	return &DiskSnapshotServiceRemoveRequest{DiskSnapshotService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DiskSnapshotService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *DiskSnapshotService) String() string {
	return fmt.Sprintf("DiskSnapshotService:%s", op.path)
}

//
// Manages the set of storage domains in the system.
//
type StorageDomainsService struct {
	BaseService
}

func NewStorageDomainsService(connection *Connection, path string) *StorageDomainsService {
	var result StorageDomainsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a new storage domain.
// Creation of a new <<types/storage_domain,StorageDomain>> requires the `name`, `type`, `host`, and `storage`
// attributes. Identify the `host` attribute with the `id` or `name` attributes. In {product-name} 3.6 and
// later you can enable the wipe after delete option by default on the storage domain. To configure this, specify
// `wipe_after_delete` in the POST request. This option can be edited after the domain is created, but doing so will
// not change the wipe after delete property of disks that already exist.
// To add a new storage domain with specified `name`, `type`, `storage.type`, `storage.address`, and `storage.path`,
// and using a host with an id `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageDomains
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>mydata</name>
//   <type>data</type>
//   <storage>
//     <type>nfs</type>
//     <address>mynfs.example.com</address>
//     <path>/exports/mydata</path>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
// To create a new NFS ISO storage domain send a request like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>myisos</name>
//   <type>iso</type>
//   <storage>
//     <type>nfs</type>
//     <address>mynfs.example.com</address>
//     <path>/export/myisos</path>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
// To create a new iSCSI storage domain send a request like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>myiscsi</name>
//   <type>data</type>
//   <storage>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="3600144f09dbd050000004eedbd340001"/>
//       <logical_unit id="3600144f09dbd050000004eedbd340002"/>
//     </logical_units>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
//
type StorageDomainsServiceAddRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	storageDomain         *StorageDomain
}

func (p *StorageDomainsServiceAddRequest) Header(key, value string) *StorageDomainsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceAddRequest) Query(key, value string) *StorageDomainsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceAddRequest) StorageDomain(storageDomain *StorageDomain) *StorageDomainsServiceAddRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *StorageDomainsServiceAddRequest) Send() (*StorageDomainsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLStorageDomainWriteOne(writer, p.storageDomain, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageDomainReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageDomainsServiceAddResponse{storageDomain: result}, nil
}

func (p *StorageDomainsServiceAddRequest) MustSend() *StorageDomainsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a new storage domain.
// Creation of a new <<types/storage_domain,StorageDomain>> requires the `name`, `type`, `host`, and `storage`
// attributes. Identify the `host` attribute with the `id` or `name` attributes. In {product-name} 3.6 and
// later you can enable the wipe after delete option by default on the storage domain. To configure this, specify
// `wipe_after_delete` in the POST request. This option can be edited after the domain is created, but doing so will
// not change the wipe after delete property of disks that already exist.
// To add a new storage domain with specified `name`, `type`, `storage.type`, `storage.address`, and `storage.path`,
// and using a host with an id `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageDomains
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>mydata</name>
//   <type>data</type>
//   <storage>
//     <type>nfs</type>
//     <address>mynfs.example.com</address>
//     <path>/exports/mydata</path>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
// To create a new NFS ISO storage domain send a request like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>myisos</name>
//   <type>iso</type>
//   <storage>
//     <type>nfs</type>
//     <address>mynfs.example.com</address>
//     <path>/export/myisos</path>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
// To create a new iSCSI storage domain send a request like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>myiscsi</name>
//   <type>data</type>
//   <storage>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="3600144f09dbd050000004eedbd340001"/>
//       <logical_unit id="3600144f09dbd050000004eedbd340002"/>
//     </logical_units>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
//
type StorageDomainsServiceAddResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainsServiceAddResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainsServiceAddResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Adds a new storage domain.
// Creation of a new <<types/storage_domain,StorageDomain>> requires the `name`, `type`, `host`, and `storage`
// attributes. Identify the `host` attribute with the `id` or `name` attributes. In {product-name} 3.6 and
// later you can enable the wipe after delete option by default on the storage domain. To configure this, specify
// `wipe_after_delete` in the POST request. This option can be edited after the domain is created, but doing so will
// not change the wipe after delete property of disks that already exist.
// To add a new storage domain with specified `name`, `type`, `storage.type`, `storage.address`, and `storage.path`,
// and using a host with an id `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageDomains
// ----
// With a request body like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>mydata</name>
//   <type>data</type>
//   <storage>
//     <type>nfs</type>
//     <address>mynfs.example.com</address>
//     <path>/exports/mydata</path>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
// To create a new NFS ISO storage domain send a request like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>myisos</name>
//   <type>iso</type>
//   <storage>
//     <type>nfs</type>
//     <address>mynfs.example.com</address>
//     <path>/export/myisos</path>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
// To create a new iSCSI storage domain send a request like this:
// [source,xml]
// ----
// <storage_domain>
//   <name>myiscsi</name>
//   <type>data</type>
//   <storage>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="3600144f09dbd050000004eedbd340001"/>
//       <logical_unit id="3600144f09dbd050000004eedbd340002"/>
//     </logical_units>
//   </storage>
//   <host>
//     <name>myhost</name>
//   </host>
// </storage_domain>
// ----
//
func (p *StorageDomainsService) Add() *StorageDomainsServiceAddRequest {
	return &StorageDomainsServiceAddRequest{StorageDomainsService: p}
}

//
// Import an existing block storage domain to the system using the targets already connected to the host.
//
type StorageDomainsServiceAddBlockDomainRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	storageDomain         *StorageDomain
}

func (p *StorageDomainsServiceAddBlockDomainRequest) Header(key, value string) *StorageDomainsServiceAddBlockDomainRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceAddBlockDomainRequest) Query(key, value string) *StorageDomainsServiceAddBlockDomainRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceAddBlockDomainRequest) StorageDomain(storageDomain *StorageDomain) *StorageDomainsServiceAddBlockDomainRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *StorageDomainsServiceAddBlockDomainRequest) Send() (*StorageDomainsServiceAddBlockDomainResponse, error) {
	rawURL := fmt.Sprintf("%s%s/blockdomain", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustStorageDomain()
	return &StorageDomainsServiceAddBlockDomainResponse{storageDomain: result}, nil
}

func (p *StorageDomainsServiceAddBlockDomainRequest) MustSend() *StorageDomainsServiceAddBlockDomainResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Import an existing block storage domain to the system using the targets already connected to the host.
//
type StorageDomainsServiceAddBlockDomainResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainsServiceAddBlockDomainResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainsServiceAddBlockDomainResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Import an existing block storage domain to the system using the targets already connected to the host.
//
func (p *StorageDomainsService) AddBlockDomain() *StorageDomainsServiceAddBlockDomainRequest {
	return &StorageDomainsServiceAddBlockDomainRequest{StorageDomainsService: p}
}

//
// Add a new storage domain to the system using the storage on the given host and path.
//
type StorageDomainsServiceAddByPathRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	storageDomain         *StorageDomain
}

func (p *StorageDomainsServiceAddByPathRequest) Header(key, value string) *StorageDomainsServiceAddByPathRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceAddByPathRequest) Query(key, value string) *StorageDomainsServiceAddByPathRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceAddByPathRequest) StorageDomain(storageDomain *StorageDomain) *StorageDomainsServiceAddByPathRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *StorageDomainsServiceAddByPathRequest) Send() (*StorageDomainsServiceAddByPathResponse, error) {
	rawURL := fmt.Sprintf("%s%s/bypath", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustStorageDomain()
	return &StorageDomainsServiceAddByPathResponse{storageDomain: result}, nil
}

func (p *StorageDomainsServiceAddByPathRequest) MustSend() *StorageDomainsServiceAddByPathResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new storage domain to the system using the storage on the given host and path.
//
type StorageDomainsServiceAddByPathResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainsServiceAddByPathResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainsServiceAddByPathResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Add a new storage domain to the system using the storage on the given host and path.
//
func (p *StorageDomainsService) AddByPath() *StorageDomainsServiceAddByPathRequest {
	return &StorageDomainsServiceAddByPathRequest{StorageDomainsService: p}
}

//
// Add a new storage domain to the system using a direct LUN.
//
type StorageDomainsServiceAddDirectLunRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	storageDomain         *StorageDomain
}

func (p *StorageDomainsServiceAddDirectLunRequest) Header(key, value string) *StorageDomainsServiceAddDirectLunRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceAddDirectLunRequest) Query(key, value string) *StorageDomainsServiceAddDirectLunRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceAddDirectLunRequest) StorageDomain(storageDomain *StorageDomain) *StorageDomainsServiceAddDirectLunRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *StorageDomainsServiceAddDirectLunRequest) Send() (*StorageDomainsServiceAddDirectLunResponse, error) {
	rawURL := fmt.Sprintf("%s%s/directlun", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustStorageDomain()
	return &StorageDomainsServiceAddDirectLunResponse{storageDomain: result}, nil
}

func (p *StorageDomainsServiceAddDirectLunRequest) MustSend() *StorageDomainsServiceAddDirectLunResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new storage domain to the system using a direct LUN.
//
type StorageDomainsServiceAddDirectLunResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainsServiceAddDirectLunResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainsServiceAddDirectLunResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Add a new storage domain to the system using a direct LUN.
//
func (p *StorageDomainsService) AddDirectLun() *StorageDomainsServiceAddDirectLunRequest {
	return &StorageDomainsServiceAddDirectLunRequest{StorageDomainsService: p}
}

//
// Add a new storage domain to the system using Gluster or POSIX FS storage.
//
type StorageDomainsServiceAddGlusterOrPostfsRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	storageDomain         *StorageDomain
}

func (p *StorageDomainsServiceAddGlusterOrPostfsRequest) Header(key, value string) *StorageDomainsServiceAddGlusterOrPostfsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceAddGlusterOrPostfsRequest) Query(key, value string) *StorageDomainsServiceAddGlusterOrPostfsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceAddGlusterOrPostfsRequest) StorageDomain(storageDomain *StorageDomain) *StorageDomainsServiceAddGlusterOrPostfsRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *StorageDomainsServiceAddGlusterOrPostfsRequest) Send() (*StorageDomainsServiceAddGlusterOrPostfsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/glusterorpostfs", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustStorageDomain()
	return &StorageDomainsServiceAddGlusterOrPostfsResponse{storageDomain: result}, nil
}

func (p *StorageDomainsServiceAddGlusterOrPostfsRequest) MustSend() *StorageDomainsServiceAddGlusterOrPostfsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new storage domain to the system using Gluster or POSIX FS storage.
//
type StorageDomainsServiceAddGlusterOrPostfsResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainsServiceAddGlusterOrPostfsResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainsServiceAddGlusterOrPostfsResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Add a new storage domain to the system using Gluster or POSIX FS storage.
//
func (p *StorageDomainsService) AddGlusterOrPostfs() *StorageDomainsServiceAddGlusterOrPostfsRequest {
	return &StorageDomainsServiceAddGlusterOrPostfsRequest{StorageDomainsService: p}
}

//
// Returns the list of storage domains in the system.
// The order of the returned list of storage domains is guaranteed only if the `sortby` clause is included
// in the `search` parameter.
//
type StorageDomainsServiceListRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	caseSensitive         *bool
	filter                *bool
	follow                *string
	max                   *int64
	search                *string
}

func (p *StorageDomainsServiceListRequest) Header(key, value string) *StorageDomainsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceListRequest) Query(key, value string) *StorageDomainsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceListRequest) CaseSensitive(caseSensitive bool) *StorageDomainsServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *StorageDomainsServiceListRequest) Filter(filter bool) *StorageDomainsServiceListRequest {
	p.filter = &filter
	return p
}

func (p *StorageDomainsServiceListRequest) Follow(follow string) *StorageDomainsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *StorageDomainsServiceListRequest) Max(max int64) *StorageDomainsServiceListRequest {
	p.max = &max
	return p
}

func (p *StorageDomainsServiceListRequest) Search(search string) *StorageDomainsServiceListRequest {
	p.search = &search
	return p
}

func (p *StorageDomainsServiceListRequest) Send() (*StorageDomainsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageDomainReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &StorageDomainsServiceListResponse{storageDomains: result}, nil
}

func (p *StorageDomainsServiceListRequest) MustSend() *StorageDomainsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of storage domains in the system.
// The order of the returned list of storage domains is guaranteed only if the `sortby` clause is included
// in the `search` parameter.
//
type StorageDomainsServiceListResponse struct {
	storageDomains *StorageDomainSlice
}

func (p *StorageDomainsServiceListResponse) StorageDomains() (*StorageDomainSlice, bool) {
	if p.storageDomains != nil {
		return p.storageDomains, true
	}
	return nil, false
}

func (p *StorageDomainsServiceListResponse) MustStorageDomains() *StorageDomainSlice {
	if p.storageDomains == nil {
		panic("storageDomains in response does not exist")
	}
	return p.storageDomains
}

//
// Returns the list of storage domains in the system.
// The order of the returned list of storage domains is guaranteed only if the `sortby` clause is included
// in the `search` parameter.
//
func (p *StorageDomainsService) List() *StorageDomainsServiceListRequest {
	return &StorageDomainsServiceListRequest{StorageDomainsService: p}
}

//
// Add a new storage domain to the system using the storage on the local host at the given path.
//
type StorageDomainsServiceAddLocalRequest struct {
	StorageDomainsService *StorageDomainsService
	header                map[string]string
	query                 map[string]string
	storageDomain         *StorageDomain
}

func (p *StorageDomainsServiceAddLocalRequest) Header(key, value string) *StorageDomainsServiceAddLocalRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainsServiceAddLocalRequest) Query(key, value string) *StorageDomainsServiceAddLocalRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainsServiceAddLocalRequest) StorageDomain(storageDomain *StorageDomain) *StorageDomainsServiceAddLocalRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *StorageDomainsServiceAddLocalRequest) Send() (*StorageDomainsServiceAddLocalResponse, error) {
	rawURL := fmt.Sprintf("%s%s/local", p.StorageDomainsService.connection.URL(), p.StorageDomainsService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.StorageDomain(p.storageDomain)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustStorageDomain()
	return &StorageDomainsServiceAddLocalResponse{storageDomain: result}, nil
}

func (p *StorageDomainsServiceAddLocalRequest) MustSend() *StorageDomainsServiceAddLocalResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new storage domain to the system using the storage on the local host at the given path.
//
type StorageDomainsServiceAddLocalResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainsServiceAddLocalResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainsServiceAddLocalResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Add a new storage domain to the system using the storage on the local host at the given path.
//
func (p *StorageDomainsService) AddLocal() *StorageDomainsServiceAddLocalRequest {
	return &StorageDomainsServiceAddLocalRequest{StorageDomainsService: p}
}

//
//
func (op *StorageDomainsService) StorageDomainService(id string) *StorageDomainService {
	return NewStorageDomainService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageDomainsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.StorageDomainService(path), nil
	}
	return op.StorageDomainService(path[:index]).Service(path[index+1:])
}

func (op *StorageDomainsService) String() string {
	return fmt.Sprintf("StorageDomainsService:%s", op.path)
}

//
// Provides capability to manage external providers.
//
type ExternalProviderService struct {
	BaseService
}

func NewExternalProviderService(connection *Connection, path string) *ExternalProviderService {
	var result ExternalProviderService
	result.connection = connection
	result.path = path
	return &result
}

//
// Import the SSL certificates of the external host provider.
//
type ExternalProviderServiceImportCertificatesRequest struct {
	ExternalProviderService *ExternalProviderService
	header                  map[string]string
	query                   map[string]string
	certificates            *CertificateSlice
}

func (p *ExternalProviderServiceImportCertificatesRequest) Header(key, value string) *ExternalProviderServiceImportCertificatesRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalProviderServiceImportCertificatesRequest) Query(key, value string) *ExternalProviderServiceImportCertificatesRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalProviderServiceImportCertificatesRequest) Certificates(certificates *CertificateSlice) *ExternalProviderServiceImportCertificatesRequest {
	p.certificates = certificates
	return p
}

func (p *ExternalProviderServiceImportCertificatesRequest) CertificatesOfAny(anys ...*Certificate) *ExternalProviderServiceImportCertificatesRequest {
	if p.certificates == nil {
		p.certificates = new(CertificateSlice)
	}
	p.certificates.slice = append(p.certificates.slice, anys...)
	return p
}

func (p *ExternalProviderServiceImportCertificatesRequest) Send() (*ExternalProviderServiceImportCertificatesResponse, error) {
	rawURL := fmt.Sprintf("%s%s/importcertificates", p.ExternalProviderService.connection.URL(), p.ExternalProviderService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Certificates(p.certificates)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(ExternalProviderServiceImportCertificatesResponse), nil
}

func (p *ExternalProviderServiceImportCertificatesRequest) MustSend() *ExternalProviderServiceImportCertificatesResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Import the SSL certificates of the external host provider.
//
type ExternalProviderServiceImportCertificatesResponse struct {
}

//
// Import the SSL certificates of the external host provider.
//
func (p *ExternalProviderService) ImportCertificates() *ExternalProviderServiceImportCertificatesRequest {
	return &ExternalProviderServiceImportCertificatesRequest{ExternalProviderService: p}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type ExternalProviderServiceTestConnectivityRequest struct {
	ExternalProviderService *ExternalProviderService
	header                  map[string]string
	query                   map[string]string
	async                   *bool
}

func (p *ExternalProviderServiceTestConnectivityRequest) Header(key, value string) *ExternalProviderServiceTestConnectivityRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalProviderServiceTestConnectivityRequest) Query(key, value string) *ExternalProviderServiceTestConnectivityRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalProviderServiceTestConnectivityRequest) Async(async bool) *ExternalProviderServiceTestConnectivityRequest {
	p.async = &async
	return p
}

func (p *ExternalProviderServiceTestConnectivityRequest) Send() (*ExternalProviderServiceTestConnectivityResponse, error) {
	rawURL := fmt.Sprintf("%s%s/testconnectivity", p.ExternalProviderService.connection.URL(), p.ExternalProviderService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(ExternalProviderServiceTestConnectivityResponse), nil
}

func (p *ExternalProviderServiceTestConnectivityRequest) MustSend() *ExternalProviderServiceTestConnectivityResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type ExternalProviderServiceTestConnectivityResponse struct {
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
func (p *ExternalProviderService) TestConnectivity() *ExternalProviderServiceTestConnectivityRequest {
	return &ExternalProviderServiceTestConnectivityRequest{ExternalProviderService: p}
}

//
// A service to view certificates for this external provider.
//
func (op *ExternalProviderService) CertificatesService() *ExternalProviderCertificatesService {
	return NewExternalProviderCertificatesService(op.connection, fmt.Sprintf("%s/certificates", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalProviderService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "certificates" {
		return op.CertificatesService(), nil
	}
	if strings.HasPrefix(path, "certificates/") {
		return op.CertificatesService().Service(path[13:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalProviderService) String() string {
	return fmt.Sprintf("ExternalProviderService:%s", op.path)
}

//
// A service managing a watchdog on virtual machines.
//
type VmWatchdogService struct {
	BaseService
}

func NewVmWatchdogService(connection *Connection, path string) *VmWatchdogService {
	var result VmWatchdogService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the information about the watchdog.
//
type VmWatchdogServiceGetRequest struct {
	VmWatchdogService *VmWatchdogService
	header            map[string]string
	query             map[string]string
	follow            *string
}

func (p *VmWatchdogServiceGetRequest) Header(key, value string) *VmWatchdogServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmWatchdogServiceGetRequest) Query(key, value string) *VmWatchdogServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmWatchdogServiceGetRequest) Follow(follow string) *VmWatchdogServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *VmWatchdogServiceGetRequest) Send() (*VmWatchdogServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmWatchdogService.connection.URL(), p.VmWatchdogService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmWatchdogService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmWatchdogService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmWatchdogService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmWatchdogService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmWatchdogService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLWatchdogReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmWatchdogServiceGetResponse{watchdog: result}, nil
}

func (p *VmWatchdogServiceGetRequest) MustSend() *VmWatchdogServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the information about the watchdog.
//
type VmWatchdogServiceGetResponse struct {
	watchdog *Watchdog
}

func (p *VmWatchdogServiceGetResponse) Watchdog() (*Watchdog, bool) {
	if p.watchdog != nil {
		return p.watchdog, true
	}
	return nil, false
}

func (p *VmWatchdogServiceGetResponse) MustWatchdog() *Watchdog {
	if p.watchdog == nil {
		panic("watchdog in response does not exist")
	}
	return p.watchdog
}

//
// Returns the information about the watchdog.
//
func (p *VmWatchdogService) Get() *VmWatchdogServiceGetRequest {
	return &VmWatchdogServiceGetRequest{VmWatchdogService: p}
}

//
// Removes the watchdog from the virtual machine.
// For example, to remove a watchdog from a virtual machine, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/vms/123/watchdogs/00000000-0000-0000-0000-000000000000
// ----
//
type VmWatchdogServiceRemoveRequest struct {
	VmWatchdogService *VmWatchdogService
	header            map[string]string
	query             map[string]string
	async             *bool
}

func (p *VmWatchdogServiceRemoveRequest) Header(key, value string) *VmWatchdogServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmWatchdogServiceRemoveRequest) Query(key, value string) *VmWatchdogServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmWatchdogServiceRemoveRequest) Async(async bool) *VmWatchdogServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *VmWatchdogServiceRemoveRequest) Send() (*VmWatchdogServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmWatchdogService.connection.URL(), p.VmWatchdogService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmWatchdogService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmWatchdogService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmWatchdogService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmWatchdogService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmWatchdogService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(VmWatchdogServiceRemoveResponse), nil
}

func (p *VmWatchdogServiceRemoveRequest) MustSend() *VmWatchdogServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the watchdog from the virtual machine.
// For example, to remove a watchdog from a virtual machine, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/vms/123/watchdogs/00000000-0000-0000-0000-000000000000
// ----
//
type VmWatchdogServiceRemoveResponse struct {
}

//
// Removes the watchdog from the virtual machine.
// For example, to remove a watchdog from a virtual machine, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/vms/123/watchdogs/00000000-0000-0000-0000-000000000000
// ----
//
func (p *VmWatchdogService) Remove() *VmWatchdogServiceRemoveRequest {
	return &VmWatchdogServiceRemoveRequest{VmWatchdogService: p}
}

//
// Updates the information about the watchdog.
// You can update the information using `action` and `model` elements.
// For example, to update a watchdog, send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/vms/123/watchdogs
// <watchdog>
//   <action>reset</action>
// </watchdog>
// ----
// with response body:
// [source,xml]
// ----
// <watchdog href="/ovirt-engine/api/vms/123/watchdogs/00000000-0000-0000-0000-000000000000" id="00000000-0000-0000-0000-000000000000">
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   <action>reset</action>
//   <model>i6300esb</model>
// </watchdog>
// ----
//
type VmWatchdogServiceUpdateRequest struct {
	VmWatchdogService *VmWatchdogService
	header            map[string]string
	query             map[string]string
	async             *bool
	watchdog          *Watchdog
}

func (p *VmWatchdogServiceUpdateRequest) Header(key, value string) *VmWatchdogServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmWatchdogServiceUpdateRequest) Query(key, value string) *VmWatchdogServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmWatchdogServiceUpdateRequest) Async(async bool) *VmWatchdogServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *VmWatchdogServiceUpdateRequest) Watchdog(watchdog *Watchdog) *VmWatchdogServiceUpdateRequest {
	p.watchdog = watchdog
	return p
}

func (p *VmWatchdogServiceUpdateRequest) Send() (*VmWatchdogServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmWatchdogService.connection.URL(), p.VmWatchdogService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLWatchdogWriteOne(writer, p.watchdog, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmWatchdogService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmWatchdogService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmWatchdogService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmWatchdogService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmWatchdogService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLWatchdogReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmWatchdogServiceUpdateResponse{watchdog: result}, nil
}

func (p *VmWatchdogServiceUpdateRequest) MustSend() *VmWatchdogServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the information about the watchdog.
// You can update the information using `action` and `model` elements.
// For example, to update a watchdog, send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/vms/123/watchdogs
// <watchdog>
//   <action>reset</action>
// </watchdog>
// ----
// with response body:
// [source,xml]
// ----
// <watchdog href="/ovirt-engine/api/vms/123/watchdogs/00000000-0000-0000-0000-000000000000" id="00000000-0000-0000-0000-000000000000">
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   <action>reset</action>
//   <model>i6300esb</model>
// </watchdog>
// ----
//
type VmWatchdogServiceUpdateResponse struct {
	watchdog *Watchdog
}

func (p *VmWatchdogServiceUpdateResponse) Watchdog() (*Watchdog, bool) {
	if p.watchdog != nil {
		return p.watchdog, true
	}
	return nil, false
}

func (p *VmWatchdogServiceUpdateResponse) MustWatchdog() *Watchdog {
	if p.watchdog == nil {
		panic("watchdog in response does not exist")
	}
	return p.watchdog
}

//
// Updates the information about the watchdog.
// You can update the information using `action` and `model` elements.
// For example, to update a watchdog, send a request like this:
// [source]
// ----
// PUT /ovirt-engine/api/vms/123/watchdogs
// <watchdog>
//   <action>reset</action>
// </watchdog>
// ----
// with response body:
// [source,xml]
// ----
// <watchdog href="/ovirt-engine/api/vms/123/watchdogs/00000000-0000-0000-0000-000000000000" id="00000000-0000-0000-0000-000000000000">
//   <vm href="/ovirt-engine/api/vms/123" id="123"/>
//   <action>reset</action>
//   <model>i6300esb</model>
// </watchdog>
// ----
//
func (p *VmWatchdogService) Update() *VmWatchdogServiceUpdateRequest {
	return &VmWatchdogServiceUpdateRequest{VmWatchdogService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmWatchdogService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *VmWatchdogService) String() string {
	return fmt.Sprintf("VmWatchdogService:%s", op.path)
}

//
//
type VmDiskService struct {
	BaseService
}

func NewVmDiskService(connection *Connection, path string) *VmDiskService {
	var result VmDiskService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type VmDiskServiceActivateRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
}

func (p *VmDiskServiceActivateRequest) Header(key, value string) *VmDiskServiceActivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceActivateRequest) Query(key, value string) *VmDiskServiceActivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceActivateRequest) Async(async bool) *VmDiskServiceActivateRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceActivateRequest) Send() (*VmDiskServiceActivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/activate", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(VmDiskServiceActivateResponse), nil
}

func (p *VmDiskServiceActivateRequest) MustSend() *VmDiskServiceActivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VmDiskServiceActivateResponse struct {
}

//
//
func (p *VmDiskService) Activate() *VmDiskServiceActivateRequest {
	return &VmDiskServiceActivateRequest{VmDiskService: p}
}

//
//
type VmDiskServiceDeactivateRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
}

func (p *VmDiskServiceDeactivateRequest) Header(key, value string) *VmDiskServiceDeactivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceDeactivateRequest) Query(key, value string) *VmDiskServiceDeactivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceDeactivateRequest) Async(async bool) *VmDiskServiceDeactivateRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceDeactivateRequest) Send() (*VmDiskServiceDeactivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/deactivate", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(VmDiskServiceDeactivateResponse), nil
}

func (p *VmDiskServiceDeactivateRequest) MustSend() *VmDiskServiceDeactivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VmDiskServiceDeactivateResponse struct {
}

//
//
func (p *VmDiskService) Deactivate() *VmDiskServiceDeactivateRequest {
	return &VmDiskServiceDeactivateRequest{VmDiskService: p}
}

//
//
type VmDiskServiceExportRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
	filter        *bool
}

func (p *VmDiskServiceExportRequest) Header(key, value string) *VmDiskServiceExportRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceExportRequest) Query(key, value string) *VmDiskServiceExportRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceExportRequest) Async(async bool) *VmDiskServiceExportRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceExportRequest) Filter(filter bool) *VmDiskServiceExportRequest {
	p.filter = &filter
	return p
}

func (p *VmDiskServiceExportRequest) Send() (*VmDiskServiceExportResponse, error) {
	rawURL := fmt.Sprintf("%s%s/export", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.filter != nil {
		actionBuilder.Filter(*p.filter)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(VmDiskServiceExportResponse), nil
}

func (p *VmDiskServiceExportRequest) MustSend() *VmDiskServiceExportResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VmDiskServiceExportResponse struct {
}

//
//
func (p *VmDiskService) Export() *VmDiskServiceExportRequest {
	return &VmDiskServiceExportRequest{VmDiskService: p}
}

//
//
type VmDiskServiceGetRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	follow        *string
}

func (p *VmDiskServiceGetRequest) Header(key, value string) *VmDiskServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceGetRequest) Query(key, value string) *VmDiskServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceGetRequest) Follow(follow string) *VmDiskServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *VmDiskServiceGetRequest) Send() (*VmDiskServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmDiskServiceGetResponse{disk: result}, nil
}

func (p *VmDiskServiceGetRequest) MustSend() *VmDiskServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VmDiskServiceGetResponse struct {
	disk *Disk
}

func (p *VmDiskServiceGetResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *VmDiskServiceGetResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
//
func (p *VmDiskService) Get() *VmDiskServiceGetRequest {
	return &VmDiskServiceGetRequest{VmDiskService: p}
}

//
//
type VmDiskServiceMoveRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
	filter        *bool
}

func (p *VmDiskServiceMoveRequest) Header(key, value string) *VmDiskServiceMoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceMoveRequest) Query(key, value string) *VmDiskServiceMoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceMoveRequest) Async(async bool) *VmDiskServiceMoveRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceMoveRequest) Filter(filter bool) *VmDiskServiceMoveRequest {
	p.filter = &filter
	return p
}

func (p *VmDiskServiceMoveRequest) Send() (*VmDiskServiceMoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s/move", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.filter != nil {
		actionBuilder.Filter(*p.filter)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(VmDiskServiceMoveResponse), nil
}

func (p *VmDiskServiceMoveRequest) MustSend() *VmDiskServiceMoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VmDiskServiceMoveResponse struct {
}

//
//
func (p *VmDiskService) Move() *VmDiskServiceMoveRequest {
	return &VmDiskServiceMoveRequest{VmDiskService: p}
}

//
// Reduces the size of the disk image.
// Invokes _reduce_ on the logical volume (i.e. this is only applicable for block storage domains).
// This is applicable for floating disks and disks attached to non-running virtual machines.
// There is no need to specify the size as the optimal size is calculated automatically.
//
type VmDiskServiceReduceRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
}

func (p *VmDiskServiceReduceRequest) Header(key, value string) *VmDiskServiceReduceRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceReduceRequest) Query(key, value string) *VmDiskServiceReduceRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceReduceRequest) Async(async bool) *VmDiskServiceReduceRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceReduceRequest) Send() (*VmDiskServiceReduceResponse, error) {
	rawURL := fmt.Sprintf("%s%s/reduce", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(VmDiskServiceReduceResponse), nil
}

func (p *VmDiskServiceReduceRequest) MustSend() *VmDiskServiceReduceResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Reduces the size of the disk image.
// Invokes _reduce_ on the logical volume (i.e. this is only applicable for block storage domains).
// This is applicable for floating disks and disks attached to non-running virtual machines.
// There is no need to specify the size as the optimal size is calculated automatically.
//
type VmDiskServiceReduceResponse struct {
}

//
// Reduces the size of the disk image.
// Invokes _reduce_ on the logical volume (i.e. this is only applicable for block storage domains).
// This is applicable for floating disks and disks attached to non-running virtual machines.
// There is no need to specify the size as the optimal size is calculated automatically.
//
func (p *VmDiskService) Reduce() *VmDiskServiceReduceRequest {
	return &VmDiskServiceReduceRequest{VmDiskService: p}
}

//
// Detach the disk from the virtual machine.
// NOTE: In version 3 of the API this used to also remove the disk completely from the system, but starting with
// version 4 it doesn't. If you need to remove it completely use the <<services/disk/methods/remove,remove
// method of the top level disk service>>.
//
type VmDiskServiceRemoveRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
}

func (p *VmDiskServiceRemoveRequest) Header(key, value string) *VmDiskServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceRemoveRequest) Query(key, value string) *VmDiskServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceRemoveRequest) Async(async bool) *VmDiskServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceRemoveRequest) Send() (*VmDiskServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(VmDiskServiceRemoveResponse), nil
}

func (p *VmDiskServiceRemoveRequest) MustSend() *VmDiskServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Detach the disk from the virtual machine.
// NOTE: In version 3 of the API this used to also remove the disk completely from the system, but starting with
// version 4 it doesn't. If you need to remove it completely use the <<services/disk/methods/remove,remove
// method of the top level disk service>>.
//
type VmDiskServiceRemoveResponse struct {
}

//
// Detach the disk from the virtual machine.
// NOTE: In version 3 of the API this used to also remove the disk completely from the system, but starting with
// version 4 it doesn't. If you need to remove it completely use the <<services/disk/methods/remove,remove
// method of the top level disk service>>.
//
func (p *VmDiskService) Remove() *VmDiskServiceRemoveRequest {
	return &VmDiskServiceRemoveRequest{VmDiskService: p}
}

//
//
type VmDiskServiceUpdateRequest struct {
	VmDiskService *VmDiskService
	header        map[string]string
	query         map[string]string
	async         *bool
	disk          *Disk
}

func (p *VmDiskServiceUpdateRequest) Header(key, value string) *VmDiskServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *VmDiskServiceUpdateRequest) Query(key, value string) *VmDiskServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *VmDiskServiceUpdateRequest) Async(async bool) *VmDiskServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *VmDiskServiceUpdateRequest) Disk(disk *Disk) *VmDiskServiceUpdateRequest {
	p.disk = disk
	return p
}

func (p *VmDiskServiceUpdateRequest) Send() (*VmDiskServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.VmDiskService.connection.URL(), p.VmDiskService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskWriteOne(writer, p.disk, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.VmDiskService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.VmDiskService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.VmDiskService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.VmDiskService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.VmDiskService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &VmDiskServiceUpdateResponse{disk: result}, nil
}

func (p *VmDiskServiceUpdateRequest) MustSend() *VmDiskServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type VmDiskServiceUpdateResponse struct {
	disk *Disk
}

func (p *VmDiskServiceUpdateResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *VmDiskServiceUpdateResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
//
func (p *VmDiskService) Update() *VmDiskServiceUpdateRequest {
	return &VmDiskServiceUpdateRequest{VmDiskService: p}
}

//
//
func (op *VmDiskService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
//
func (op *VmDiskService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *VmDiskService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *VmDiskService) String() string {
	return fmt.Sprintf("VmDiskService:%s", op.path)
}

//
//
type StorageDomainService struct {
	BaseService
}

func NewStorageDomainService(connection *Connection, path string) *StorageDomainService {
	var result StorageDomainService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the description of the storage domain.
//
type StorageDomainServiceGetRequest struct {
	StorageDomainService *StorageDomainService
	header               map[string]string
	query                map[string]string
	filter               *bool
	follow               *string
}

func (p *StorageDomainServiceGetRequest) Header(key, value string) *StorageDomainServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServiceGetRequest) Query(key, value string) *StorageDomainServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServiceGetRequest) Filter(filter bool) *StorageDomainServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *StorageDomainServiceGetRequest) Follow(follow string) *StorageDomainServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *StorageDomainServiceGetRequest) Send() (*StorageDomainServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainService.connection.URL(), p.StorageDomainService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageDomainReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageDomainServiceGetResponse{storageDomain: result}, nil
}

func (p *StorageDomainServiceGetRequest) MustSend() *StorageDomainServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the description of the storage domain.
//
type StorageDomainServiceGetResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainServiceGetResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainServiceGetResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Retrieves the description of the storage domain.
//
func (p *StorageDomainService) Get() *StorageDomainServiceGetRequest {
	return &StorageDomainServiceGetRequest{StorageDomainService: p}
}

//
// Used for querying if the storage domain is already attached to a data center using
// the is_attached boolean field, which is part of the storage server. IMPORTANT:
// Executing this API will cause the host to disconnect from the storage domain.
//
type StorageDomainServiceIsAttachedRequest struct {
	StorageDomainService *StorageDomainService
	header               map[string]string
	query                map[string]string
	async                *bool
	host                 *Host
}

func (p *StorageDomainServiceIsAttachedRequest) Header(key, value string) *StorageDomainServiceIsAttachedRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServiceIsAttachedRequest) Query(key, value string) *StorageDomainServiceIsAttachedRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServiceIsAttachedRequest) Async(async bool) *StorageDomainServiceIsAttachedRequest {
	p.async = &async
	return p
}

func (p *StorageDomainServiceIsAttachedRequest) Host(host *Host) *StorageDomainServiceIsAttachedRequest {
	p.host = host
	return p
}

func (p *StorageDomainServiceIsAttachedRequest) Send() (*StorageDomainServiceIsAttachedResponse, error) {
	rawURL := fmt.Sprintf("%s%s/isattached", p.StorageDomainService.connection.URL(), p.StorageDomainService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Host(p.host)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustIsAttached()
	return &StorageDomainServiceIsAttachedResponse{isAttached: &result}, nil
}

func (p *StorageDomainServiceIsAttachedRequest) MustSend() *StorageDomainServiceIsAttachedResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Used for querying if the storage domain is already attached to a data center using
// the is_attached boolean field, which is part of the storage server. IMPORTANT:
// Executing this API will cause the host to disconnect from the storage domain.
//
type StorageDomainServiceIsAttachedResponse struct {
	isAttached *bool
}

func (p *StorageDomainServiceIsAttachedResponse) IsAttached() (bool, bool) {
	if p.isAttached != nil {
		return *p.isAttached, true
	}
	var zero bool
	return zero, false
}

func (p *StorageDomainServiceIsAttachedResponse) MustIsAttached() bool {
	if p.isAttached == nil {
		panic("isAttached in response does not exist")
	}
	return *p.isAttached
}

//
// Used for querying if the storage domain is already attached to a data center using
// the is_attached boolean field, which is part of the storage server. IMPORTANT:
// Executing this API will cause the host to disconnect from the storage domain.
//
func (p *StorageDomainService) IsAttached() *StorageDomainServiceIsAttachedRequest {
	return &StorageDomainServiceIsAttachedRequest{StorageDomainService: p}
}

//
// This operation reduces logical units from the storage domain.
// In order to do so the data stored on the provided logical units will be moved to other logical units of the
// storage domain and only then they will be reduced from the storage domain.
// For example, in order to reduce two logical units from a storage domain send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageDomains/123/reduceluns
// ----
// With a request body like this:
// [source,xml]
// ----
//  <action>
//    <logical_units>
//      <logical_unit id="1IET_00010001"/>
//      <logical_unit id="1IET_00010002"/>
//    </logical_units>
//  </action>
// ----
//  Note that this operation is only applicable to block storage domains (i.e., storage domains with the
//  <<types/storage_type, storage type> of iSCSI or FCP).
//
type StorageDomainServiceReduceLunsRequest struct {
	StorageDomainService *StorageDomainService
	header               map[string]string
	query                map[string]string
	logicalUnits         *LogicalUnitSlice
}

func (p *StorageDomainServiceReduceLunsRequest) Header(key, value string) *StorageDomainServiceReduceLunsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServiceReduceLunsRequest) Query(key, value string) *StorageDomainServiceReduceLunsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServiceReduceLunsRequest) LogicalUnits(logicalUnits *LogicalUnitSlice) *StorageDomainServiceReduceLunsRequest {
	p.logicalUnits = logicalUnits
	return p
}

func (p *StorageDomainServiceReduceLunsRequest) LogicalUnitsOfAny(anys ...*LogicalUnit) *StorageDomainServiceReduceLunsRequest {
	if p.logicalUnits == nil {
		p.logicalUnits = new(LogicalUnitSlice)
	}
	p.logicalUnits.slice = append(p.logicalUnits.slice, anys...)
	return p
}

func (p *StorageDomainServiceReduceLunsRequest) Send() (*StorageDomainServiceReduceLunsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/reduceluns", p.StorageDomainService.connection.URL(), p.StorageDomainService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.LogicalUnits(p.logicalUnits)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(StorageDomainServiceReduceLunsResponse), nil
}

func (p *StorageDomainServiceReduceLunsRequest) MustSend() *StorageDomainServiceReduceLunsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation reduces logical units from the storage domain.
// In order to do so the data stored on the provided logical units will be moved to other logical units of the
// storage domain and only then they will be reduced from the storage domain.
// For example, in order to reduce two logical units from a storage domain send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageDomains/123/reduceluns
// ----
// With a request body like this:
// [source,xml]
// ----
//  <action>
//    <logical_units>
//      <logical_unit id="1IET_00010001"/>
//      <logical_unit id="1IET_00010002"/>
//    </logical_units>
//  </action>
// ----
//  Note that this operation is only applicable to block storage domains (i.e., storage domains with the
//  <<types/storage_type, storage type> of iSCSI or FCP).
//
type StorageDomainServiceReduceLunsResponse struct {
}

//
// This operation reduces logical units from the storage domain.
// In order to do so the data stored on the provided logical units will be moved to other logical units of the
// storage domain and only then they will be reduced from the storage domain.
// For example, in order to reduce two logical units from a storage domain send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageDomains/123/reduceluns
// ----
// With a request body like this:
// [source,xml]
// ----
//  <action>
//    <logical_units>
//      <logical_unit id="1IET_00010001"/>
//      <logical_unit id="1IET_00010002"/>
//    </logical_units>
//  </action>
// ----
//  Note that this operation is only applicable to block storage domains (i.e., storage domains with the
//  <<types/storage_type, storage type> of iSCSI or FCP).
//
func (p *StorageDomainService) ReduceLuns() *StorageDomainServiceReduceLunsRequest {
	return &StorageDomainServiceReduceLunsRequest{StorageDomainService: p}
}

//
// This operation refreshes the LUN size.
// After increasing the size of the underlying LUN on the storage server,
// the user can refresh the LUN size.
// This action forces a rescan of the provided LUNs and
// updates the database with the new size, if required.
// For example, in order to refresh the size of two LUNs send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageDomains/262b056b-aede-40f1-9666-b883eff59d40/refreshluns
// ----
// With a request body like this:
// [source,xml]
// ----
//  <action>
//    <logical_units>
//      <logical_unit id="1IET_00010001"/>
//      <logical_unit id="1IET_00010002"/>
//    </logical_units>
//  </action>
// ----
//
type StorageDomainServiceRefreshLunsRequest struct {
	StorageDomainService *StorageDomainService
	header               map[string]string
	query                map[string]string
	async                *bool
	logicalUnits         *LogicalUnitSlice
}

func (p *StorageDomainServiceRefreshLunsRequest) Header(key, value string) *StorageDomainServiceRefreshLunsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServiceRefreshLunsRequest) Query(key, value string) *StorageDomainServiceRefreshLunsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServiceRefreshLunsRequest) Async(async bool) *StorageDomainServiceRefreshLunsRequest {
	p.async = &async
	return p
}

func (p *StorageDomainServiceRefreshLunsRequest) LogicalUnits(logicalUnits *LogicalUnitSlice) *StorageDomainServiceRefreshLunsRequest {
	p.logicalUnits = logicalUnits
	return p
}

func (p *StorageDomainServiceRefreshLunsRequest) LogicalUnitsOfAny(anys ...*LogicalUnit) *StorageDomainServiceRefreshLunsRequest {
	if p.logicalUnits == nil {
		p.logicalUnits = new(LogicalUnitSlice)
	}
	p.logicalUnits.slice = append(p.logicalUnits.slice, anys...)
	return p
}

func (p *StorageDomainServiceRefreshLunsRequest) Send() (*StorageDomainServiceRefreshLunsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/refreshluns", p.StorageDomainService.connection.URL(), p.StorageDomainService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.LogicalUnits(p.logicalUnits)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(StorageDomainServiceRefreshLunsResponse), nil
}

func (p *StorageDomainServiceRefreshLunsRequest) MustSend() *StorageDomainServiceRefreshLunsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation refreshes the LUN size.
// After increasing the size of the underlying LUN on the storage server,
// the user can refresh the LUN size.
// This action forces a rescan of the provided LUNs and
// updates the database with the new size, if required.
// For example, in order to refresh the size of two LUNs send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageDomains/262b056b-aede-40f1-9666-b883eff59d40/refreshluns
// ----
// With a request body like this:
// [source,xml]
// ----
//  <action>
//    <logical_units>
//      <logical_unit id="1IET_00010001"/>
//      <logical_unit id="1IET_00010002"/>
//    </logical_units>
//  </action>
// ----
//
type StorageDomainServiceRefreshLunsResponse struct {
}

//
// This operation refreshes the LUN size.
// After increasing the size of the underlying LUN on the storage server,
// the user can refresh the LUN size.
// This action forces a rescan of the provided LUNs and
// updates the database with the new size, if required.
// For example, in order to refresh the size of two LUNs send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/storageDomains/262b056b-aede-40f1-9666-b883eff59d40/refreshluns
// ----
// With a request body like this:
// [source,xml]
// ----
//  <action>
//    <logical_units>
//      <logical_unit id="1IET_00010001"/>
//      <logical_unit id="1IET_00010002"/>
//    </logical_units>
//  </action>
// ----
//
func (p *StorageDomainService) RefreshLuns() *StorageDomainServiceRefreshLunsRequest {
	return &StorageDomainServiceRefreshLunsRequest{StorageDomainService: p}
}

//
// Removes the storage domain.
// Without any special parameters, the storage domain is detached from the system and removed from the database. The
// storage domain can then be imported to the same or to a different setup, with all the data on it. If the storage is
// not accessible the operation will fail.
// If the `destroy` parameter is `true` then the operation will always succeed, even if the storage is not
// accessible, the failure is just ignored and the storage domain is removed from the database anyway.
// If the `format` parameter is `true` then the actual storage is formatted, and the metadata is removed from the
// LUN or directory, so it can no longer be imported to the same or to a different setup.
//
type StorageDomainServiceRemoveRequest struct {
	StorageDomainService *StorageDomainService
	header               map[string]string
	query                map[string]string
	async                *bool
	destroy              *bool
	format               *bool
	host                 *string
}

func (p *StorageDomainServiceRemoveRequest) Header(key, value string) *StorageDomainServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServiceRemoveRequest) Query(key, value string) *StorageDomainServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServiceRemoveRequest) Async(async bool) *StorageDomainServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *StorageDomainServiceRemoveRequest) Destroy(destroy bool) *StorageDomainServiceRemoveRequest {
	p.destroy = &destroy
	return p
}

func (p *StorageDomainServiceRemoveRequest) Format(format bool) *StorageDomainServiceRemoveRequest {
	p.format = &format
	return p
}

func (p *StorageDomainServiceRemoveRequest) Host(host string) *StorageDomainServiceRemoveRequest {
	p.host = &host
	return p
}

func (p *StorageDomainServiceRemoveRequest) Send() (*StorageDomainServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainService.connection.URL(), p.StorageDomainService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.destroy != nil {
		values["destroy"] = []string{fmt.Sprintf("%v", *p.destroy)}
	}

	if p.format != nil {
		values["format"] = []string{fmt.Sprintf("%v", *p.format)}
	}

	if p.host != nil {
		values["host"] = []string{fmt.Sprintf("%v", *p.host)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(StorageDomainServiceRemoveResponse), nil
}

func (p *StorageDomainServiceRemoveRequest) MustSend() *StorageDomainServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the storage domain.
// Without any special parameters, the storage domain is detached from the system and removed from the database. The
// storage domain can then be imported to the same or to a different setup, with all the data on it. If the storage is
// not accessible the operation will fail.
// If the `destroy` parameter is `true` then the operation will always succeed, even if the storage is not
// accessible, the failure is just ignored and the storage domain is removed from the database anyway.
// If the `format` parameter is `true` then the actual storage is formatted, and the metadata is removed from the
// LUN or directory, so it can no longer be imported to the same or to a different setup.
//
type StorageDomainServiceRemoveResponse struct {
}

//
// Removes the storage domain.
// Without any special parameters, the storage domain is detached from the system and removed from the database. The
// storage domain can then be imported to the same or to a different setup, with all the data on it. If the storage is
// not accessible the operation will fail.
// If the `destroy` parameter is `true` then the operation will always succeed, even if the storage is not
// accessible, the failure is just ignored and the storage domain is removed from the database anyway.
// If the `format` parameter is `true` then the actual storage is formatted, and the metadata is removed from the
// LUN or directory, so it can no longer be imported to the same or to a different setup.
//
func (p *StorageDomainService) Remove() *StorageDomainServiceRemoveRequest {
	return &StorageDomainServiceRemoveRequest{StorageDomainService: p}
}

//
// Updates a storage domain.
// Not all of the <<types/storage_domain,StorageDomain>>'s attributes are updatable after creation. Those that can be
// updated are: `name`, `description`, `comment`, `warning_low_space_indicator`, `critical_space_action_blocker` and
// `wipe_after_delete.` (Note that changing the `wipe_after_delete` attribute will not change the wipe after delete
// property of disks that already exist).
// To update the `name` and `wipe_after_delete` attributes of a storage domain with an identifier `123`, send a
// request as follows:
// [source]
// ----
// PUT /ovirt-engine/api/storageDomains/123
// ----
// With a request body as follows:
// [source,xml]
// ----
// <storage_domain>
//   <name>data2</name>
//   <wipe_after_delete>true</wipe_after_delete>
// </storage_domain>
// ----
//
type StorageDomainServiceUpdateRequest struct {
	StorageDomainService *StorageDomainService
	header               map[string]string
	query                map[string]string
	async                *bool
	storageDomain        *StorageDomain
}

func (p *StorageDomainServiceUpdateRequest) Header(key, value string) *StorageDomainServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServiceUpdateRequest) Query(key, value string) *StorageDomainServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServiceUpdateRequest) Async(async bool) *StorageDomainServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *StorageDomainServiceUpdateRequest) StorageDomain(storageDomain *StorageDomain) *StorageDomainServiceUpdateRequest {
	p.storageDomain = storageDomain
	return p
}

func (p *StorageDomainServiceUpdateRequest) Send() (*StorageDomainServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainService.connection.URL(), p.StorageDomainService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLStorageDomainWriteOne(writer, p.storageDomain, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageDomainReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageDomainServiceUpdateResponse{storageDomain: result}, nil
}

func (p *StorageDomainServiceUpdateRequest) MustSend() *StorageDomainServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates a storage domain.
// Not all of the <<types/storage_domain,StorageDomain>>'s attributes are updatable after creation. Those that can be
// updated are: `name`, `description`, `comment`, `warning_low_space_indicator`, `critical_space_action_blocker` and
// `wipe_after_delete.` (Note that changing the `wipe_after_delete` attribute will not change the wipe after delete
// property of disks that already exist).
// To update the `name` and `wipe_after_delete` attributes of a storage domain with an identifier `123`, send a
// request as follows:
// [source]
// ----
// PUT /ovirt-engine/api/storageDomains/123
// ----
// With a request body as follows:
// [source,xml]
// ----
// <storage_domain>
//   <name>data2</name>
//   <wipe_after_delete>true</wipe_after_delete>
// </storage_domain>
// ----
//
type StorageDomainServiceUpdateResponse struct {
	storageDomain *StorageDomain
}

func (p *StorageDomainServiceUpdateResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *StorageDomainServiceUpdateResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
// Updates a storage domain.
// Not all of the <<types/storage_domain,StorageDomain>>'s attributes are updatable after creation. Those that can be
// updated are: `name`, `description`, `comment`, `warning_low_space_indicator`, `critical_space_action_blocker` and
// `wipe_after_delete.` (Note that changing the `wipe_after_delete` attribute will not change the wipe after delete
// property of disks that already exist).
// To update the `name` and `wipe_after_delete` attributes of a storage domain with an identifier `123`, send a
// request as follows:
// [source]
// ----
// PUT /ovirt-engine/api/storageDomains/123
// ----
// With a request body as follows:
// [source,xml]
// ----
// <storage_domain>
//   <name>data2</name>
//   <wipe_after_delete>true</wipe_after_delete>
// </storage_domain>
// ----
//
func (p *StorageDomainService) Update() *StorageDomainServiceUpdateRequest {
	return &StorageDomainServiceUpdateRequest{StorageDomainService: p}
}

//
// This operation forces the update of the `OVF_STORE`
// of this storage domain.
// The `OVF_STORE` is a disk image that contains the metadata
// of virtual machines and disks that reside in the
// storage domain. This metadata is used in case the
// domain is imported or exported to or from a different
// data center or a different installation.
// By default the `OVF_STORE` is updated periodically
// (set by default to 60 minutes) but users might want to force an
// update after an important change, or when the they believe the
// `OVF_STORE` is corrupt.
// When initiated by the user, `OVF_STORE` update will be performed whether
// an update is needed or not.
//
type StorageDomainServiceUpdateOvfStoreRequest struct {
	StorageDomainService *StorageDomainService
	header               map[string]string
	query                map[string]string
	async                *bool
}

func (p *StorageDomainServiceUpdateOvfStoreRequest) Header(key, value string) *StorageDomainServiceUpdateOvfStoreRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServiceUpdateOvfStoreRequest) Query(key, value string) *StorageDomainServiceUpdateOvfStoreRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServiceUpdateOvfStoreRequest) Async(async bool) *StorageDomainServiceUpdateOvfStoreRequest {
	p.async = &async
	return p
}

func (p *StorageDomainServiceUpdateOvfStoreRequest) Send() (*StorageDomainServiceUpdateOvfStoreResponse, error) {
	rawURL := fmt.Sprintf("%s%s/updateovfstore", p.StorageDomainService.connection.URL(), p.StorageDomainService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(StorageDomainServiceUpdateOvfStoreResponse), nil
}

func (p *StorageDomainServiceUpdateOvfStoreRequest) MustSend() *StorageDomainServiceUpdateOvfStoreResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation forces the update of the `OVF_STORE`
// of this storage domain.
// The `OVF_STORE` is a disk image that contains the metadata
// of virtual machines and disks that reside in the
// storage domain. This metadata is used in case the
// domain is imported or exported to or from a different
// data center or a different installation.
// By default the `OVF_STORE` is updated periodically
// (set by default to 60 minutes) but users might want to force an
// update after an important change, or when the they believe the
// `OVF_STORE` is corrupt.
// When initiated by the user, `OVF_STORE` update will be performed whether
// an update is needed or not.
//
type StorageDomainServiceUpdateOvfStoreResponse struct {
}

//
// This operation forces the update of the `OVF_STORE`
// of this storage domain.
// The `OVF_STORE` is a disk image that contains the metadata
// of virtual machines and disks that reside in the
// storage domain. This metadata is used in case the
// domain is imported or exported to or from a different
// data center or a different installation.
// By default the `OVF_STORE` is updated periodically
// (set by default to 60 minutes) but users might want to force an
// update after an important change, or when the they believe the
// `OVF_STORE` is corrupt.
// When initiated by the user, `OVF_STORE` update will be performed whether
// an update is needed or not.
//
func (p *StorageDomainService) UpdateOvfStore() *StorageDomainServiceUpdateOvfStoreRequest {
	return &StorageDomainServiceUpdateOvfStoreRequest{StorageDomainService: p}
}

//
//
func (op *StorageDomainService) DiskProfilesService() *AssignedDiskProfilesService {
	return NewAssignedDiskProfilesService(op.connection, fmt.Sprintf("%s/diskprofiles", op.path))
}

//
//
func (op *StorageDomainService) DiskSnapshotsService() *DiskSnapshotsService {
	return NewDiskSnapshotsService(op.connection, fmt.Sprintf("%s/disksnapshots", op.path))
}

//
// Reference to the service that manages the disks available in the storage domain.
//
func (op *StorageDomainService) DisksService() *StorageDomainDisksService {
	return NewStorageDomainDisksService(op.connection, fmt.Sprintf("%s/disks", op.path))
}

//
// Returns a reference to the service that manages the files available in the storage domain.
//
func (op *StorageDomainService) FilesService() *FilesService {
	return NewFilesService(op.connection, fmt.Sprintf("%s/files", op.path))
}

//
//
func (op *StorageDomainService) ImagesService() *ImagesService {
	return NewImagesService(op.connection, fmt.Sprintf("%s/images", op.path))
}

//
//
func (op *StorageDomainService) PermissionsService() *AssignedPermissionsService {
	return NewAssignedPermissionsService(op.connection, fmt.Sprintf("%s/permissions", op.path))
}

//
// Returns a reference to the service that manages the storage connections.
//
func (op *StorageDomainService) StorageConnectionsService() *StorageDomainServerConnectionsService {
	return NewStorageDomainServerConnectionsService(op.connection, fmt.Sprintf("%s/storageconnections", op.path))
}

//
//
func (op *StorageDomainService) TemplatesService() *StorageDomainTemplatesService {
	return NewStorageDomainTemplatesService(op.connection, fmt.Sprintf("%s/templates", op.path))
}

//
//
func (op *StorageDomainService) VmsService() *StorageDomainVmsService {
	return NewStorageDomainVmsService(op.connection, fmt.Sprintf("%s/vms", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageDomainService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "diskprofiles" {
		return op.DiskProfilesService(), nil
	}
	if strings.HasPrefix(path, "diskprofiles/") {
		return op.DiskProfilesService().Service(path[13:])
	}
	if path == "disksnapshots" {
		return op.DiskSnapshotsService(), nil
	}
	if strings.HasPrefix(path, "disksnapshots/") {
		return op.DiskSnapshotsService().Service(path[14:])
	}
	if path == "disks" {
		return op.DisksService(), nil
	}
	if strings.HasPrefix(path, "disks/") {
		return op.DisksService().Service(path[6:])
	}
	if path == "files" {
		return op.FilesService(), nil
	}
	if strings.HasPrefix(path, "files/") {
		return op.FilesService().Service(path[6:])
	}
	if path == "images" {
		return op.ImagesService(), nil
	}
	if strings.HasPrefix(path, "images/") {
		return op.ImagesService().Service(path[7:])
	}
	if path == "permissions" {
		return op.PermissionsService(), nil
	}
	if strings.HasPrefix(path, "permissions/") {
		return op.PermissionsService().Service(path[12:])
	}
	if path == "storageconnections" {
		return op.StorageConnectionsService(), nil
	}
	if strings.HasPrefix(path, "storageconnections/") {
		return op.StorageConnectionsService().Service(path[19:])
	}
	if path == "templates" {
		return op.TemplatesService(), nil
	}
	if strings.HasPrefix(path, "templates/") {
		return op.TemplatesService().Service(path[10:])
	}
	if path == "vms" {
		return op.VmsService(), nil
	}
	if strings.HasPrefix(path, "vms/") {
		return op.VmsService().Service(path[4:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *StorageDomainService) String() string {
	return fmt.Sprintf("StorageDomainService:%s", op.path)
}

//
//
type InstanceTypeWatchdogsService struct {
	BaseService
}

func NewInstanceTypeWatchdogsService(connection *Connection, path string) *InstanceTypeWatchdogsService {
	var result InstanceTypeWatchdogsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add new watchdog to the instance type.
//
type InstanceTypeWatchdogsServiceAddRequest struct {
	InstanceTypeWatchdogsService *InstanceTypeWatchdogsService
	header                       map[string]string
	query                        map[string]string
	watchdog                     *Watchdog
}

func (p *InstanceTypeWatchdogsServiceAddRequest) Header(key, value string) *InstanceTypeWatchdogsServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *InstanceTypeWatchdogsServiceAddRequest) Query(key, value string) *InstanceTypeWatchdogsServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *InstanceTypeWatchdogsServiceAddRequest) Watchdog(watchdog *Watchdog) *InstanceTypeWatchdogsServiceAddRequest {
	p.watchdog = watchdog
	return p
}

func (p *InstanceTypeWatchdogsServiceAddRequest) Send() (*InstanceTypeWatchdogsServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.InstanceTypeWatchdogsService.connection.URL(), p.InstanceTypeWatchdogsService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLWatchdogWriteOne(writer, p.watchdog, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.InstanceTypeWatchdogsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.InstanceTypeWatchdogsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.InstanceTypeWatchdogsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.InstanceTypeWatchdogsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.InstanceTypeWatchdogsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLWatchdogReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &InstanceTypeWatchdogsServiceAddResponse{watchdog: result}, nil
}

func (p *InstanceTypeWatchdogsServiceAddRequest) MustSend() *InstanceTypeWatchdogsServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add new watchdog to the instance type.
//
type InstanceTypeWatchdogsServiceAddResponse struct {
	watchdog *Watchdog
}

func (p *InstanceTypeWatchdogsServiceAddResponse) Watchdog() (*Watchdog, bool) {
	if p.watchdog != nil {
		return p.watchdog, true
	}
	return nil, false
}

func (p *InstanceTypeWatchdogsServiceAddResponse) MustWatchdog() *Watchdog {
	if p.watchdog == nil {
		panic("watchdog in response does not exist")
	}
	return p.watchdog
}

//
// Add new watchdog to the instance type.
//
func (p *InstanceTypeWatchdogsService) Add() *InstanceTypeWatchdogsServiceAddRequest {
	return &InstanceTypeWatchdogsServiceAddRequest{InstanceTypeWatchdogsService: p}
}

//
// Lists all the configured watchdogs of the instance type.
// The order of the returned list of watchdogs isn't guaranteed.
//
type InstanceTypeWatchdogsServiceListRequest struct {
	InstanceTypeWatchdogsService *InstanceTypeWatchdogsService
	header                       map[string]string
	query                        map[string]string
	follow                       *string
	max                          *int64
	search                       *string
}

func (p *InstanceTypeWatchdogsServiceListRequest) Header(key, value string) *InstanceTypeWatchdogsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *InstanceTypeWatchdogsServiceListRequest) Query(key, value string) *InstanceTypeWatchdogsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *InstanceTypeWatchdogsServiceListRequest) Follow(follow string) *InstanceTypeWatchdogsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *InstanceTypeWatchdogsServiceListRequest) Max(max int64) *InstanceTypeWatchdogsServiceListRequest {
	p.max = &max
	return p
}

func (p *InstanceTypeWatchdogsServiceListRequest) Search(search string) *InstanceTypeWatchdogsServiceListRequest {
	p.search = &search
	return p
}

func (p *InstanceTypeWatchdogsServiceListRequest) Send() (*InstanceTypeWatchdogsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.InstanceTypeWatchdogsService.connection.URL(), p.InstanceTypeWatchdogsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.InstanceTypeWatchdogsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.InstanceTypeWatchdogsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.InstanceTypeWatchdogsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.InstanceTypeWatchdogsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.InstanceTypeWatchdogsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLWatchdogReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &InstanceTypeWatchdogsServiceListResponse{watchdogs: result}, nil
}

func (p *InstanceTypeWatchdogsServiceListRequest) MustSend() *InstanceTypeWatchdogsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists all the configured watchdogs of the instance type.
// The order of the returned list of watchdogs isn't guaranteed.
//
type InstanceTypeWatchdogsServiceListResponse struct {
	watchdogs *WatchdogSlice
}

func (p *InstanceTypeWatchdogsServiceListResponse) Watchdogs() (*WatchdogSlice, bool) {
	if p.watchdogs != nil {
		return p.watchdogs, true
	}
	return nil, false
}

func (p *InstanceTypeWatchdogsServiceListResponse) MustWatchdogs() *WatchdogSlice {
	if p.watchdogs == nil {
		panic("watchdogs in response does not exist")
	}
	return p.watchdogs
}

//
// Lists all the configured watchdogs of the instance type.
// The order of the returned list of watchdogs isn't guaranteed.
//
func (p *InstanceTypeWatchdogsService) List() *InstanceTypeWatchdogsServiceListRequest {
	return &InstanceTypeWatchdogsServiceListRequest{InstanceTypeWatchdogsService: p}
}

//
//
func (op *InstanceTypeWatchdogsService) WatchdogService(id string) *InstanceTypeWatchdogService {
	return NewInstanceTypeWatchdogService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *InstanceTypeWatchdogsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.WatchdogService(path), nil
	}
	return op.WatchdogService(path[:index]).Service(path[index+1:])
}

func (op *InstanceTypeWatchdogsService) String() string {
	return fmt.Sprintf("InstanceTypeWatchdogsService:%s", op.path)
}

//
// A service to fetch information elements received by Link Layer Discovery Protocol (LLDP).
//
type LinkLayerDiscoveryProtocolService struct {
	BaseService
}

func NewLinkLayerDiscoveryProtocolService(connection *Connection, path string) *LinkLayerDiscoveryProtocolService {
	var result LinkLayerDiscoveryProtocolService
	result.connection = connection
	result.path = path
	return &result
}

//
// Fetches information elements received by LLDP.
//
type LinkLayerDiscoveryProtocolServiceListRequest struct {
	LinkLayerDiscoveryProtocolService *LinkLayerDiscoveryProtocolService
	header                            map[string]string
	query                             map[string]string
	follow                            *string
}

func (p *LinkLayerDiscoveryProtocolServiceListRequest) Header(key, value string) *LinkLayerDiscoveryProtocolServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *LinkLayerDiscoveryProtocolServiceListRequest) Query(key, value string) *LinkLayerDiscoveryProtocolServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *LinkLayerDiscoveryProtocolServiceListRequest) Follow(follow string) *LinkLayerDiscoveryProtocolServiceListRequest {
	p.follow = &follow
	return p
}

func (p *LinkLayerDiscoveryProtocolServiceListRequest) Send() (*LinkLayerDiscoveryProtocolServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.LinkLayerDiscoveryProtocolService.connection.URL(), p.LinkLayerDiscoveryProtocolService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.LinkLayerDiscoveryProtocolService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.LinkLayerDiscoveryProtocolService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.LinkLayerDiscoveryProtocolService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.LinkLayerDiscoveryProtocolService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.LinkLayerDiscoveryProtocolService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLLinkLayerDiscoveryProtocolElementReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &LinkLayerDiscoveryProtocolServiceListResponse{elements: result}, nil
}

func (p *LinkLayerDiscoveryProtocolServiceListRequest) MustSend() *LinkLayerDiscoveryProtocolServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Fetches information elements received by LLDP.
//
type LinkLayerDiscoveryProtocolServiceListResponse struct {
	elements *LinkLayerDiscoveryProtocolElementSlice
}

func (p *LinkLayerDiscoveryProtocolServiceListResponse) Elements() (*LinkLayerDiscoveryProtocolElementSlice, bool) {
	if p.elements != nil {
		return p.elements, true
	}
	return nil, false
}

func (p *LinkLayerDiscoveryProtocolServiceListResponse) MustElements() *LinkLayerDiscoveryProtocolElementSlice {
	if p.elements == nil {
		panic("elements in response does not exist")
	}
	return p.elements
}

//
// Fetches information elements received by LLDP.
//
func (p *LinkLayerDiscoveryProtocolService) List() *LinkLayerDiscoveryProtocolServiceListRequest {
	return &LinkLayerDiscoveryProtocolServiceListRequest{LinkLayerDiscoveryProtocolService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *LinkLayerDiscoveryProtocolService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *LinkLayerDiscoveryProtocolService) String() string {
	return fmt.Sprintf("LinkLayerDiscoveryProtocolService:%s", op.path)
}

//
//
type ExternalHostService struct {
	BaseService
}

func NewExternalHostService(connection *Connection, path string) *ExternalHostService {
	var result ExternalHostService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type ExternalHostServiceGetRequest struct {
	ExternalHostService *ExternalHostService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *ExternalHostServiceGetRequest) Header(key, value string) *ExternalHostServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostServiceGetRequest) Query(key, value string) *ExternalHostServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostServiceGetRequest) Follow(follow string) *ExternalHostServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalHostServiceGetRequest) Send() (*ExternalHostServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostService.connection.URL(), p.ExternalHostService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalHostServiceGetResponse{host: result}, nil
}

func (p *ExternalHostServiceGetRequest) MustSend() *ExternalHostServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type ExternalHostServiceGetResponse struct {
	host *ExternalHost
}

func (p *ExternalHostServiceGetResponse) Host() (*ExternalHost, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

func (p *ExternalHostServiceGetResponse) MustHost() *ExternalHost {
	if p.host == nil {
		panic("host in response does not exist")
	}
	return p.host
}

//
//
func (p *ExternalHostService) Get() *ExternalHostServiceGetRequest {
	return &ExternalHostServiceGetRequest{ExternalHostService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalHostService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalHostService) String() string {
	return fmt.Sprintf("ExternalHostService:%s", op.path)
}

//
// A service to manage Katello errata.
// The information is retrieved from Katello.
//
type KatelloErrataService struct {
	BaseService
}

func NewKatelloErrataService(connection *Connection, path string) *KatelloErrataService {
	var result KatelloErrataService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the representation of the Katello errata.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_errata>
//   <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//     <name>RHBA-2013:XYZ</name>
//     <description>The description of the erratum</description>
//     <title>some bug fix update</title>
//     <type>bugfix</type>
//     <issued>2013-11-20T02:00:00.000+02:00</issued>
//     <solution>Few guidelines regarding the solution</solution>
//     <summary>Updated packages that fix one bug are now available for XYZ</summary>
//     <packages>
//       <package>
//         <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//       </package>
//       ...
//     </packages>
//   </katello_erratum>
//   ...
// </katello_errata>
// ----
// The order of the returned list of erratum isn't guaranteed.
//
type KatelloErrataServiceListRequest struct {
	KatelloErrataService *KatelloErrataService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *KatelloErrataServiceListRequest) Header(key, value string) *KatelloErrataServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *KatelloErrataServiceListRequest) Query(key, value string) *KatelloErrataServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *KatelloErrataServiceListRequest) Follow(follow string) *KatelloErrataServiceListRequest {
	p.follow = &follow
	return p
}

func (p *KatelloErrataServiceListRequest) Max(max int64) *KatelloErrataServiceListRequest {
	p.max = &max
	return p
}

func (p *KatelloErrataServiceListRequest) Send() (*KatelloErrataServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.KatelloErrataService.connection.URL(), p.KatelloErrataService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.KatelloErrataService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.KatelloErrataService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.KatelloErrataService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.KatelloErrataService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.KatelloErrataService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLKatelloErratumReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &KatelloErrataServiceListResponse{errata: result}, nil
}

func (p *KatelloErrataServiceListRequest) MustSend() *KatelloErrataServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the representation of the Katello errata.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_errata>
//   <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//     <name>RHBA-2013:XYZ</name>
//     <description>The description of the erratum</description>
//     <title>some bug fix update</title>
//     <type>bugfix</type>
//     <issued>2013-11-20T02:00:00.000+02:00</issued>
//     <solution>Few guidelines regarding the solution</solution>
//     <summary>Updated packages that fix one bug are now available for XYZ</summary>
//     <packages>
//       <package>
//         <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//       </package>
//       ...
//     </packages>
//   </katello_erratum>
//   ...
// </katello_errata>
// ----
// The order of the returned list of erratum isn't guaranteed.
//
type KatelloErrataServiceListResponse struct {
	errata *KatelloErratumSlice
}

func (p *KatelloErrataServiceListResponse) Errata() (*KatelloErratumSlice, bool) {
	if p.errata != nil {
		return p.errata, true
	}
	return nil, false
}

func (p *KatelloErrataServiceListResponse) MustErrata() *KatelloErratumSlice {
	if p.errata == nil {
		panic("errata in response does not exist")
	}
	return p.errata
}

//
// Retrieves the representation of the Katello errata.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_errata>
//   <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//     <name>RHBA-2013:XYZ</name>
//     <description>The description of the erratum</description>
//     <title>some bug fix update</title>
//     <type>bugfix</type>
//     <issued>2013-11-20T02:00:00.000+02:00</issued>
//     <solution>Few guidelines regarding the solution</solution>
//     <summary>Updated packages that fix one bug are now available for XYZ</summary>
//     <packages>
//       <package>
//         <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//       </package>
//       ...
//     </packages>
//   </katello_erratum>
//   ...
// </katello_errata>
// ----
// The order of the returned list of erratum isn't guaranteed.
//
func (p *KatelloErrataService) List() *KatelloErrataServiceListRequest {
	return &KatelloErrataServiceListRequest{KatelloErrataService: p}
}

//
// Reference to the Katello erratum service.
// Use this service to view the erratum by its id.
//
func (op *KatelloErrataService) KatelloErratumService(id string) *KatelloErratumService {
	return NewKatelloErratumService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *KatelloErrataService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.KatelloErratumService(path), nil
	}
	return op.KatelloErratumService(path[:index]).Service(path[index+1:])
}

func (op *KatelloErrataService) String() string {
	return fmt.Sprintf("KatelloErrataService:%s", op.path)
}

//
// This service manages external discovered hosts.
//
type ExternalDiscoveredHostsService struct {
	BaseService
}

func NewExternalDiscoveredHostsService(connection *Connection, path string) *ExternalDiscoveredHostsService {
	var result ExternalDiscoveredHostsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get list of discovered hosts' information.
// Discovered hosts are fetched from third-party providers such as Foreman.
// To list all discovered hosts for provider `123` send the following:
// [source]
// ----
// GET /ovirt-engine/api/externalhostproviders/123/discoveredhost
// ----
// [source,xml]
// ----
// <external_discovered_hosts>
//  <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/456" id="456">
//   <name>mac001a4ad04031</name>
//   <ip>10.34.67.42</ip>
//   <last_report>2017-04-24 11:05:41 UTC</last_report>
//   <mac>00:1a:4a:d0:40:31</mac>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//  </external_discovered_host>
//  <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/789" id="789">
//   <name>mac001a4ad04040</name>
//   <ip>10.34.67.43</ip>
//   <last_report>2017-04-24 11:05:41 UTC</last_report>
//   <mac>00:1a:4a:d0:40:40</mac>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//  </external_discovered_host>
//  ...
// </external_discovered_hosts>
// ----
// The order of the returned list of hosts isn't guaranteed.
//
type ExternalDiscoveredHostsServiceListRequest struct {
	ExternalDiscoveredHostsService *ExternalDiscoveredHostsService
	header                         map[string]string
	query                          map[string]string
	follow                         *string
	max                            *int64
}

func (p *ExternalDiscoveredHostsServiceListRequest) Header(key, value string) *ExternalDiscoveredHostsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalDiscoveredHostsServiceListRequest) Query(key, value string) *ExternalDiscoveredHostsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalDiscoveredHostsServiceListRequest) Follow(follow string) *ExternalDiscoveredHostsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ExternalDiscoveredHostsServiceListRequest) Max(max int64) *ExternalDiscoveredHostsServiceListRequest {
	p.max = &max
	return p
}

func (p *ExternalDiscoveredHostsServiceListRequest) Send() (*ExternalDiscoveredHostsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalDiscoveredHostsService.connection.URL(), p.ExternalDiscoveredHostsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalDiscoveredHostsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalDiscoveredHostsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalDiscoveredHostsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalDiscoveredHostsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalDiscoveredHostsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalDiscoveredHostReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ExternalDiscoveredHostsServiceListResponse{hosts: result}, nil
}

func (p *ExternalDiscoveredHostsServiceListRequest) MustSend() *ExternalDiscoveredHostsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get list of discovered hosts' information.
// Discovered hosts are fetched from third-party providers such as Foreman.
// To list all discovered hosts for provider `123` send the following:
// [source]
// ----
// GET /ovirt-engine/api/externalhostproviders/123/discoveredhost
// ----
// [source,xml]
// ----
// <external_discovered_hosts>
//  <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/456" id="456">
//   <name>mac001a4ad04031</name>
//   <ip>10.34.67.42</ip>
//   <last_report>2017-04-24 11:05:41 UTC</last_report>
//   <mac>00:1a:4a:d0:40:31</mac>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//  </external_discovered_host>
//  <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/789" id="789">
//   <name>mac001a4ad04040</name>
//   <ip>10.34.67.43</ip>
//   <last_report>2017-04-24 11:05:41 UTC</last_report>
//   <mac>00:1a:4a:d0:40:40</mac>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//  </external_discovered_host>
//  ...
// </external_discovered_hosts>
// ----
// The order of the returned list of hosts isn't guaranteed.
//
type ExternalDiscoveredHostsServiceListResponse struct {
	hosts *ExternalDiscoveredHostSlice
}

func (p *ExternalDiscoveredHostsServiceListResponse) Hosts() (*ExternalDiscoveredHostSlice, bool) {
	if p.hosts != nil {
		return p.hosts, true
	}
	return nil, false
}

func (p *ExternalDiscoveredHostsServiceListResponse) MustHosts() *ExternalDiscoveredHostSlice {
	if p.hosts == nil {
		panic("hosts in response does not exist")
	}
	return p.hosts
}

//
// Get list of discovered hosts' information.
// Discovered hosts are fetched from third-party providers such as Foreman.
// To list all discovered hosts for provider `123` send the following:
// [source]
// ----
// GET /ovirt-engine/api/externalhostproviders/123/discoveredhost
// ----
// [source,xml]
// ----
// <external_discovered_hosts>
//  <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/456" id="456">
//   <name>mac001a4ad04031</name>
//   <ip>10.34.67.42</ip>
//   <last_report>2017-04-24 11:05:41 UTC</last_report>
//   <mac>00:1a:4a:d0:40:31</mac>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//  </external_discovered_host>
//  <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/789" id="789">
//   <name>mac001a4ad04040</name>
//   <ip>10.34.67.43</ip>
//   <last_report>2017-04-24 11:05:41 UTC</last_report>
//   <mac>00:1a:4a:d0:40:40</mac>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//  </external_discovered_host>
//  ...
// </external_discovered_hosts>
// ----
// The order of the returned list of hosts isn't guaranteed.
//
func (p *ExternalDiscoveredHostsService) List() *ExternalDiscoveredHostsServiceListRequest {
	return &ExternalDiscoveredHostsServiceListRequest{ExternalDiscoveredHostsService: p}
}

//
//
func (op *ExternalDiscoveredHostsService) HostService(id string) *ExternalDiscoveredHostService {
	return NewExternalDiscoveredHostService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalDiscoveredHostsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.HostService(path), nil
	}
	return op.HostService(path[:index]).Service(path[index+1:])
}

func (op *ExternalDiscoveredHostsService) String() string {
	return fmt.Sprintf("ExternalDiscoveredHostsService:%s", op.path)
}

//
// Represents an external host provider, such as Foreman or Satellite.
// See https://www.theforeman.org/ for more details on Foreman.
// See https://access.redhat.com/products/red-hat-satellite for more details on Red Hat Satellite.
//
type ExternalHostProviderService struct {
	BaseService
}

func NewExternalHostProviderService(connection *Connection, path string) *ExternalHostProviderService {
	var result ExternalHostProviderService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get external host provider information
// Host provider, Foreman or Satellite, can be set as an external provider in ovirt. To see details about specific
// host providers attached to ovirt use this API.
// For example, to get the details of host provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123
// ....
// The response will be like this:
// [source,xml]
// ----
// <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123">
//   <name>mysatellite</name>
//   <requires_authentication>true</requires_authentication>
//   <url>https://mysatellite.example.com</url>
//   <username>admin</username>
// </external_host_provider>
// ----
//
type ExternalHostProviderServiceGetRequest struct {
	ExternalHostProviderService *ExternalHostProviderService
	header                      map[string]string
	query                       map[string]string
	follow                      *string
}

func (p *ExternalHostProviderServiceGetRequest) Header(key, value string) *ExternalHostProviderServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProviderServiceGetRequest) Query(key, value string) *ExternalHostProviderServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProviderServiceGetRequest) Follow(follow string) *ExternalHostProviderServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalHostProviderServiceGetRequest) Send() (*ExternalHostProviderServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostProviderService.connection.URL(), p.ExternalHostProviderService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalHostProviderServiceGetResponse{provider: result}, nil
}

func (p *ExternalHostProviderServiceGetRequest) MustSend() *ExternalHostProviderServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get external host provider information
// Host provider, Foreman or Satellite, can be set as an external provider in ovirt. To see details about specific
// host providers attached to ovirt use this API.
// For example, to get the details of host provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123
// ....
// The response will be like this:
// [source,xml]
// ----
// <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123">
//   <name>mysatellite</name>
//   <requires_authentication>true</requires_authentication>
//   <url>https://mysatellite.example.com</url>
//   <username>admin</username>
// </external_host_provider>
// ----
//
type ExternalHostProviderServiceGetResponse struct {
	provider *ExternalHostProvider
}

func (p *ExternalHostProviderServiceGetResponse) Provider() (*ExternalHostProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *ExternalHostProviderServiceGetResponse) MustProvider() *ExternalHostProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Get external host provider information
// Host provider, Foreman or Satellite, can be set as an external provider in ovirt. To see details about specific
// host providers attached to ovirt use this API.
// For example, to get the details of host provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123
// ....
// The response will be like this:
// [source,xml]
// ----
// <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123">
//   <name>mysatellite</name>
//   <requires_authentication>true</requires_authentication>
//   <url>https://mysatellite.example.com</url>
//   <username>admin</username>
// </external_host_provider>
// ----
//
func (p *ExternalHostProviderService) Get() *ExternalHostProviderServiceGetRequest {
	return &ExternalHostProviderServiceGetRequest{ExternalHostProviderService: p}
}

//
// Import the SSL certificates of the external host provider.
//
type ExternalHostProviderServiceImportCertificatesRequest struct {
	ExternalHostProviderService *ExternalHostProviderService
	header                      map[string]string
	query                       map[string]string
	certificates                *CertificateSlice
}

func (p *ExternalHostProviderServiceImportCertificatesRequest) Header(key, value string) *ExternalHostProviderServiceImportCertificatesRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProviderServiceImportCertificatesRequest) Query(key, value string) *ExternalHostProviderServiceImportCertificatesRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProviderServiceImportCertificatesRequest) Certificates(certificates *CertificateSlice) *ExternalHostProviderServiceImportCertificatesRequest {
	p.certificates = certificates
	return p
}

func (p *ExternalHostProviderServiceImportCertificatesRequest) CertificatesOfAny(anys ...*Certificate) *ExternalHostProviderServiceImportCertificatesRequest {
	if p.certificates == nil {
		p.certificates = new(CertificateSlice)
	}
	p.certificates.slice = append(p.certificates.slice, anys...)
	return p
}

func (p *ExternalHostProviderServiceImportCertificatesRequest) Send() (*ExternalHostProviderServiceImportCertificatesResponse, error) {
	rawURL := fmt.Sprintf("%s%s/importcertificates", p.ExternalHostProviderService.connection.URL(), p.ExternalHostProviderService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Certificates(p.certificates)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(ExternalHostProviderServiceImportCertificatesResponse), nil
}

func (p *ExternalHostProviderServiceImportCertificatesRequest) MustSend() *ExternalHostProviderServiceImportCertificatesResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Import the SSL certificates of the external host provider.
//
type ExternalHostProviderServiceImportCertificatesResponse struct {
}

//
// Import the SSL certificates of the external host provider.
//
func (p *ExternalHostProviderService) ImportCertificates() *ExternalHostProviderServiceImportCertificatesRequest {
	return &ExternalHostProviderServiceImportCertificatesRequest{ExternalHostProviderService: p}
}

//
//
type ExternalHostProviderServiceRemoveRequest struct {
	ExternalHostProviderService *ExternalHostProviderService
	header                      map[string]string
	query                       map[string]string
	async                       *bool
}

func (p *ExternalHostProviderServiceRemoveRequest) Header(key, value string) *ExternalHostProviderServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProviderServiceRemoveRequest) Query(key, value string) *ExternalHostProviderServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProviderServiceRemoveRequest) Async(async bool) *ExternalHostProviderServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *ExternalHostProviderServiceRemoveRequest) Send() (*ExternalHostProviderServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostProviderService.connection.URL(), p.ExternalHostProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(ExternalHostProviderServiceRemoveResponse), nil
}

func (p *ExternalHostProviderServiceRemoveRequest) MustSend() *ExternalHostProviderServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type ExternalHostProviderServiceRemoveResponse struct {
}

//
//
func (p *ExternalHostProviderService) Remove() *ExternalHostProviderServiceRemoveRequest {
	return &ExternalHostProviderServiceRemoveRequest{ExternalHostProviderService: p}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type ExternalHostProviderServiceTestConnectivityRequest struct {
	ExternalHostProviderService *ExternalHostProviderService
	header                      map[string]string
	query                       map[string]string
	async                       *bool
}

func (p *ExternalHostProviderServiceTestConnectivityRequest) Header(key, value string) *ExternalHostProviderServiceTestConnectivityRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProviderServiceTestConnectivityRequest) Query(key, value string) *ExternalHostProviderServiceTestConnectivityRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProviderServiceTestConnectivityRequest) Async(async bool) *ExternalHostProviderServiceTestConnectivityRequest {
	p.async = &async
	return p
}

func (p *ExternalHostProviderServiceTestConnectivityRequest) Send() (*ExternalHostProviderServiceTestConnectivityResponse, error) {
	rawURL := fmt.Sprintf("%s%s/testconnectivity", p.ExternalHostProviderService.connection.URL(), p.ExternalHostProviderService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(ExternalHostProviderServiceTestConnectivityResponse), nil
}

func (p *ExternalHostProviderServiceTestConnectivityRequest) MustSend() *ExternalHostProviderServiceTestConnectivityResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
type ExternalHostProviderServiceTestConnectivityResponse struct {
}

//
// In order to test connectivity for external provider we need
// to run following request where 123 is an id of a provider.
// [source]
// ----
// POST /ovirt-engine/api/externalhostproviders/123/testconnectivity
// ----
//
func (p *ExternalHostProviderService) TestConnectivity() *ExternalHostProviderServiceTestConnectivityRequest {
	return &ExternalHostProviderServiceTestConnectivityRequest{ExternalHostProviderService: p}
}

//
// Update the specified external host provider in the system.
//
type ExternalHostProviderServiceUpdateRequest struct {
	ExternalHostProviderService *ExternalHostProviderService
	header                      map[string]string
	query                       map[string]string
	async                       *bool
	provider                    *ExternalHostProvider
}

func (p *ExternalHostProviderServiceUpdateRequest) Header(key, value string) *ExternalHostProviderServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProviderServiceUpdateRequest) Query(key, value string) *ExternalHostProviderServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProviderServiceUpdateRequest) Async(async bool) *ExternalHostProviderServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *ExternalHostProviderServiceUpdateRequest) Provider(provider *ExternalHostProvider) *ExternalHostProviderServiceUpdateRequest {
	p.provider = provider
	return p
}

func (p *ExternalHostProviderServiceUpdateRequest) Send() (*ExternalHostProviderServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostProviderService.connection.URL(), p.ExternalHostProviderService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLExternalHostProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProviderService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProviderService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProviderService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProviderService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProviderService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalHostProviderServiceUpdateResponse{provider: result}, nil
}

func (p *ExternalHostProviderServiceUpdateRequest) MustSend() *ExternalHostProviderServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Update the specified external host provider in the system.
//
type ExternalHostProviderServiceUpdateResponse struct {
	provider *ExternalHostProvider
}

func (p *ExternalHostProviderServiceUpdateResponse) Provider() (*ExternalHostProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *ExternalHostProviderServiceUpdateResponse) MustProvider() *ExternalHostProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Update the specified external host provider in the system.
//
func (p *ExternalHostProviderService) Update() *ExternalHostProviderServiceUpdateRequest {
	return &ExternalHostProviderServiceUpdateRequest{ExternalHostProviderService: p}
}

//
// A service to view certificates for this external provider.
//
func (op *ExternalHostProviderService) CertificatesService() *ExternalProviderCertificatesService {
	return NewExternalProviderCertificatesService(op.connection, fmt.Sprintf("%s/certificates", op.path))
}

//
//
func (op *ExternalHostProviderService) ComputeResourcesService() *ExternalComputeResourcesService {
	return NewExternalComputeResourcesService(op.connection, fmt.Sprintf("%s/computeresources", op.path))
}

//
//
func (op *ExternalHostProviderService) DiscoveredHostsService() *ExternalDiscoveredHostsService {
	return NewExternalDiscoveredHostsService(op.connection, fmt.Sprintf("%s/discoveredhosts", op.path))
}

//
//
func (op *ExternalHostProviderService) HostGroupsService() *ExternalHostGroupsService {
	return NewExternalHostGroupsService(op.connection, fmt.Sprintf("%s/hostgroups", op.path))
}

//
//
func (op *ExternalHostProviderService) HostsService() *ExternalHostsService {
	return NewExternalHostsService(op.connection, fmt.Sprintf("%s/hosts", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalHostProviderService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "certificates" {
		return op.CertificatesService(), nil
	}
	if strings.HasPrefix(path, "certificates/") {
		return op.CertificatesService().Service(path[13:])
	}
	if path == "computeresources" {
		return op.ComputeResourcesService(), nil
	}
	if strings.HasPrefix(path, "computeresources/") {
		return op.ComputeResourcesService().Service(path[17:])
	}
	if path == "discoveredhosts" {
		return op.DiscoveredHostsService(), nil
	}
	if strings.HasPrefix(path, "discoveredhosts/") {
		return op.DiscoveredHostsService().Service(path[16:])
	}
	if path == "hostgroups" {
		return op.HostGroupsService(), nil
	}
	if strings.HasPrefix(path, "hostgroups/") {
		return op.HostGroupsService().Service(path[11:])
	}
	if path == "hosts" {
		return op.HostsService(), nil
	}
	if strings.HasPrefix(path, "hosts/") {
		return op.HostsService().Service(path[6:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalHostProviderService) String() string {
	return fmt.Sprintf("ExternalHostProviderService:%s", op.path)
}

//
// Manages a single external compute resource.
// Compute resource is a term of host external provider. The external provider also needs to know to where the
// provisioned host needs to register. The login details of the engine are saved as a compute resource  in the external
// provider side.
//
type ExternalComputeResourceService struct {
	BaseService
}

func NewExternalComputeResourceService(connection *Connection, path string) *ExternalComputeResourceService {
	var result ExternalComputeResourceService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves external compute resource details.
// For example, to get the details of compute resource `234` of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/computeresources/234
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_compute_resource href="/ovirt-engine/api/externalhostproviders/123/computeresources/234" id="234">
//   <name>hostname</name>
//   <provider>oVirt</provider>
//   <url>https://hostname/api</url>
//   <user>admin@internal</user>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_compute_resource>
// ----
//
type ExternalComputeResourceServiceGetRequest struct {
	ExternalComputeResourceService *ExternalComputeResourceService
	header                         map[string]string
	query                          map[string]string
	follow                         *string
}

func (p *ExternalComputeResourceServiceGetRequest) Header(key, value string) *ExternalComputeResourceServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalComputeResourceServiceGetRequest) Query(key, value string) *ExternalComputeResourceServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalComputeResourceServiceGetRequest) Follow(follow string) *ExternalComputeResourceServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalComputeResourceServiceGetRequest) Send() (*ExternalComputeResourceServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalComputeResourceService.connection.URL(), p.ExternalComputeResourceService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalComputeResourceService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalComputeResourceService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalComputeResourceService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalComputeResourceService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalComputeResourceService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalComputeResourceReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalComputeResourceServiceGetResponse{resource: result}, nil
}

func (p *ExternalComputeResourceServiceGetRequest) MustSend() *ExternalComputeResourceServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves external compute resource details.
// For example, to get the details of compute resource `234` of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/computeresources/234
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_compute_resource href="/ovirt-engine/api/externalhostproviders/123/computeresources/234" id="234">
//   <name>hostname</name>
//   <provider>oVirt</provider>
//   <url>https://hostname/api</url>
//   <user>admin@internal</user>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_compute_resource>
// ----
//
type ExternalComputeResourceServiceGetResponse struct {
	resource *ExternalComputeResource
}

func (p *ExternalComputeResourceServiceGetResponse) Resource() (*ExternalComputeResource, bool) {
	if p.resource != nil {
		return p.resource, true
	}
	return nil, false
}

func (p *ExternalComputeResourceServiceGetResponse) MustResource() *ExternalComputeResource {
	if p.resource == nil {
		panic("resource in response does not exist")
	}
	return p.resource
}

//
// Retrieves external compute resource details.
// For example, to get the details of compute resource `234` of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/computeresources/234
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_compute_resource href="/ovirt-engine/api/externalhostproviders/123/computeresources/234" id="234">
//   <name>hostname</name>
//   <provider>oVirt</provider>
//   <url>https://hostname/api</url>
//   <user>admin@internal</user>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_compute_resource>
// ----
//
func (p *ExternalComputeResourceService) Get() *ExternalComputeResourceServiceGetRequest {
	return &ExternalComputeResourceServiceGetRequest{ExternalComputeResourceService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalComputeResourceService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalComputeResourceService) String() string {
	return fmt.Sprintf("ExternalComputeResourceService:%s", op.path)
}

//
// A service to manage a Katello erratum.
//
type KatelloErratumService struct {
	BaseService
}

func NewKatelloErratumService(connection *Connection, path string) *KatelloErratumService {
	var result KatelloErratumService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves a Katello erratum.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata/123
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//   <name>RHBA-2013:XYZ</name>
//   <description>The description of the erratum</description>
//   <title>some bug fix update</title>
//   <type>bugfix</type>
//   <issued>2013-11-20T02:00:00.000+02:00</issued>
//   <solution>Few guidelines regarding the solution</solution>
//   <summary>Updated packages that fix one bug are now available for XYZ</summary>
//   <packages>
//     <package>
//       <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//     </package>
//     ...
//   </packages>
// </katello_erratum>
// ----
//
type KatelloErratumServiceGetRequest struct {
	KatelloErratumService *KatelloErratumService
	header                map[string]string
	query                 map[string]string
	follow                *string
}

func (p *KatelloErratumServiceGetRequest) Header(key, value string) *KatelloErratumServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *KatelloErratumServiceGetRequest) Query(key, value string) *KatelloErratumServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *KatelloErratumServiceGetRequest) Follow(follow string) *KatelloErratumServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *KatelloErratumServiceGetRequest) Send() (*KatelloErratumServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.KatelloErratumService.connection.URL(), p.KatelloErratumService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.KatelloErratumService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.KatelloErratumService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.KatelloErratumService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.KatelloErratumService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.KatelloErratumService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLKatelloErratumReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &KatelloErratumServiceGetResponse{erratum: result}, nil
}

func (p *KatelloErratumServiceGetRequest) MustSend() *KatelloErratumServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves a Katello erratum.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata/123
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//   <name>RHBA-2013:XYZ</name>
//   <description>The description of the erratum</description>
//   <title>some bug fix update</title>
//   <type>bugfix</type>
//   <issued>2013-11-20T02:00:00.000+02:00</issued>
//   <solution>Few guidelines regarding the solution</solution>
//   <summary>Updated packages that fix one bug are now available for XYZ</summary>
//   <packages>
//     <package>
//       <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//     </package>
//     ...
//   </packages>
// </katello_erratum>
// ----
//
type KatelloErratumServiceGetResponse struct {
	erratum *KatelloErratum
}

func (p *KatelloErratumServiceGetResponse) Erratum() (*KatelloErratum, bool) {
	if p.erratum != nil {
		return p.erratum, true
	}
	return nil, false
}

func (p *KatelloErratumServiceGetResponse) MustErratum() *KatelloErratum {
	if p.erratum == nil {
		panic("erratum in response does not exist")
	}
	return p.erratum
}

//
// Retrieves a Katello erratum.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata/123
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//   <name>RHBA-2013:XYZ</name>
//   <description>The description of the erratum</description>
//   <title>some bug fix update</title>
//   <type>bugfix</type>
//   <issued>2013-11-20T02:00:00.000+02:00</issued>
//   <solution>Few guidelines regarding the solution</solution>
//   <summary>Updated packages that fix one bug are now available for XYZ</summary>
//   <packages>
//     <package>
//       <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//     </package>
//     ...
//   </packages>
// </katello_erratum>
// ----
//
func (p *KatelloErratumService) Get() *KatelloErratumServiceGetRequest {
	return &KatelloErratumServiceGetRequest{KatelloErratumService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *KatelloErratumService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *KatelloErratumService) String() string {
	return fmt.Sprintf("KatelloErratumService:%s", op.path)
}

//
// A service to manage Katello errata assigned to the engine.
// The information is retrieved from Katello.
//
type EngineKatelloErrataService struct {
	BaseService
}

func NewEngineKatelloErrataService(connection *Connection, path string) *EngineKatelloErrataService {
	var result EngineKatelloErrataService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the representation of the Katello errata.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_errata>
//   <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//     <name>RHBA-2013:XYZ</name>
//     <description>The description of the erratum</description>
//     <title>some bug fix update</title>
//     <type>bugfix</type>
//     <issued>2013-11-20T02:00:00.000+02:00</issued>
//     <solution>Few guidelines regarding the solution</solution>
//     <summary>Updated packages that fix one bug are now available for XYZ</summary>
//     <packages>
//       <package>
//         <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//       </package>
//       ...
//     </packages>
//   </katello_erratum>
//   ...
// </katello_errata>
// ----
// The order of the returned list of erratum isn't guaranteed.
//
type EngineKatelloErrataServiceListRequest struct {
	EngineKatelloErrataService *EngineKatelloErrataService
	header                     map[string]string
	query                      map[string]string
	follow                     *string
	max                        *int64
}

func (p *EngineKatelloErrataServiceListRequest) Header(key, value string) *EngineKatelloErrataServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *EngineKatelloErrataServiceListRequest) Query(key, value string) *EngineKatelloErrataServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *EngineKatelloErrataServiceListRequest) Follow(follow string) *EngineKatelloErrataServiceListRequest {
	p.follow = &follow
	return p
}

func (p *EngineKatelloErrataServiceListRequest) Max(max int64) *EngineKatelloErrataServiceListRequest {
	p.max = &max
	return p
}

func (p *EngineKatelloErrataServiceListRequest) Send() (*EngineKatelloErrataServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.EngineKatelloErrataService.connection.URL(), p.EngineKatelloErrataService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.EngineKatelloErrataService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.EngineKatelloErrataService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.EngineKatelloErrataService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.EngineKatelloErrataService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.EngineKatelloErrataService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLKatelloErratumReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &EngineKatelloErrataServiceListResponse{errata: result}, nil
}

func (p *EngineKatelloErrataServiceListRequest) MustSend() *EngineKatelloErrataServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the representation of the Katello errata.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_errata>
//   <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//     <name>RHBA-2013:XYZ</name>
//     <description>The description of the erratum</description>
//     <title>some bug fix update</title>
//     <type>bugfix</type>
//     <issued>2013-11-20T02:00:00.000+02:00</issued>
//     <solution>Few guidelines regarding the solution</solution>
//     <summary>Updated packages that fix one bug are now available for XYZ</summary>
//     <packages>
//       <package>
//         <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//       </package>
//       ...
//     </packages>
//   </katello_erratum>
//   ...
// </katello_errata>
// ----
// The order of the returned list of erratum isn't guaranteed.
//
type EngineKatelloErrataServiceListResponse struct {
	errata *KatelloErratumSlice
}

func (p *EngineKatelloErrataServiceListResponse) Errata() (*KatelloErratumSlice, bool) {
	if p.errata != nil {
		return p.errata, true
	}
	return nil, false
}

func (p *EngineKatelloErrataServiceListResponse) MustErrata() *KatelloErratumSlice {
	if p.errata == nil {
		panic("errata in response does not exist")
	}
	return p.errata
}

//
// Retrieves the representation of the Katello errata.
// [source]
// ----
// GET /ovirt-engine/api/katelloerrata
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <katello_errata>
//   <katello_erratum href="/ovirt-engine/api/katelloerrata/123" id="123">
//     <name>RHBA-2013:XYZ</name>
//     <description>The description of the erratum</description>
//     <title>some bug fix update</title>
//     <type>bugfix</type>
//     <issued>2013-11-20T02:00:00.000+02:00</issued>
//     <solution>Few guidelines regarding the solution</solution>
//     <summary>Updated packages that fix one bug are now available for XYZ</summary>
//     <packages>
//       <package>
//         <name>libipa_hbac-1.9.2-82.11.el6_4.i686</name>
//       </package>
//       ...
//     </packages>
//   </katello_erratum>
//   ...
// </katello_errata>
// ----
// The order of the returned list of erratum isn't guaranteed.
//
func (p *EngineKatelloErrataService) List() *EngineKatelloErrataServiceListRequest {
	return &EngineKatelloErrataServiceListRequest{EngineKatelloErrataService: p}
}

//
// Reference to the Katello erratum service.
// Use this service to view the erratum by its id.
//
func (op *EngineKatelloErrataService) KatelloErratumService(id string) *KatelloErratumService {
	return NewKatelloErratumService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *EngineKatelloErrataService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.KatelloErratumService(path), nil
	}
	return op.KatelloErratumService(path[:index]).Service(path[index+1:])
}

func (op *EngineKatelloErrataService) String() string {
	return fmt.Sprintf("EngineKatelloErrataService:%s", op.path)
}

//
// This service manages a single host group information.
// Host group is a term of host provider - the host group includes provision details that are applied to new discovered
// host. Information such as subnet, operating system, domain, etc.
//
type ExternalHostGroupService struct {
	BaseService
}

func NewExternalHostGroupService(connection *Connection, path string) *ExternalHostGroupService {
	var result ExternalHostGroupService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get host group information.
// For example, to get the details of hostgroup `234` of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/hostgroups/234
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_host_group href="/ovirt-engine/api/externalhostproviders/123/hostgroups/234" id="234">
//   <name>rhel7</name>
//   <architecture_name>x86_64</architecture_name>
//   <domain_name>s.com</domain_name>
//   <operating_system_name>RedHat 7.3</operating_system_name>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_host_group>
// ----
//
type ExternalHostGroupServiceGetRequest struct {
	ExternalHostGroupService *ExternalHostGroupService
	header                   map[string]string
	query                    map[string]string
	follow                   *string
}

func (p *ExternalHostGroupServiceGetRequest) Header(key, value string) *ExternalHostGroupServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostGroupServiceGetRequest) Query(key, value string) *ExternalHostGroupServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostGroupServiceGetRequest) Follow(follow string) *ExternalHostGroupServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalHostGroupServiceGetRequest) Send() (*ExternalHostGroupServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostGroupService.connection.URL(), p.ExternalHostGroupService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostGroupService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostGroupService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostGroupService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostGroupService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostGroupService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostGroupReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalHostGroupServiceGetResponse{group: result}, nil
}

func (p *ExternalHostGroupServiceGetRequest) MustSend() *ExternalHostGroupServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get host group information.
// For example, to get the details of hostgroup `234` of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/hostgroups/234
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_host_group href="/ovirt-engine/api/externalhostproviders/123/hostgroups/234" id="234">
//   <name>rhel7</name>
//   <architecture_name>x86_64</architecture_name>
//   <domain_name>s.com</domain_name>
//   <operating_system_name>RedHat 7.3</operating_system_name>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_host_group>
// ----
//
type ExternalHostGroupServiceGetResponse struct {
	group *ExternalHostGroup
}

func (p *ExternalHostGroupServiceGetResponse) Group() (*ExternalHostGroup, bool) {
	if p.group != nil {
		return p.group, true
	}
	return nil, false
}

func (p *ExternalHostGroupServiceGetResponse) MustGroup() *ExternalHostGroup {
	if p.group == nil {
		panic("group in response does not exist")
	}
	return p.group
}

//
// Get host group information.
// For example, to get the details of hostgroup `234` of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/hostgroups/234
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_host_group href="/ovirt-engine/api/externalhostproviders/123/hostgroups/234" id="234">
//   <name>rhel7</name>
//   <architecture_name>x86_64</architecture_name>
//   <domain_name>s.com</domain_name>
//   <operating_system_name>RedHat 7.3</operating_system_name>
//   <subnet_name>sat0</subnet_name>
//   <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_host_group>
// ----
//
func (p *ExternalHostGroupService) Get() *ExternalHostGroupServiceGetRequest {
	return &ExternalHostGroupServiceGetRequest{ExternalHostGroupService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalHostGroupService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalHostGroupService) String() string {
	return fmt.Sprintf("ExternalHostGroupService:%s", op.path)
}

//
//
type ExternalHostProvidersService struct {
	BaseService
}

func NewExternalHostProvidersService(connection *Connection, path string) *ExternalHostProvidersService {
	var result ExternalHostProvidersService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a new external host provider to the system.
//
type ExternalHostProvidersServiceAddRequest struct {
	ExternalHostProvidersService *ExternalHostProvidersService
	header                       map[string]string
	query                        map[string]string
	provider                     *ExternalHostProvider
}

func (p *ExternalHostProvidersServiceAddRequest) Header(key, value string) *ExternalHostProvidersServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProvidersServiceAddRequest) Query(key, value string) *ExternalHostProvidersServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProvidersServiceAddRequest) Provider(provider *ExternalHostProvider) *ExternalHostProvidersServiceAddRequest {
	p.provider = provider
	return p
}

func (p *ExternalHostProvidersServiceAddRequest) Send() (*ExternalHostProvidersServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostProvidersService.connection.URL(), p.ExternalHostProvidersService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLExternalHostProviderWriteOne(writer, p.provider, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostProviderReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalHostProvidersServiceAddResponse{provider: result}, nil
}

func (p *ExternalHostProvidersServiceAddRequest) MustSend() *ExternalHostProvidersServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new external host provider to the system.
//
type ExternalHostProvidersServiceAddResponse struct {
	provider *ExternalHostProvider
}

func (p *ExternalHostProvidersServiceAddResponse) Provider() (*ExternalHostProvider, bool) {
	if p.provider != nil {
		return p.provider, true
	}
	return nil, false
}

func (p *ExternalHostProvidersServiceAddResponse) MustProvider() *ExternalHostProvider {
	if p.provider == nil {
		panic("provider in response does not exist")
	}
	return p.provider
}

//
// Add a new external host provider to the system.
//
func (p *ExternalHostProvidersService) Add() *ExternalHostProvidersServiceAddRequest {
	return &ExternalHostProvidersServiceAddRequest{ExternalHostProvidersService: p}
}

//
// Returns the list of external host providers.
// The order of the returned list of host providers isn't guaranteed.
//
type ExternalHostProvidersServiceListRequest struct {
	ExternalHostProvidersService *ExternalHostProvidersService
	header                       map[string]string
	query                        map[string]string
	follow                       *string
	max                          *int64
	search                       *string
}

func (p *ExternalHostProvidersServiceListRequest) Header(key, value string) *ExternalHostProvidersServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostProvidersServiceListRequest) Query(key, value string) *ExternalHostProvidersServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostProvidersServiceListRequest) Follow(follow string) *ExternalHostProvidersServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ExternalHostProvidersServiceListRequest) Max(max int64) *ExternalHostProvidersServiceListRequest {
	p.max = &max
	return p
}

func (p *ExternalHostProvidersServiceListRequest) Search(search string) *ExternalHostProvidersServiceListRequest {
	p.search = &search
	return p
}

func (p *ExternalHostProvidersServiceListRequest) Send() (*ExternalHostProvidersServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostProvidersService.connection.URL(), p.ExternalHostProvidersService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostProvidersService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostProvidersService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostProvidersService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostProvidersService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostProvidersService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostProviderReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ExternalHostProvidersServiceListResponse{providers: result}, nil
}

func (p *ExternalHostProvidersServiceListRequest) MustSend() *ExternalHostProvidersServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of external host providers.
// The order of the returned list of host providers isn't guaranteed.
//
type ExternalHostProvidersServiceListResponse struct {
	providers *ExternalHostProviderSlice
}

func (p *ExternalHostProvidersServiceListResponse) Providers() (*ExternalHostProviderSlice, bool) {
	if p.providers != nil {
		return p.providers, true
	}
	return nil, false
}

func (p *ExternalHostProvidersServiceListResponse) MustProviders() *ExternalHostProviderSlice {
	if p.providers == nil {
		panic("providers in response does not exist")
	}
	return p.providers
}

//
// Returns the list of external host providers.
// The order of the returned list of host providers isn't guaranteed.
//
func (p *ExternalHostProvidersService) List() *ExternalHostProvidersServiceListRequest {
	return &ExternalHostProvidersServiceListRequest{ExternalHostProvidersService: p}
}

//
//
func (op *ExternalHostProvidersService) ProviderService(id string) *ExternalHostProviderService {
	return NewExternalHostProviderService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalHostProvidersService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ProviderService(path), nil
	}
	return op.ProviderService(path[:index]).Service(path[index+1:])
}

func (op *ExternalHostProvidersService) String() string {
	return fmt.Sprintf("ExternalHostProvidersService:%s", op.path)
}

//
// This service manages a single discovered host.
//
type ExternalDiscoveredHostService struct {
	BaseService
}

func NewExternalDiscoveredHostService(connection *Connection, path string) *ExternalDiscoveredHostService {
	var result ExternalDiscoveredHostService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get discovered host info.
// Retrieves information about an host that is managed in external provider management system, such as Foreman. The
// information includes hostname, address, subnet, base image and more.
// For example, to get the details of host `234` from provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/discoveredhosts/234
// ....
// The result will be like this:
// [source,xml]
// ----
// <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/234" id="234">
//  <name>mac001a4ad04040</name>
//  <ip>10.34.67.43</ip>
//  <last_report>2017-04-24 11:05:41 UTC</last_report>
//  <mac>00:1a:4a:d0:40:40</mac>
//  <subnet_name>sat0</subnet_name>
//  <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_discovered_host>
// ----
//
type ExternalDiscoveredHostServiceGetRequest struct {
	ExternalDiscoveredHostService *ExternalDiscoveredHostService
	header                        map[string]string
	query                         map[string]string
	follow                        *string
}

func (p *ExternalDiscoveredHostServiceGetRequest) Header(key, value string) *ExternalDiscoveredHostServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalDiscoveredHostServiceGetRequest) Query(key, value string) *ExternalDiscoveredHostServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalDiscoveredHostServiceGetRequest) Follow(follow string) *ExternalDiscoveredHostServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *ExternalDiscoveredHostServiceGetRequest) Send() (*ExternalDiscoveredHostServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalDiscoveredHostService.connection.URL(), p.ExternalDiscoveredHostService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalDiscoveredHostService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalDiscoveredHostService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalDiscoveredHostService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalDiscoveredHostService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalDiscoveredHostService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalDiscoveredHostReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &ExternalDiscoveredHostServiceGetResponse{host: result}, nil
}

func (p *ExternalDiscoveredHostServiceGetRequest) MustSend() *ExternalDiscoveredHostServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get discovered host info.
// Retrieves information about an host that is managed in external provider management system, such as Foreman. The
// information includes hostname, address, subnet, base image and more.
// For example, to get the details of host `234` from provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/discoveredhosts/234
// ....
// The result will be like this:
// [source,xml]
// ----
// <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/234" id="234">
//  <name>mac001a4ad04040</name>
//  <ip>10.34.67.43</ip>
//  <last_report>2017-04-24 11:05:41 UTC</last_report>
//  <mac>00:1a:4a:d0:40:40</mac>
//  <subnet_name>sat0</subnet_name>
//  <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_discovered_host>
// ----
//
type ExternalDiscoveredHostServiceGetResponse struct {
	host *ExternalDiscoveredHost
}

func (p *ExternalDiscoveredHostServiceGetResponse) Host() (*ExternalDiscoveredHost, bool) {
	if p.host != nil {
		return p.host, true
	}
	return nil, false
}

func (p *ExternalDiscoveredHostServiceGetResponse) MustHost() *ExternalDiscoveredHost {
	if p.host == nil {
		panic("host in response does not exist")
	}
	return p.host
}

//
// Get discovered host info.
// Retrieves information about an host that is managed in external provider management system, such as Foreman. The
// information includes hostname, address, subnet, base image and more.
// For example, to get the details of host `234` from provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/discoveredhosts/234
// ....
// The result will be like this:
// [source,xml]
// ----
// <external_discovered_host href="/ovirt-engine/api/externalhostproviders/123/discoveredhosts/234" id="234">
//  <name>mac001a4ad04040</name>
//  <ip>10.34.67.43</ip>
//  <last_report>2017-04-24 11:05:41 UTC</last_report>
//  <mac>00:1a:4a:d0:40:40</mac>
//  <subnet_name>sat0</subnet_name>
//  <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
// </external_discovered_host>
// ----
//
func (p *ExternalDiscoveredHostService) Get() *ExternalDiscoveredHostServiceGetRequest {
	return &ExternalDiscoveredHostServiceGetRequest{ExternalDiscoveredHostService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalDiscoveredHostService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *ExternalDiscoveredHostService) String() string {
	return fmt.Sprintf("ExternalDiscoveredHostService:%s", op.path)
}

//
// This service manages hostgroups.
//
type ExternalHostGroupsService struct {
	BaseService
}

func NewExternalHostGroupsService(connection *Connection, path string) *ExternalHostGroupsService {
	var result ExternalHostGroupsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get host groups list from external host provider.
// Host group is a term of host providers - the host group includes provision details. This API returns all possible
// hostgroups exposed by the external provider.
// For example, to get the details of all host groups of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/hostgroups
// ....
// The response will be like this:
// [source,xml]
// ----
// <external_host_groups>
//   <external_host_group href="/ovirt-engine/api/externalhostproviders/123/hostgroups/234" id="234">
//     <name>rhel7</name>
//     <architecture_name>x86_64</architecture_name>
//     <domain_name>example.com</domain_name>
//     <operating_system_name>RedHat 7.3</operating_system_name>
//     <subnet_name>sat0</subnet_name>
//     <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//   </external_host_group>
//   ...
// </external_host_groups>
// ----
// The order of the returned list of host groups isn't guaranteed.
//
type ExternalHostGroupsServiceListRequest struct {
	ExternalHostGroupsService *ExternalHostGroupsService
	header                    map[string]string
	query                     map[string]string
	follow                    *string
	max                       *int64
}

func (p *ExternalHostGroupsServiceListRequest) Header(key, value string) *ExternalHostGroupsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostGroupsServiceListRequest) Query(key, value string) *ExternalHostGroupsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostGroupsServiceListRequest) Follow(follow string) *ExternalHostGroupsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ExternalHostGroupsServiceListRequest) Max(max int64) *ExternalHostGroupsServiceListRequest {
	p.max = &max
	return p
}

func (p *ExternalHostGroupsServiceListRequest) Send() (*ExternalHostGroupsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostGroupsService.connection.URL(), p.ExternalHostGroupsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostGroupsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostGroupsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostGroupsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostGroupsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostGroupsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostGroupReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ExternalHostGroupsServiceListResponse{groups: result}, nil
}

func (p *ExternalHostGroupsServiceListRequest) MustSend() *ExternalHostGroupsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get host groups list from external host provider.
// Host group is a term of host providers - the host group includes provision details. This API returns all possible
// hostgroups exposed by the external provider.
// For example, to get the details of all host groups of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/hostgroups
// ....
// The response will be like this:
// [source,xml]
// ----
// <external_host_groups>
//   <external_host_group href="/ovirt-engine/api/externalhostproviders/123/hostgroups/234" id="234">
//     <name>rhel7</name>
//     <architecture_name>x86_64</architecture_name>
//     <domain_name>example.com</domain_name>
//     <operating_system_name>RedHat 7.3</operating_system_name>
//     <subnet_name>sat0</subnet_name>
//     <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//   </external_host_group>
//   ...
// </external_host_groups>
// ----
// The order of the returned list of host groups isn't guaranteed.
//
type ExternalHostGroupsServiceListResponse struct {
	groups *ExternalHostGroupSlice
}

func (p *ExternalHostGroupsServiceListResponse) Groups() (*ExternalHostGroupSlice, bool) {
	if p.groups != nil {
		return p.groups, true
	}
	return nil, false
}

func (p *ExternalHostGroupsServiceListResponse) MustGroups() *ExternalHostGroupSlice {
	if p.groups == nil {
		panic("groups in response does not exist")
	}
	return p.groups
}

//
// Get host groups list from external host provider.
// Host group is a term of host providers - the host group includes provision details. This API returns all possible
// hostgroups exposed by the external provider.
// For example, to get the details of all host groups of provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/hostgroups
// ....
// The response will be like this:
// [source,xml]
// ----
// <external_host_groups>
//   <external_host_group href="/ovirt-engine/api/externalhostproviders/123/hostgroups/234" id="234">
//     <name>rhel7</name>
//     <architecture_name>x86_64</architecture_name>
//     <domain_name>example.com</domain_name>
//     <operating_system_name>RedHat 7.3</operating_system_name>
//     <subnet_name>sat0</subnet_name>
//     <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//   </external_host_group>
//   ...
// </external_host_groups>
// ----
// The order of the returned list of host groups isn't guaranteed.
//
func (p *ExternalHostGroupsService) List() *ExternalHostGroupsServiceListRequest {
	return &ExternalHostGroupsServiceListRequest{ExternalHostGroupsService: p}
}

//
// This service manages hostgroup instance.
//
func (op *ExternalHostGroupsService) GroupService(id string) *ExternalHostGroupService {
	return NewExternalHostGroupService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalHostGroupsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.GroupService(path), nil
	}
	return op.GroupService(path[:index]).Service(path[index+1:])
}

func (op *ExternalHostGroupsService) String() string {
	return fmt.Sprintf("ExternalHostGroupsService:%s", op.path)
}

//
// Manages a collection of external compute resources.
// Compute resource is a term of host external provider. The external provider also needs to know to where the
// provisioned host needs to register. The login details of the engine is saved as a compute resource in the external
// provider side.
//
type ExternalComputeResourcesService struct {
	BaseService
}

func NewExternalComputeResourcesService(connection *Connection, path string) *ExternalComputeResourcesService {
	var result ExternalComputeResourcesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves a list of external compute resources.
// For example, to retrieve the compute resources of external host provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/computeresources
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_compute_resources>
//   <external_compute_resource href="/ovirt-engine/api/externalhostproviders/123/computeresources/234" id="234">
//     <name>hostname</name>
//     <provider>oVirt</provider>
//     <url>https://address/api</url>
//     <user>admin@internal</user>
//     <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//    </external_compute_resource>
//    ...
// </external_compute_resources>
// ----
// The order of the returned list of compute resources isn't guaranteed.
//
type ExternalComputeResourcesServiceListRequest struct {
	ExternalComputeResourcesService *ExternalComputeResourcesService
	header                          map[string]string
	query                           map[string]string
	follow                          *string
	max                             *int64
}

func (p *ExternalComputeResourcesServiceListRequest) Header(key, value string) *ExternalComputeResourcesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalComputeResourcesServiceListRequest) Query(key, value string) *ExternalComputeResourcesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalComputeResourcesServiceListRequest) Follow(follow string) *ExternalComputeResourcesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ExternalComputeResourcesServiceListRequest) Max(max int64) *ExternalComputeResourcesServiceListRequest {
	p.max = &max
	return p
}

func (p *ExternalComputeResourcesServiceListRequest) Send() (*ExternalComputeResourcesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalComputeResourcesService.connection.URL(), p.ExternalComputeResourcesService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalComputeResourcesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalComputeResourcesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalComputeResourcesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalComputeResourcesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalComputeResourcesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalComputeResourceReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ExternalComputeResourcesServiceListResponse{resources: result}, nil
}

func (p *ExternalComputeResourcesServiceListRequest) MustSend() *ExternalComputeResourcesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves a list of external compute resources.
// For example, to retrieve the compute resources of external host provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/computeresources
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_compute_resources>
//   <external_compute_resource href="/ovirt-engine/api/externalhostproviders/123/computeresources/234" id="234">
//     <name>hostname</name>
//     <provider>oVirt</provider>
//     <url>https://address/api</url>
//     <user>admin@internal</user>
//     <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//    </external_compute_resource>
//    ...
// </external_compute_resources>
// ----
// The order of the returned list of compute resources isn't guaranteed.
//
type ExternalComputeResourcesServiceListResponse struct {
	resources *ExternalComputeResourceSlice
}

func (p *ExternalComputeResourcesServiceListResponse) Resources() (*ExternalComputeResourceSlice, bool) {
	if p.resources != nil {
		return p.resources, true
	}
	return nil, false
}

func (p *ExternalComputeResourcesServiceListResponse) MustResources() *ExternalComputeResourceSlice {
	if p.resources == nil {
		panic("resources in response does not exist")
	}
	return p.resources
}

//
// Retrieves a list of external compute resources.
// For example, to retrieve the compute resources of external host provider `123`, send a request like this:
// ....
// GET /ovirt-engine/api/externalhostproviders/123/computeresources
// ....
// It will return a response like this:
// [source,xml]
// ----
// <external_compute_resources>
//   <external_compute_resource href="/ovirt-engine/api/externalhostproviders/123/computeresources/234" id="234">
//     <name>hostname</name>
//     <provider>oVirt</provider>
//     <url>https://address/api</url>
//     <user>admin@internal</user>
//     <external_host_provider href="/ovirt-engine/api/externalhostproviders/123" id="123"/>
//    </external_compute_resource>
//    ...
// </external_compute_resources>
// ----
// The order of the returned list of compute resources isn't guaranteed.
//
func (p *ExternalComputeResourcesService) List() *ExternalComputeResourcesServiceListRequest {
	return &ExternalComputeResourcesServiceListRequest{ExternalComputeResourcesService: p}
}

//
// This service manages compute resource instance
//
func (op *ExternalComputeResourcesService) ResourceService(id string) *ExternalComputeResourceService {
	return NewExternalComputeResourceService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalComputeResourcesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.ResourceService(path), nil
	}
	return op.ResourceService(path[:index]).Service(path[index+1:])
}

func (op *ExternalComputeResourcesService) String() string {
	return fmt.Sprintf("ExternalComputeResourcesService:%s", op.path)
}

//
//
type ExternalHostsService struct {
	BaseService
}

func NewExternalHostsService(connection *Connection, path string) *ExternalHostsService {
	var result ExternalHostsService
	result.connection = connection
	result.path = path
	return &result
}

//
// Return the list of external hosts.
// The order of the returned list of hosts isn't guaranteed.
//
type ExternalHostsServiceListRequest struct {
	ExternalHostsService *ExternalHostsService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *ExternalHostsServiceListRequest) Header(key, value string) *ExternalHostsServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *ExternalHostsServiceListRequest) Query(key, value string) *ExternalHostsServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *ExternalHostsServiceListRequest) Follow(follow string) *ExternalHostsServiceListRequest {
	p.follow = &follow
	return p
}

func (p *ExternalHostsServiceListRequest) Max(max int64) *ExternalHostsServiceListRequest {
	p.max = &max
	return p
}

func (p *ExternalHostsServiceListRequest) Send() (*ExternalHostsServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.ExternalHostsService.connection.URL(), p.ExternalHostsService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.ExternalHostsService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.ExternalHostsService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.ExternalHostsService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.ExternalHostsService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.ExternalHostsService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLExternalHostReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &ExternalHostsServiceListResponse{hosts: result}, nil
}

func (p *ExternalHostsServiceListRequest) MustSend() *ExternalHostsServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Return the list of external hosts.
// The order of the returned list of hosts isn't guaranteed.
//
type ExternalHostsServiceListResponse struct {
	hosts *ExternalHostSlice
}

func (p *ExternalHostsServiceListResponse) Hosts() (*ExternalHostSlice, bool) {
	if p.hosts != nil {
		return p.hosts, true
	}
	return nil, false
}

func (p *ExternalHostsServiceListResponse) MustHosts() *ExternalHostSlice {
	if p.hosts == nil {
		panic("hosts in response does not exist")
	}
	return p.hosts
}

//
// Return the list of external hosts.
// The order of the returned list of hosts isn't guaranteed.
//
func (p *ExternalHostsService) List() *ExternalHostsServiceListRequest {
	return &ExternalHostsServiceListRequest{ExternalHostsService: p}
}

//
//
func (op *ExternalHostsService) HostService(id string) *ExternalHostService {
	return NewExternalHostService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *ExternalHostsService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.HostService(path), nil
	}
	return op.HostService(path[:index]).Service(path[index+1:])
}

func (op *ExternalHostsService) String() string {
	return fmt.Sprintf("ExternalHostsService:%s", op.path)
}

//
//
type CopyableService struct {
	BaseService
}

func NewCopyableService(connection *Connection, path string) *CopyableService {
	var result CopyableService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type CopyableServiceCopyRequest struct {
	CopyableService *CopyableService
	header          map[string]string
	query           map[string]string
	async           *bool
}

func (p *CopyableServiceCopyRequest) Header(key, value string) *CopyableServiceCopyRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *CopyableServiceCopyRequest) Query(key, value string) *CopyableServiceCopyRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *CopyableServiceCopyRequest) Async(async bool) *CopyableServiceCopyRequest {
	p.async = &async
	return p
}

func (p *CopyableServiceCopyRequest) Send() (*CopyableServiceCopyResponse, error) {
	rawURL := fmt.Sprintf("%s%s/copy", p.CopyableService.connection.URL(), p.CopyableService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.CopyableService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.CopyableService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.CopyableService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.CopyableService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.CopyableService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(CopyableServiceCopyResponse), nil
}

func (p *CopyableServiceCopyRequest) MustSend() *CopyableServiceCopyResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type CopyableServiceCopyResponse struct {
}

//
//
func (p *CopyableService) Copy() *CopyableServiceCopyRequest {
	return &CopyableServiceCopyRequest{CopyableService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *CopyableService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *CopyableService) String() string {
	return fmt.Sprintf("CopyableService:%s", op.path)
}

//
// A service to manage an icon (read-only).
//
type IconService struct {
	BaseService
}

func NewIconService(connection *Connection, path string) *IconService {
	var result IconService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get an icon.
// [source]
// ----
// GET /ovirt-engine/api/icons/123
// ----
// You will get a XML response like this one:
// [source,xml]
// ----
// <icon id="123">
//   <data>Some binary data here</data>
//   <media_type>image/png</media_type>
// </icon>
// ----
//
type IconServiceGetRequest struct {
	IconService *IconService
	header      map[string]string
	query       map[string]string
	follow      *string
}

func (p *IconServiceGetRequest) Header(key, value string) *IconServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *IconServiceGetRequest) Query(key, value string) *IconServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *IconServiceGetRequest) Follow(follow string) *IconServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *IconServiceGetRequest) Send() (*IconServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.IconService.connection.URL(), p.IconService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.IconService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.IconService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.IconService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.IconService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.IconService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLIconReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &IconServiceGetResponse{icon: result}, nil
}

func (p *IconServiceGetRequest) MustSend() *IconServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get an icon.
// [source]
// ----
// GET /ovirt-engine/api/icons/123
// ----
// You will get a XML response like this one:
// [source,xml]
// ----
// <icon id="123">
//   <data>Some binary data here</data>
//   <media_type>image/png</media_type>
// </icon>
// ----
//
type IconServiceGetResponse struct {
	icon *Icon
}

func (p *IconServiceGetResponse) Icon() (*Icon, bool) {
	if p.icon != nil {
		return p.icon, true
	}
	return nil, false
}

func (p *IconServiceGetResponse) MustIcon() *Icon {
	if p.icon == nil {
		panic("icon in response does not exist")
	}
	return p.icon
}

//
// Get an icon.
// [source]
// ----
// GET /ovirt-engine/api/icons/123
// ----
// You will get a XML response like this one:
// [source,xml]
// ----
// <icon id="123">
//   <data>Some binary data here</data>
//   <media_type>image/png</media_type>
// </icon>
// ----
//
func (p *IconService) Get() *IconServiceGetRequest {
	return &IconServiceGetRequest{IconService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *IconService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *IconService) String() string {
	return fmt.Sprintf("IconService:%s", op.path)
}

//
// A service to manage a job.
//
type JobService struct {
	BaseService
}

func NewJobService(connection *Connection, path string) *JobService {
	var result JobService
	result.connection = connection
	result.path = path
	return &result
}

//
// Set an external job execution to be cleared by the system.
// For example, to set a job with identifier `123` send the following request:
// [source]
// ----
// POST /ovirt-engine/api/jobs/clear
// ----
// With the following request body:
// [source,xml]
// ----
// <action/>
// ----
//
type JobServiceClearRequest struct {
	JobService *JobService
	header     map[string]string
	query      map[string]string
	async      *bool
}

func (p *JobServiceClearRequest) Header(key, value string) *JobServiceClearRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *JobServiceClearRequest) Query(key, value string) *JobServiceClearRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *JobServiceClearRequest) Async(async bool) *JobServiceClearRequest {
	p.async = &async
	return p
}

func (p *JobServiceClearRequest) Send() (*JobServiceClearResponse, error) {
	rawURL := fmt.Sprintf("%s%s/clear", p.JobService.connection.URL(), p.JobService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.JobService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.JobService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.JobService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.JobService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.JobService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(JobServiceClearResponse), nil
}

func (p *JobServiceClearRequest) MustSend() *JobServiceClearResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Set an external job execution to be cleared by the system.
// For example, to set a job with identifier `123` send the following request:
// [source]
// ----
// POST /ovirt-engine/api/jobs/clear
// ----
// With the following request body:
// [source,xml]
// ----
// <action/>
// ----
//
type JobServiceClearResponse struct {
}

//
// Set an external job execution to be cleared by the system.
// For example, to set a job with identifier `123` send the following request:
// [source]
// ----
// POST /ovirt-engine/api/jobs/clear
// ----
// With the following request body:
// [source,xml]
// ----
// <action/>
// ----
//
func (p *JobService) Clear() *JobServiceClearRequest {
	return &JobServiceClearRequest{JobService: p}
}

//
// Marks an external job execution as ended.
// For example, to terminate a job with identifier `123` send the following request:
// [source]
// ----
// POST /ovirt-engine/api/jobs/end
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//   <force>true</force>
//   <status>finished</status>
// </action>
// ----
//
type JobServiceEndRequest struct {
	JobService *JobService
	header     map[string]string
	query      map[string]string
	async      *bool
	force      *bool
	succeeded  *bool
}

func (p *JobServiceEndRequest) Header(key, value string) *JobServiceEndRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *JobServiceEndRequest) Query(key, value string) *JobServiceEndRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *JobServiceEndRequest) Async(async bool) *JobServiceEndRequest {
	p.async = &async
	return p
}

func (p *JobServiceEndRequest) Force(force bool) *JobServiceEndRequest {
	p.force = &force
	return p
}

func (p *JobServiceEndRequest) Succeeded(succeeded bool) *JobServiceEndRequest {
	p.succeeded = &succeeded
	return p
}

func (p *JobServiceEndRequest) Send() (*JobServiceEndResponse, error) {
	rawURL := fmt.Sprintf("%s%s/end", p.JobService.connection.URL(), p.JobService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	if p.succeeded != nil {
		actionBuilder.Succeeded(*p.succeeded)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.JobService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.JobService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.JobService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.JobService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.JobService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(JobServiceEndResponse), nil
}

func (p *JobServiceEndRequest) MustSend() *JobServiceEndResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Marks an external job execution as ended.
// For example, to terminate a job with identifier `123` send the following request:
// [source]
// ----
// POST /ovirt-engine/api/jobs/end
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//   <force>true</force>
//   <status>finished</status>
// </action>
// ----
//
type JobServiceEndResponse struct {
}

//
// Marks an external job execution as ended.
// For example, to terminate a job with identifier `123` send the following request:
// [source]
// ----
// POST /ovirt-engine/api/jobs/end
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//   <force>true</force>
//   <status>finished</status>
// </action>
// ----
//
func (p *JobService) End() *JobServiceEndRequest {
	return &JobServiceEndRequest{JobService: p}
}

//
// Retrieves a job.
// [source]
// ----
// GET /ovirt-engine/api/jobs/123
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <job href="/ovirt-engine/api/jobs/123" id="123">
//   <actions>
//     <link href="/ovirt-engine/api/jobs/123/clear" rel="clear"/>
//     <link href="/ovirt-engine/api/jobs/123/end" rel="end"/>
//   </actions>
//   <description>Adding Disk</description>
//   <link href="/ovirt-engine/api/jobs/123/steps" rel="steps"/>
//   <auto_cleared>true</auto_cleared>
//   <end_time>2016-12-12T23:07:29.758+02:00</end_time>
//   <external>false</external>
//   <last_updated>2016-12-12T23:07:29.758+02:00</last_updated>
//   <start_time>2016-12-12T23:07:26.593+02:00</start_time>
//   <status>failed</status>
//   <owner href="/ovirt-engine/api/users/456" id="456"/>
// </job>
// ----
//
type JobServiceGetRequest struct {
	JobService *JobService
	header     map[string]string
	query      map[string]string
	follow     *string
}

func (p *JobServiceGetRequest) Header(key, value string) *JobServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *JobServiceGetRequest) Query(key, value string) *JobServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *JobServiceGetRequest) Follow(follow string) *JobServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *JobServiceGetRequest) Send() (*JobServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.JobService.connection.URL(), p.JobService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.JobService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.JobService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.JobService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.JobService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.JobService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLJobReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &JobServiceGetResponse{job: result}, nil
}

func (p *JobServiceGetRequest) MustSend() *JobServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves a job.
// [source]
// ----
// GET /ovirt-engine/api/jobs/123
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <job href="/ovirt-engine/api/jobs/123" id="123">
//   <actions>
//     <link href="/ovirt-engine/api/jobs/123/clear" rel="clear"/>
//     <link href="/ovirt-engine/api/jobs/123/end" rel="end"/>
//   </actions>
//   <description>Adding Disk</description>
//   <link href="/ovirt-engine/api/jobs/123/steps" rel="steps"/>
//   <auto_cleared>true</auto_cleared>
//   <end_time>2016-12-12T23:07:29.758+02:00</end_time>
//   <external>false</external>
//   <last_updated>2016-12-12T23:07:29.758+02:00</last_updated>
//   <start_time>2016-12-12T23:07:26.593+02:00</start_time>
//   <status>failed</status>
//   <owner href="/ovirt-engine/api/users/456" id="456"/>
// </job>
// ----
//
type JobServiceGetResponse struct {
	job *Job
}

func (p *JobServiceGetResponse) Job() (*Job, bool) {
	if p.job != nil {
		return p.job, true
	}
	return nil, false
}

func (p *JobServiceGetResponse) MustJob() *Job {
	if p.job == nil {
		panic("job in response does not exist")
	}
	return p.job
}

//
// Retrieves a job.
// [source]
// ----
// GET /ovirt-engine/api/jobs/123
// ----
// You will receive response in XML like this one:
// [source,xml]
// ----
// <job href="/ovirt-engine/api/jobs/123" id="123">
//   <actions>
//     <link href="/ovirt-engine/api/jobs/123/clear" rel="clear"/>
//     <link href="/ovirt-engine/api/jobs/123/end" rel="end"/>
//   </actions>
//   <description>Adding Disk</description>
//   <link href="/ovirt-engine/api/jobs/123/steps" rel="steps"/>
//   <auto_cleared>true</auto_cleared>
//   <end_time>2016-12-12T23:07:29.758+02:00</end_time>
//   <external>false</external>
//   <last_updated>2016-12-12T23:07:29.758+02:00</last_updated>
//   <start_time>2016-12-12T23:07:26.593+02:00</start_time>
//   <status>failed</status>
//   <owner href="/ovirt-engine/api/users/456" id="456"/>
// </job>
// ----
//
func (p *JobService) Get() *JobServiceGetRequest {
	return &JobServiceGetRequest{JobService: p}
}

//
// List all the steps of the job.
// The order of the returned list of steps isn't guaranteed.
//
func (op *JobService) StepsService() *StepsService {
	return NewStepsService(op.connection, fmt.Sprintf("%s/steps", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *JobService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "steps" {
		return op.StepsService(), nil
	}
	if strings.HasPrefix(path, "steps/") {
		return op.StepsService().Service(path[6:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *JobService) String() string {
	return fmt.Sprintf("JobService:%s", op.path)
}

//
// A service to manage a specific tag in the system.
//
type TagService struct {
	BaseService
}

func NewTagService(connection *Connection, path string) *TagService {
	var result TagService
	result.connection = connection
	result.path = path
	return &result
}

//
// Gets the information about the tag.
// For example to retrieve the information about the tag with the id `123` send a request like this:
// ....
// GET /ovirt-engine/api/tags/123
// ....
// [source,xml]
// ----
// <tag href="/ovirt-engine/api/tags/123" id="123">
//   <name>root</name>
//   <description>root</description>
// </tag>
// ----
//
type TagServiceGetRequest struct {
	TagService *TagService
	header     map[string]string
	query      map[string]string
	follow     *string
}

func (p *TagServiceGetRequest) Header(key, value string) *TagServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TagServiceGetRequest) Query(key, value string) *TagServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TagServiceGetRequest) Follow(follow string) *TagServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *TagServiceGetRequest) Send() (*TagServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TagService.connection.URL(), p.TagService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TagService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TagService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TagService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TagService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TagService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLTagReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &TagServiceGetResponse{tag: result}, nil
}

func (p *TagServiceGetRequest) MustSend() *TagServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Gets the information about the tag.
// For example to retrieve the information about the tag with the id `123` send a request like this:
// ....
// GET /ovirt-engine/api/tags/123
// ....
// [source,xml]
// ----
// <tag href="/ovirt-engine/api/tags/123" id="123">
//   <name>root</name>
//   <description>root</description>
// </tag>
// ----
//
type TagServiceGetResponse struct {
	tag *Tag
}

func (p *TagServiceGetResponse) Tag() (*Tag, bool) {
	if p.tag != nil {
		return p.tag, true
	}
	return nil, false
}

func (p *TagServiceGetResponse) MustTag() *Tag {
	if p.tag == nil {
		panic("tag in response does not exist")
	}
	return p.tag
}

//
// Gets the information about the tag.
// For example to retrieve the information about the tag with the id `123` send a request like this:
// ....
// GET /ovirt-engine/api/tags/123
// ....
// [source,xml]
// ----
// <tag href="/ovirt-engine/api/tags/123" id="123">
//   <name>root</name>
//   <description>root</description>
// </tag>
// ----
//
func (p *TagService) Get() *TagServiceGetRequest {
	return &TagServiceGetRequest{TagService: p}
}

//
// Removes the tag from the system.
// For example to remove the tag with id `123` send a request like this:
// ....
// DELETE /ovirt-engine/api/tags/123
// ....
//
type TagServiceRemoveRequest struct {
	TagService *TagService
	header     map[string]string
	query      map[string]string
	async      *bool
}

func (p *TagServiceRemoveRequest) Header(key, value string) *TagServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TagServiceRemoveRequest) Query(key, value string) *TagServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TagServiceRemoveRequest) Async(async bool) *TagServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *TagServiceRemoveRequest) Send() (*TagServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TagService.connection.URL(), p.TagService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TagService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TagService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TagService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TagService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TagService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(TagServiceRemoveResponse), nil
}

func (p *TagServiceRemoveRequest) MustSend() *TagServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the tag from the system.
// For example to remove the tag with id `123` send a request like this:
// ....
// DELETE /ovirt-engine/api/tags/123
// ....
//
type TagServiceRemoveResponse struct {
}

//
// Removes the tag from the system.
// For example to remove the tag with id `123` send a request like this:
// ....
// DELETE /ovirt-engine/api/tags/123
// ....
//
func (p *TagService) Remove() *TagServiceRemoveRequest {
	return &TagServiceRemoveRequest{TagService: p}
}

//
// Updates the tag entity.
// For example to update parent tag to tag with id `456` of the tag with id `123` send a request like this:
// ....
// PUT /ovirt-engine/api/tags/123
// ....
// With request body like:
// [source,xml]
// ----
// <tag>
//   <parent id="456"/>
// </tag>
// ----
// You may also specify a tag name instead of id. For example to update parent tag to tag with name `mytag`
// of the tag with id `123` send a request like this:
// [source,xml]
// ----
// <tag>
//   <parent>
//     <name>mytag</name>
//   </parent>
// </tag>
// ----
//
type TagServiceUpdateRequest struct {
	TagService *TagService
	header     map[string]string
	query      map[string]string
	async      *bool
	tag        *Tag
}

func (p *TagServiceUpdateRequest) Header(key, value string) *TagServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *TagServiceUpdateRequest) Query(key, value string) *TagServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *TagServiceUpdateRequest) Async(async bool) *TagServiceUpdateRequest {
	p.async = &async
	return p
}

func (p *TagServiceUpdateRequest) Tag(tag *Tag) *TagServiceUpdateRequest {
	p.tag = tag
	return p
}

func (p *TagServiceUpdateRequest) Send() (*TagServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.TagService.connection.URL(), p.TagService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLTagWriteOne(writer, p.tag, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.TagService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.TagService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.TagService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.TagService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.TagService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLTagReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &TagServiceUpdateResponse{tag: result}, nil
}

func (p *TagServiceUpdateRequest) MustSend() *TagServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates the tag entity.
// For example to update parent tag to tag with id `456` of the tag with id `123` send a request like this:
// ....
// PUT /ovirt-engine/api/tags/123
// ....
// With request body like:
// [source,xml]
// ----
// <tag>
//   <parent id="456"/>
// </tag>
// ----
// You may also specify a tag name instead of id. For example to update parent tag to tag with name `mytag`
// of the tag with id `123` send a request like this:
// [source,xml]
// ----
// <tag>
//   <parent>
//     <name>mytag</name>
//   </parent>
// </tag>
// ----
//
type TagServiceUpdateResponse struct {
	tag *Tag
}

func (p *TagServiceUpdateResponse) Tag() (*Tag, bool) {
	if p.tag != nil {
		return p.tag, true
	}
	return nil, false
}

func (p *TagServiceUpdateResponse) MustTag() *Tag {
	if p.tag == nil {
		panic("tag in response does not exist")
	}
	return p.tag
}

//
// Updates the tag entity.
// For example to update parent tag to tag with id `456` of the tag with id `123` send a request like this:
// ....
// PUT /ovirt-engine/api/tags/123
// ....
// With request body like:
// [source,xml]
// ----
// <tag>
//   <parent id="456"/>
// </tag>
// ----
// You may also specify a tag name instead of id. For example to update parent tag to tag with name `mytag`
// of the tag with id `123` send a request like this:
// [source,xml]
// ----
// <tag>
//   <parent>
//     <name>mytag</name>
//   </parent>
// </tag>
// ----
//
func (p *TagService) Update() *TagServiceUpdateRequest {
	return &TagServiceUpdateRequest{TagService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *TagService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *TagService) String() string {
	return fmt.Sprintf("TagService:%s", op.path)
}

//
//
type SchedulingPolicyUnitService struct {
	BaseService
}

func NewSchedulingPolicyUnitService(connection *Connection, path string) *SchedulingPolicyUnitService {
	var result SchedulingPolicyUnitService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type SchedulingPolicyUnitServiceGetRequest struct {
	SchedulingPolicyUnitService *SchedulingPolicyUnitService
	header                      map[string]string
	query                       map[string]string
	filter                      *bool
	follow                      *string
}

func (p *SchedulingPolicyUnitServiceGetRequest) Header(key, value string) *SchedulingPolicyUnitServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPolicyUnitServiceGetRequest) Query(key, value string) *SchedulingPolicyUnitServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPolicyUnitServiceGetRequest) Filter(filter bool) *SchedulingPolicyUnitServiceGetRequest {
	p.filter = &filter
	return p
}

func (p *SchedulingPolicyUnitServiceGetRequest) Follow(follow string) *SchedulingPolicyUnitServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *SchedulingPolicyUnitServiceGetRequest) Send() (*SchedulingPolicyUnitServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPolicyUnitService.connection.URL(), p.SchedulingPolicyUnitService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPolicyUnitService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPolicyUnitService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPolicyUnitService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPolicyUnitService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPolicyUnitService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSchedulingPolicyUnitReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SchedulingPolicyUnitServiceGetResponse{unit: result}, nil
}

func (p *SchedulingPolicyUnitServiceGetRequest) MustSend() *SchedulingPolicyUnitServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SchedulingPolicyUnitServiceGetResponse struct {
	unit *SchedulingPolicyUnit
}

func (p *SchedulingPolicyUnitServiceGetResponse) Unit() (*SchedulingPolicyUnit, bool) {
	if p.unit != nil {
		return p.unit, true
	}
	return nil, false
}

func (p *SchedulingPolicyUnitServiceGetResponse) MustUnit() *SchedulingPolicyUnit {
	if p.unit == nil {
		panic("unit in response does not exist")
	}
	return p.unit
}

//
//
func (p *SchedulingPolicyUnitService) Get() *SchedulingPolicyUnitServiceGetRequest {
	return &SchedulingPolicyUnitServiceGetRequest{SchedulingPolicyUnitService: p}
}

//
//
type SchedulingPolicyUnitServiceRemoveRequest struct {
	SchedulingPolicyUnitService *SchedulingPolicyUnitService
	header                      map[string]string
	query                       map[string]string
	async                       *bool
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) Header(key, value string) *SchedulingPolicyUnitServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) Query(key, value string) *SchedulingPolicyUnitServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) Async(async bool) *SchedulingPolicyUnitServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) Send() (*SchedulingPolicyUnitServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SchedulingPolicyUnitService.connection.URL(), p.SchedulingPolicyUnitService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SchedulingPolicyUnitService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SchedulingPolicyUnitService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SchedulingPolicyUnitService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SchedulingPolicyUnitService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SchedulingPolicyUnitService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(SchedulingPolicyUnitServiceRemoveResponse), nil
}

func (p *SchedulingPolicyUnitServiceRemoveRequest) MustSend() *SchedulingPolicyUnitServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SchedulingPolicyUnitServiceRemoveResponse struct {
}

//
//
func (p *SchedulingPolicyUnitService) Remove() *SchedulingPolicyUnitServiceRemoveRequest {
	return &SchedulingPolicyUnitServiceRemoveRequest{SchedulingPolicyUnitService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SchedulingPolicyUnitService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *SchedulingPolicyUnitService) String() string {
	return fmt.Sprintf("SchedulingPolicyUnitService:%s", op.path)
}

//
//
type BalancesService struct {
	BaseService
}

func NewBalancesService(connection *Connection, path string) *BalancesService {
	var result BalancesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Add a balance module to a specified user defined scheduling policy.
//
type BalancesServiceAddRequest struct {
	BalancesService *BalancesService
	header          map[string]string
	query           map[string]string
	balance         *Balance
}

func (p *BalancesServiceAddRequest) Header(key, value string) *BalancesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BalancesServiceAddRequest) Query(key, value string) *BalancesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BalancesServiceAddRequest) Balance(balance *Balance) *BalancesServiceAddRequest {
	p.balance = balance
	return p
}

func (p *BalancesServiceAddRequest) Send() (*BalancesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BalancesService.connection.URL(), p.BalancesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLBalanceWriteOne(writer, p.balance, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BalancesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BalancesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BalancesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BalancesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BalancesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBalanceReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &BalancesServiceAddResponse{balance: result}, nil
}

func (p *BalancesServiceAddRequest) MustSend() *BalancesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a balance module to a specified user defined scheduling policy.
//
type BalancesServiceAddResponse struct {
	balance *Balance
}

func (p *BalancesServiceAddResponse) Balance() (*Balance, bool) {
	if p.balance != nil {
		return p.balance, true
	}
	return nil, false
}

func (p *BalancesServiceAddResponse) MustBalance() *Balance {
	if p.balance == nil {
		panic("balance in response does not exist")
	}
	return p.balance
}

//
// Add a balance module to a specified user defined scheduling policy.
//
func (p *BalancesService) Add() *BalancesServiceAddRequest {
	return &BalancesServiceAddRequest{BalancesService: p}
}

//
// Returns the list of balance modules used by the scheduling policy.
// The order of the returned balance modules isn't guaranteed.
//
type BalancesServiceListRequest struct {
	BalancesService *BalancesService
	header          map[string]string
	query           map[string]string
	filter          *bool
	follow          *string
	max             *int64
}

func (p *BalancesServiceListRequest) Header(key, value string) *BalancesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *BalancesServiceListRequest) Query(key, value string) *BalancesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *BalancesServiceListRequest) Filter(filter bool) *BalancesServiceListRequest {
	p.filter = &filter
	return p
}

func (p *BalancesServiceListRequest) Follow(follow string) *BalancesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *BalancesServiceListRequest) Max(max int64) *BalancesServiceListRequest {
	p.max = &max
	return p
}

func (p *BalancesServiceListRequest) Send() (*BalancesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.BalancesService.connection.URL(), p.BalancesService.path)
	values := make(url.Values)
	if p.filter != nil {
		values["filter"] = []string{fmt.Sprintf("%v", *p.filter)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.BalancesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.BalancesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.BalancesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.BalancesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.BalancesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLBalanceReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &BalancesServiceListResponse{balances: result}, nil
}

func (p *BalancesServiceListRequest) MustSend() *BalancesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of balance modules used by the scheduling policy.
// The order of the returned balance modules isn't guaranteed.
//
type BalancesServiceListResponse struct {
	balances *BalanceSlice
}

func (p *BalancesServiceListResponse) Balances() (*BalanceSlice, bool) {
	if p.balances != nil {
		return p.balances, true
	}
	return nil, false
}

func (p *BalancesServiceListResponse) MustBalances() *BalanceSlice {
	if p.balances == nil {
		panic("balances in response does not exist")
	}
	return p.balances
}

//
// Returns the list of balance modules used by the scheduling policy.
// The order of the returned balance modules isn't guaranteed.
//
func (p *BalancesService) List() *BalancesServiceListRequest {
	return &BalancesServiceListRequest{BalancesService: p}
}

//
//
func (op *BalancesService) BalanceService(id string) *BalanceService {
	return NewBalanceService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *BalancesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.BalanceService(path), nil
	}
	return op.BalanceService(path[:index]).Service(path[index+1:])
}

func (op *BalancesService) String() string {
	return fmt.Sprintf("BalancesService:%s", op.path)
}

//
// This service manages a single gluster brick.
//
type GlusterBrickService struct {
	BaseService
}

func NewGlusterBrickService(connection *Connection, path string) *GlusterBrickService {
	var result GlusterBrickService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get details of a brick.
// Retrieves status details of brick from underlying gluster volume with header `All-Content` set to `true`. This is
// the equivalent of running `gluster volume status <volumename> <brickname> detail`.
// For example, to get the details of brick `234` of gluster volume `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/234
// ----
// Which will return a response body like this:
// [source,xml]
// ----
// <brick id="234">
//   <name>host1:/rhgs/data/brick1</name>
//   <brick_dir>/rhgs/data/brick1</brick_dir>
//   <server_id>111</server_id>
//   <status>up</status>
//   <device>/dev/mapper/RHGS_vg1-lv_vmaddldisks</device>
//   <fs_name>xfs</fs_name>
//   <gluster_clients>
//     <gluster_client>
//       <bytes_read>2818417648</bytes_read>
//       <bytes_written>1384694844</bytes_written>
//       <client_port>1011</client_port>
//       <host_name>client2</host_name>
//     </gluster_client>
//   </gluster_clients>
//   <memory_pools>
//     <memory_pool>
//       <name>data-server:fd_t</name>
//       <alloc_count>1626348</alloc_count>
//       <cold_count>1020</cold_count>
//       <hot_count>4</hot_count>
//       <max_alloc>23</max_alloc>
//       <max_stdalloc>0</max_stdalloc>
//       <padded_size>140</padded_size>
//       <pool_misses>0</pool_misses>
//     </memory_pool>
//   </memory_pools>
//   <mnt_options>rw,seclabel,noatime,nodiratime,attr2,inode64,sunit=512,swidth=2048,noquota</mnt_options>
//   <pid>25589</pid>
//   <port>49155</port>
// </brick>
// ----
//
type GlusterBrickServiceGetRequest struct {
	GlusterBrickService *GlusterBrickService
	header              map[string]string
	query               map[string]string
	follow              *string
}

func (p *GlusterBrickServiceGetRequest) Header(key, value string) *GlusterBrickServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBrickServiceGetRequest) Query(key, value string) *GlusterBrickServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBrickServiceGetRequest) Follow(follow string) *GlusterBrickServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *GlusterBrickServiceGetRequest) Send() (*GlusterBrickServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterBrickService.connection.URL(), p.GlusterBrickService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBrickService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBrickService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBrickService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBrickService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBrickService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterBrickReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &GlusterBrickServiceGetResponse{brick: result}, nil
}

func (p *GlusterBrickServiceGetRequest) MustSend() *GlusterBrickServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get details of a brick.
// Retrieves status details of brick from underlying gluster volume with header `All-Content` set to `true`. This is
// the equivalent of running `gluster volume status <volumename> <brickname> detail`.
// For example, to get the details of brick `234` of gluster volume `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/234
// ----
// Which will return a response body like this:
// [source,xml]
// ----
// <brick id="234">
//   <name>host1:/rhgs/data/brick1</name>
//   <brick_dir>/rhgs/data/brick1</brick_dir>
//   <server_id>111</server_id>
//   <status>up</status>
//   <device>/dev/mapper/RHGS_vg1-lv_vmaddldisks</device>
//   <fs_name>xfs</fs_name>
//   <gluster_clients>
//     <gluster_client>
//       <bytes_read>2818417648</bytes_read>
//       <bytes_written>1384694844</bytes_written>
//       <client_port>1011</client_port>
//       <host_name>client2</host_name>
//     </gluster_client>
//   </gluster_clients>
//   <memory_pools>
//     <memory_pool>
//       <name>data-server:fd_t</name>
//       <alloc_count>1626348</alloc_count>
//       <cold_count>1020</cold_count>
//       <hot_count>4</hot_count>
//       <max_alloc>23</max_alloc>
//       <max_stdalloc>0</max_stdalloc>
//       <padded_size>140</padded_size>
//       <pool_misses>0</pool_misses>
//     </memory_pool>
//   </memory_pools>
//   <mnt_options>rw,seclabel,noatime,nodiratime,attr2,inode64,sunit=512,swidth=2048,noquota</mnt_options>
//   <pid>25589</pid>
//   <port>49155</port>
// </brick>
// ----
//
type GlusterBrickServiceGetResponse struct {
	brick *GlusterBrick
}

func (p *GlusterBrickServiceGetResponse) Brick() (*GlusterBrick, bool) {
	if p.brick != nil {
		return p.brick, true
	}
	return nil, false
}

func (p *GlusterBrickServiceGetResponse) MustBrick() *GlusterBrick {
	if p.brick == nil {
		panic("brick in response does not exist")
	}
	return p.brick
}

//
// Get details of a brick.
// Retrieves status details of brick from underlying gluster volume with header `All-Content` set to `true`. This is
// the equivalent of running `gluster volume status <volumename> <brickname> detail`.
// For example, to get the details of brick `234` of gluster volume `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/234
// ----
// Which will return a response body like this:
// [source,xml]
// ----
// <brick id="234">
//   <name>host1:/rhgs/data/brick1</name>
//   <brick_dir>/rhgs/data/brick1</brick_dir>
//   <server_id>111</server_id>
//   <status>up</status>
//   <device>/dev/mapper/RHGS_vg1-lv_vmaddldisks</device>
//   <fs_name>xfs</fs_name>
//   <gluster_clients>
//     <gluster_client>
//       <bytes_read>2818417648</bytes_read>
//       <bytes_written>1384694844</bytes_written>
//       <client_port>1011</client_port>
//       <host_name>client2</host_name>
//     </gluster_client>
//   </gluster_clients>
//   <memory_pools>
//     <memory_pool>
//       <name>data-server:fd_t</name>
//       <alloc_count>1626348</alloc_count>
//       <cold_count>1020</cold_count>
//       <hot_count>4</hot_count>
//       <max_alloc>23</max_alloc>
//       <max_stdalloc>0</max_stdalloc>
//       <padded_size>140</padded_size>
//       <pool_misses>0</pool_misses>
//     </memory_pool>
//   </memory_pools>
//   <mnt_options>rw,seclabel,noatime,nodiratime,attr2,inode64,sunit=512,swidth=2048,noquota</mnt_options>
//   <pid>25589</pid>
//   <port>49155</port>
// </brick>
// ----
//
func (p *GlusterBrickService) Get() *GlusterBrickServiceGetRequest {
	return &GlusterBrickServiceGetRequest{GlusterBrickService: p}
}

//
// Removes a brick.
// Removes a brick from the underlying gluster volume and deletes entries from database. This can be used only when
// removing a single brick without data migration. To remove multiple bricks and with data migration, use
// <<services/gluster_bricks/methods/migrate, migrate>> instead.
// For example, to delete brick `234` from gluster volume `123`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/234
// ----
//
type GlusterBrickServiceRemoveRequest struct {
	GlusterBrickService *GlusterBrickService
	header              map[string]string
	query               map[string]string
	async               *bool
}

func (p *GlusterBrickServiceRemoveRequest) Header(key, value string) *GlusterBrickServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBrickServiceRemoveRequest) Query(key, value string) *GlusterBrickServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBrickServiceRemoveRequest) Async(async bool) *GlusterBrickServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *GlusterBrickServiceRemoveRequest) Send() (*GlusterBrickServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterBrickService.connection.URL(), p.GlusterBrickService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBrickService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBrickService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBrickService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBrickService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBrickService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(GlusterBrickServiceRemoveResponse), nil
}

func (p *GlusterBrickServiceRemoveRequest) MustSend() *GlusterBrickServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a brick.
// Removes a brick from the underlying gluster volume and deletes entries from database. This can be used only when
// removing a single brick without data migration. To remove multiple bricks and with data migration, use
// <<services/gluster_bricks/methods/migrate, migrate>> instead.
// For example, to delete brick `234` from gluster volume `123`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/234
// ----
//
type GlusterBrickServiceRemoveResponse struct {
}

//
// Removes a brick.
// Removes a brick from the underlying gluster volume and deletes entries from database. This can be used only when
// removing a single brick without data migration. To remove multiple bricks and with data migration, use
// <<services/gluster_bricks/methods/migrate, migrate>> instead.
// For example, to delete brick `234` from gluster volume `123`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/234
// ----
//
func (p *GlusterBrickService) Remove() *GlusterBrickServiceRemoveRequest {
	return &GlusterBrickServiceRemoveRequest{GlusterBrickService: p}
}

//
// Replaces this brick with a new one.
// IMPORTANT: This operation has been deprecated since version 3.5 of the engine and will be removed in the future.
// Use <<services/gluster_bricks/methods/add, add brick(s)>> and
// <<services/gluster_bricks/methods/migrate, migrate brick(s)>> instead.
//
type GlusterBrickServiceReplaceRequest struct {
	GlusterBrickService *GlusterBrickService
	header              map[string]string
	query               map[string]string
	async               *bool
	force               *bool
}

func (p *GlusterBrickServiceReplaceRequest) Header(key, value string) *GlusterBrickServiceReplaceRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBrickServiceReplaceRequest) Query(key, value string) *GlusterBrickServiceReplaceRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBrickServiceReplaceRequest) Async(async bool) *GlusterBrickServiceReplaceRequest {
	p.async = &async
	return p
}

func (p *GlusterBrickServiceReplaceRequest) Force(force bool) *GlusterBrickServiceReplaceRequest {
	p.force = &force
	return p
}

func (p *GlusterBrickServiceReplaceRequest) Send() (*GlusterBrickServiceReplaceResponse, error) {
	rawURL := fmt.Sprintf("%s%s/replace", p.GlusterBrickService.connection.URL(), p.GlusterBrickService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBrickService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBrickService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBrickService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBrickService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBrickService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterBrickServiceReplaceResponse), nil
}

func (p *GlusterBrickServiceReplaceRequest) MustSend() *GlusterBrickServiceReplaceResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Replaces this brick with a new one.
// IMPORTANT: This operation has been deprecated since version 3.5 of the engine and will be removed in the future.
// Use <<services/gluster_bricks/methods/add, add brick(s)>> and
// <<services/gluster_bricks/methods/migrate, migrate brick(s)>> instead.
//
type GlusterBrickServiceReplaceResponse struct {
}

//
// Replaces this brick with a new one.
// IMPORTANT: This operation has been deprecated since version 3.5 of the engine and will be removed in the future.
// Use <<services/gluster_bricks/methods/add, add brick(s)>> and
// <<services/gluster_bricks/methods/migrate, migrate brick(s)>> instead.
//
func (p *GlusterBrickService) Replace() *GlusterBrickServiceReplaceRequest {
	return &GlusterBrickServiceReplaceRequest{GlusterBrickService: p}
}

//
//
func (op *GlusterBrickService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GlusterBrickService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *GlusterBrickService) String() string {
	return fmt.Sprintf("GlusterBrickService:%s", op.path)
}

//
// This service manages a collection of gluster volumes available in a cluster.
//
type GlusterVolumesService struct {
	BaseService
}

func NewGlusterVolumesService(connection *Connection, path string) *GlusterVolumesService {
	var result GlusterVolumesService
	result.connection = connection
	result.path = path
	return &result
}

//
// Creates a new gluster volume.
// The volume is created based on properties of the `volume` parameter. The properties `name`, `volume_type` and
// `bricks` are required.
// For example, to add a volume with name `myvolume` to the cluster `123`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/glustervolumes
// ----
// With the following request body:
// [source,xml]
// ----
// <gluster_volume>
//   <name>myvolume</name>
//   <volume_type>replicate</volume_type>
//   <replica_count>3</replica_count>
//   <bricks>
//     <brick>
//       <server_id>server1</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//     <brick>
//       <server_id>server2</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//     <brick>
//       <server_id>server3</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//   <bricks>
// </gluster_volume>
// ----
//
type GlusterVolumesServiceAddRequest struct {
	GlusterVolumesService *GlusterVolumesService
	header                map[string]string
	query                 map[string]string
	volume                *GlusterVolume
}

func (p *GlusterVolumesServiceAddRequest) Header(key, value string) *GlusterVolumesServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumesServiceAddRequest) Query(key, value string) *GlusterVolumesServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumesServiceAddRequest) Volume(volume *GlusterVolume) *GlusterVolumesServiceAddRequest {
	p.volume = volume
	return p
}

func (p *GlusterVolumesServiceAddRequest) Send() (*GlusterVolumesServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterVolumesService.connection.URL(), p.GlusterVolumesService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLGlusterVolumeWriteOne(writer, p.volume, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterVolumeReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &GlusterVolumesServiceAddResponse{volume: result}, nil
}

func (p *GlusterVolumesServiceAddRequest) MustSend() *GlusterVolumesServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Creates a new gluster volume.
// The volume is created based on properties of the `volume` parameter. The properties `name`, `volume_type` and
// `bricks` are required.
// For example, to add a volume with name `myvolume` to the cluster `123`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/glustervolumes
// ----
// With the following request body:
// [source,xml]
// ----
// <gluster_volume>
//   <name>myvolume</name>
//   <volume_type>replicate</volume_type>
//   <replica_count>3</replica_count>
//   <bricks>
//     <brick>
//       <server_id>server1</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//     <brick>
//       <server_id>server2</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//     <brick>
//       <server_id>server3</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//   <bricks>
// </gluster_volume>
// ----
//
type GlusterVolumesServiceAddResponse struct {
	volume *GlusterVolume
}

func (p *GlusterVolumesServiceAddResponse) Volume() (*GlusterVolume, bool) {
	if p.volume != nil {
		return p.volume, true
	}
	return nil, false
}

func (p *GlusterVolumesServiceAddResponse) MustVolume() *GlusterVolume {
	if p.volume == nil {
		panic("volume in response does not exist")
	}
	return p.volume
}

//
// Creates a new gluster volume.
// The volume is created based on properties of the `volume` parameter. The properties `name`, `volume_type` and
// `bricks` are required.
// For example, to add a volume with name `myvolume` to the cluster `123`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/clusters/123/glustervolumes
// ----
// With the following request body:
// [source,xml]
// ----
// <gluster_volume>
//   <name>myvolume</name>
//   <volume_type>replicate</volume_type>
//   <replica_count>3</replica_count>
//   <bricks>
//     <brick>
//       <server_id>server1</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//     <brick>
//       <server_id>server2</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//     <brick>
//       <server_id>server3</server_id>
//       <brick_dir>/exp1</brick_dir>
//     </brick>
//   <bricks>
// </gluster_volume>
// ----
//
func (p *GlusterVolumesService) Add() *GlusterVolumesServiceAddRequest {
	return &GlusterVolumesServiceAddRequest{GlusterVolumesService: p}
}

//
// Lists all gluster volumes in the cluster.
// For example, to list all Gluster Volumes in cluster `456`, send a request like
// this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/456/glustervolumes
// ----
// The order of the returned list of volumes isn't guaranteed.
//
type GlusterVolumesServiceListRequest struct {
	GlusterVolumesService *GlusterVolumesService
	header                map[string]string
	query                 map[string]string
	caseSensitive         *bool
	follow                *string
	max                   *int64
	search                *string
}

func (p *GlusterVolumesServiceListRequest) Header(key, value string) *GlusterVolumesServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumesServiceListRequest) Query(key, value string) *GlusterVolumesServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumesServiceListRequest) CaseSensitive(caseSensitive bool) *GlusterVolumesServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *GlusterVolumesServiceListRequest) Follow(follow string) *GlusterVolumesServiceListRequest {
	p.follow = &follow
	return p
}

func (p *GlusterVolumesServiceListRequest) Max(max int64) *GlusterVolumesServiceListRequest {
	p.max = &max
	return p
}

func (p *GlusterVolumesServiceListRequest) Search(search string) *GlusterVolumesServiceListRequest {
	p.search = &search
	return p
}

func (p *GlusterVolumesServiceListRequest) Send() (*GlusterVolumesServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterVolumesService.connection.URL(), p.GlusterVolumesService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumesService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumesService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumesService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumesService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumesService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterVolumeReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &GlusterVolumesServiceListResponse{volumes: result}, nil
}

func (p *GlusterVolumesServiceListRequest) MustSend() *GlusterVolumesServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists all gluster volumes in the cluster.
// For example, to list all Gluster Volumes in cluster `456`, send a request like
// this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/456/glustervolumes
// ----
// The order of the returned list of volumes isn't guaranteed.
//
type GlusterVolumesServiceListResponse struct {
	volumes *GlusterVolumeSlice
}

func (p *GlusterVolumesServiceListResponse) Volumes() (*GlusterVolumeSlice, bool) {
	if p.volumes != nil {
		return p.volumes, true
	}
	return nil, false
}

func (p *GlusterVolumesServiceListResponse) MustVolumes() *GlusterVolumeSlice {
	if p.volumes == nil {
		panic("volumes in response does not exist")
	}
	return p.volumes
}

//
// Lists all gluster volumes in the cluster.
// For example, to list all Gluster Volumes in cluster `456`, send a request like
// this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/456/glustervolumes
// ----
// The order of the returned list of volumes isn't guaranteed.
//
func (p *GlusterVolumesService) List() *GlusterVolumesServiceListRequest {
	return &GlusterVolumesServiceListRequest{GlusterVolumesService: p}
}

//
// Reference to a service managing gluster volume.
//
func (op *GlusterVolumesService) VolumeService(id string) *GlusterVolumeService {
	return NewGlusterVolumeService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GlusterVolumesService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.VolumeService(path), nil
	}
	return op.VolumeService(path[:index]).Service(path[index+1:])
}

func (op *GlusterVolumesService) String() string {
	return fmt.Sprintf("GlusterVolumesService:%s", op.path)
}

//
//
type GlusterHooksService struct {
	BaseService
}

func NewGlusterHooksService(connection *Connection, path string) *GlusterHooksService {
	var result GlusterHooksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Returns the list of hooks.
// The order of the returned list of hooks isn't guaranteed.
//
type GlusterHooksServiceListRequest struct {
	GlusterHooksService *GlusterHooksService
	header              map[string]string
	query               map[string]string
	follow              *string
	max                 *int64
}

func (p *GlusterHooksServiceListRequest) Header(key, value string) *GlusterHooksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterHooksServiceListRequest) Query(key, value string) *GlusterHooksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterHooksServiceListRequest) Follow(follow string) *GlusterHooksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *GlusterHooksServiceListRequest) Max(max int64) *GlusterHooksServiceListRequest {
	p.max = &max
	return p
}

func (p *GlusterHooksServiceListRequest) Send() (*GlusterHooksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterHooksService.connection.URL(), p.GlusterHooksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterHooksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterHooksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterHooksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterHooksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterHooksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterHookReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &GlusterHooksServiceListResponse{hooks: result}, nil
}

func (p *GlusterHooksServiceListRequest) MustSend() *GlusterHooksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Returns the list of hooks.
// The order of the returned list of hooks isn't guaranteed.
//
type GlusterHooksServiceListResponse struct {
	hooks *GlusterHookSlice
}

func (p *GlusterHooksServiceListResponse) Hooks() (*GlusterHookSlice, bool) {
	if p.hooks != nil {
		return p.hooks, true
	}
	return nil, false
}

func (p *GlusterHooksServiceListResponse) MustHooks() *GlusterHookSlice {
	if p.hooks == nil {
		panic("hooks in response does not exist")
	}
	return p.hooks
}

//
// Returns the list of hooks.
// The order of the returned list of hooks isn't guaranteed.
//
func (p *GlusterHooksService) List() *GlusterHooksServiceListRequest {
	return &GlusterHooksServiceListRequest{GlusterHooksService: p}
}

//
//
func (op *GlusterHooksService) HookService(id string) *GlusterHookService {
	return NewGlusterHookService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GlusterHooksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.HookService(path), nil
	}
	return op.HookService(path[:index]).Service(path[index+1:])
}

func (op *GlusterHooksService) String() string {
	return fmt.Sprintf("GlusterHooksService:%s", op.path)
}

//
// This service manages the gluster bricks in a gluster volume
//
type GlusterBricksService struct {
	BaseService
}

func NewGlusterBricksService(connection *Connection, path string) *GlusterBricksService {
	var result GlusterBricksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Activate the bricks post data migration of remove brick operation.
// Used to activate brick(s) once the data migration from bricks is complete but user no longer wishes to remove
// bricks. The bricks that were previously marked for removal will now be used as normal bricks.
// For example, to retain the bricks that on glustervolume `123` from which data was migrated, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/activate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <bricks>
//     <brick>
//       <name>host1:/rhgs/brick1</name>
//     </brick>
//   </bricks>
// </action>
// ----
//
type GlusterBricksServiceActivateRequest struct {
	GlusterBricksService *GlusterBricksService
	header               map[string]string
	query                map[string]string
	async                *bool
	bricks               *GlusterBrickSlice
}

func (p *GlusterBricksServiceActivateRequest) Header(key, value string) *GlusterBricksServiceActivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBricksServiceActivateRequest) Query(key, value string) *GlusterBricksServiceActivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBricksServiceActivateRequest) Async(async bool) *GlusterBricksServiceActivateRequest {
	p.async = &async
	return p
}

func (p *GlusterBricksServiceActivateRequest) Bricks(bricks *GlusterBrickSlice) *GlusterBricksServiceActivateRequest {
	p.bricks = bricks
	return p
}

func (p *GlusterBricksServiceActivateRequest) BricksOfAny(anys ...*GlusterBrick) *GlusterBricksServiceActivateRequest {
	if p.bricks == nil {
		p.bricks = new(GlusterBrickSlice)
	}
	p.bricks.slice = append(p.bricks.slice, anys...)
	return p
}

func (p *GlusterBricksServiceActivateRequest) Send() (*GlusterBricksServiceActivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/activate", p.GlusterBricksService.connection.URL(), p.GlusterBricksService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Bricks(p.bricks)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBricksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBricksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBricksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBricksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBricksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterBricksServiceActivateResponse), nil
}

func (p *GlusterBricksServiceActivateRequest) MustSend() *GlusterBricksServiceActivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Activate the bricks post data migration of remove brick operation.
// Used to activate brick(s) once the data migration from bricks is complete but user no longer wishes to remove
// bricks. The bricks that were previously marked for removal will now be used as normal bricks.
// For example, to retain the bricks that on glustervolume `123` from which data was migrated, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/activate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <bricks>
//     <brick>
//       <name>host1:/rhgs/brick1</name>
//     </brick>
//   </bricks>
// </action>
// ----
//
type GlusterBricksServiceActivateResponse struct {
}

//
// Activate the bricks post data migration of remove brick operation.
// Used to activate brick(s) once the data migration from bricks is complete but user no longer wishes to remove
// bricks. The bricks that were previously marked for removal will now be used as normal bricks.
// For example, to retain the bricks that on glustervolume `123` from which data was migrated, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/activate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <bricks>
//     <brick>
//       <name>host1:/rhgs/brick1</name>
//     </brick>
//   </bricks>
// </action>
// ----
//
func (p *GlusterBricksService) Activate() *GlusterBricksServiceActivateRequest {
	return &GlusterBricksServiceActivateRequest{GlusterBricksService: p}
}

//
// Adds a list of bricks to gluster volume.
// Used to expand a gluster volume by adding bricks. For replicated volume types, the parameter `replica_count`
// needs to be passed. In case the replica count is being increased, then the number of bricks needs to be
// equivalent to the number of replica sets.
// For example, to add bricks to gluster volume `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <server_id>111</server_id>
//     <brick_dir>/export/data/brick3</brick_dir>
//   </brick>
// </bricks>
// ----
//
type GlusterBricksServiceAddRequest struct {
	GlusterBricksService *GlusterBricksService
	header               map[string]string
	query                map[string]string
	bricks               *GlusterBrickSlice
	replicaCount         *int64
	stripeCount          *int64
}

func (p *GlusterBricksServiceAddRequest) Header(key, value string) *GlusterBricksServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBricksServiceAddRequest) Query(key, value string) *GlusterBricksServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBricksServiceAddRequest) Bricks(bricks *GlusterBrickSlice) *GlusterBricksServiceAddRequest {
	p.bricks = bricks
	return p
}

func (p *GlusterBricksServiceAddRequest) BricksOfAny(anys ...*GlusterBrick) *GlusterBricksServiceAddRequest {
	if p.bricks == nil {
		p.bricks = new(GlusterBrickSlice)
	}
	p.bricks.slice = append(p.bricks.slice, anys...)
	return p
}

func (p *GlusterBricksServiceAddRequest) ReplicaCount(replicaCount int64) *GlusterBricksServiceAddRequest {
	p.replicaCount = &replicaCount
	return p
}

func (p *GlusterBricksServiceAddRequest) StripeCount(stripeCount int64) *GlusterBricksServiceAddRequest {
	p.stripeCount = &stripeCount
	return p
}

func (p *GlusterBricksServiceAddRequest) Send() (*GlusterBricksServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterBricksService.connection.URL(), p.GlusterBricksService.path)
	values := make(url.Values)
	if p.replicaCount != nil {
		values["replica_count"] = []string{fmt.Sprintf("%v", *p.replicaCount)}
	}

	if p.stripeCount != nil {
		values["stripe_count"] = []string{fmt.Sprintf("%v", *p.stripeCount)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLGlusterBrickWriteMany(writer, p.bricks, "", "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBricksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBricksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBricksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBricksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBricksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterBrickReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &GlusterBricksServiceAddResponse{bricks: result}, nil
}

func (p *GlusterBricksServiceAddRequest) MustSend() *GlusterBricksServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a list of bricks to gluster volume.
// Used to expand a gluster volume by adding bricks. For replicated volume types, the parameter `replica_count`
// needs to be passed. In case the replica count is being increased, then the number of bricks needs to be
// equivalent to the number of replica sets.
// For example, to add bricks to gluster volume `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <server_id>111</server_id>
//     <brick_dir>/export/data/brick3</brick_dir>
//   </brick>
// </bricks>
// ----
//
type GlusterBricksServiceAddResponse struct {
	bricks *GlusterBrickSlice
}

func (p *GlusterBricksServiceAddResponse) Bricks() (*GlusterBrickSlice, bool) {
	if p.bricks != nil {
		return p.bricks, true
	}
	return nil, false
}

func (p *GlusterBricksServiceAddResponse) MustBricks() *GlusterBrickSlice {
	if p.bricks == nil {
		panic("bricks in response does not exist")
	}
	return p.bricks
}

//
// Adds a list of bricks to gluster volume.
// Used to expand a gluster volume by adding bricks. For replicated volume types, the parameter `replica_count`
// needs to be passed. In case the replica count is being increased, then the number of bricks needs to be
// equivalent to the number of replica sets.
// For example, to add bricks to gluster volume `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <server_id>111</server_id>
//     <brick_dir>/export/data/brick3</brick_dir>
//   </brick>
// </bricks>
// ----
//
func (p *GlusterBricksService) Add() *GlusterBricksServiceAddRequest {
	return &GlusterBricksServiceAddRequest{GlusterBricksService: p}
}

//
// Lists the bricks of a gluster volume.
// For example, to list bricks of gluster volume `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// Provides an output as below:
// [source,xml]
// ----
// <bricks>
//   <brick id="234">
//     <name>host1:/rhgs/data/brick1</name>
//     <brick_dir>/rhgs/data/brick1</brick_dir>
//     <server_id>111</server_id>
//     <status>up</status>
//   </brick>
//   <brick id="233">
//     <name>host2:/rhgs/data/brick1</name>
//     <brick_dir>/rhgs/data/brick1</brick_dir>
//     <server_id>222</server_id>
//     <status>up</status>
//   </brick>
// </bricks>
// ----
// The order of the returned list is based on the brick order provided at gluster volume creation.
//
type GlusterBricksServiceListRequest struct {
	GlusterBricksService *GlusterBricksService
	header               map[string]string
	query                map[string]string
	follow               *string
	max                  *int64
}

func (p *GlusterBricksServiceListRequest) Header(key, value string) *GlusterBricksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBricksServiceListRequest) Query(key, value string) *GlusterBricksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBricksServiceListRequest) Follow(follow string) *GlusterBricksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *GlusterBricksServiceListRequest) Max(max int64) *GlusterBricksServiceListRequest {
	p.max = &max
	return p
}

func (p *GlusterBricksServiceListRequest) Send() (*GlusterBricksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterBricksService.connection.URL(), p.GlusterBricksService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBricksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBricksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBricksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBricksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBricksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterBrickReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &GlusterBricksServiceListResponse{bricks: result}, nil
}

func (p *GlusterBricksServiceListRequest) MustSend() *GlusterBricksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Lists the bricks of a gluster volume.
// For example, to list bricks of gluster volume `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// Provides an output as below:
// [source,xml]
// ----
// <bricks>
//   <brick id="234">
//     <name>host1:/rhgs/data/brick1</name>
//     <brick_dir>/rhgs/data/brick1</brick_dir>
//     <server_id>111</server_id>
//     <status>up</status>
//   </brick>
//   <brick id="233">
//     <name>host2:/rhgs/data/brick1</name>
//     <brick_dir>/rhgs/data/brick1</brick_dir>
//     <server_id>222</server_id>
//     <status>up</status>
//   </brick>
// </bricks>
// ----
// The order of the returned list is based on the brick order provided at gluster volume creation.
//
type GlusterBricksServiceListResponse struct {
	bricks *GlusterBrickSlice
}

func (p *GlusterBricksServiceListResponse) Bricks() (*GlusterBrickSlice, bool) {
	if p.bricks != nil {
		return p.bricks, true
	}
	return nil, false
}

func (p *GlusterBricksServiceListResponse) MustBricks() *GlusterBrickSlice {
	if p.bricks == nil {
		panic("bricks in response does not exist")
	}
	return p.bricks
}

//
// Lists the bricks of a gluster volume.
// For example, to list bricks of gluster volume `123`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// Provides an output as below:
// [source,xml]
// ----
// <bricks>
//   <brick id="234">
//     <name>host1:/rhgs/data/brick1</name>
//     <brick_dir>/rhgs/data/brick1</brick_dir>
//     <server_id>111</server_id>
//     <status>up</status>
//   </brick>
//   <brick id="233">
//     <name>host2:/rhgs/data/brick1</name>
//     <brick_dir>/rhgs/data/brick1</brick_dir>
//     <server_id>222</server_id>
//     <status>up</status>
//   </brick>
// </bricks>
// ----
// The order of the returned list is based on the brick order provided at gluster volume creation.
//
func (p *GlusterBricksService) List() *GlusterBricksServiceListRequest {
	return &GlusterBricksServiceListRequest{GlusterBricksService: p}
}

//
// Start migration of data prior to removing bricks.
// Removing bricks is a two-step process, where the data on bricks to be removed, is first migrated to remaining
// bricks. Once migration is completed the removal of bricks is confirmed via the API
// <<services/gluster_bricks/methods/remove, remove>>. If at any point, the action needs to be cancelled
// <<services/gluster_bricks/methods/stop_migrate, stopmigrate>> has to be called.
// For instance, to delete a brick from a gluster volume with id `123`, send a request:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/migrate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <bricks>
//     <brick>
//       <name>host1:/rhgs/brick1</name>
//     </brick>
//   </bricks>
// </action>
// ----
// The migration process can be tracked from the job id returned from the API using
// <<services/job/methods/get, job>> and steps in job using <<services/step/methods/get, step>>
//
type GlusterBricksServiceMigrateRequest struct {
	GlusterBricksService *GlusterBricksService
	header               map[string]string
	query                map[string]string
	async                *bool
	bricks               *GlusterBrickSlice
}

func (p *GlusterBricksServiceMigrateRequest) Header(key, value string) *GlusterBricksServiceMigrateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBricksServiceMigrateRequest) Query(key, value string) *GlusterBricksServiceMigrateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBricksServiceMigrateRequest) Async(async bool) *GlusterBricksServiceMigrateRequest {
	p.async = &async
	return p
}

func (p *GlusterBricksServiceMigrateRequest) Bricks(bricks *GlusterBrickSlice) *GlusterBricksServiceMigrateRequest {
	p.bricks = bricks
	return p
}

func (p *GlusterBricksServiceMigrateRequest) BricksOfAny(anys ...*GlusterBrick) *GlusterBricksServiceMigrateRequest {
	if p.bricks == nil {
		p.bricks = new(GlusterBrickSlice)
	}
	p.bricks.slice = append(p.bricks.slice, anys...)
	return p
}

func (p *GlusterBricksServiceMigrateRequest) Send() (*GlusterBricksServiceMigrateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/migrate", p.GlusterBricksService.connection.URL(), p.GlusterBricksService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Bricks(p.bricks)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBricksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBricksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBricksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBricksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBricksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterBricksServiceMigrateResponse), nil
}

func (p *GlusterBricksServiceMigrateRequest) MustSend() *GlusterBricksServiceMigrateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Start migration of data prior to removing bricks.
// Removing bricks is a two-step process, where the data on bricks to be removed, is first migrated to remaining
// bricks. Once migration is completed the removal of bricks is confirmed via the API
// <<services/gluster_bricks/methods/remove, remove>>. If at any point, the action needs to be cancelled
// <<services/gluster_bricks/methods/stop_migrate, stopmigrate>> has to be called.
// For instance, to delete a brick from a gluster volume with id `123`, send a request:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/migrate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <bricks>
//     <brick>
//       <name>host1:/rhgs/brick1</name>
//     </brick>
//   </bricks>
// </action>
// ----
// The migration process can be tracked from the job id returned from the API using
// <<services/job/methods/get, job>> and steps in job using <<services/step/methods/get, step>>
//
type GlusterBricksServiceMigrateResponse struct {
}

//
// Start migration of data prior to removing bricks.
// Removing bricks is a two-step process, where the data on bricks to be removed, is first migrated to remaining
// bricks. Once migration is completed the removal of bricks is confirmed via the API
// <<services/gluster_bricks/methods/remove, remove>>. If at any point, the action needs to be cancelled
// <<services/gluster_bricks/methods/stop_migrate, stopmigrate>> has to be called.
// For instance, to delete a brick from a gluster volume with id `123`, send a request:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/migrate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action>
//   <bricks>
//     <brick>
//       <name>host1:/rhgs/brick1</name>
//     </brick>
//   </bricks>
// </action>
// ----
// The migration process can be tracked from the job id returned from the API using
// <<services/job/methods/get, job>> and steps in job using <<services/step/methods/get, step>>
//
func (p *GlusterBricksService) Migrate() *GlusterBricksServiceMigrateRequest {
	return &GlusterBricksServiceMigrateRequest{GlusterBricksService: p}
}

//
// Removes bricks from gluster volume.
// The recommended way to remove bricks without data loss is to first migrate the data using
// <<services/gluster_bricks/methods/stop_migrate, stopmigrate>> and then removing them. If migrate was not called on
// bricks prior to remove, the bricks are removed without data migration which may lead to data loss.
// For example, to delete the bricks from gluster volume `123`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <name>host:brick_directory</name>
//   </brick>
// </bricks>
// ----
//
type GlusterBricksServiceRemoveRequest struct {
	GlusterBricksService *GlusterBricksService
	header               map[string]string
	query                map[string]string
	async                *bool
	bricks               *GlusterBrickSlice
	replicaCount         *int64
}

func (p *GlusterBricksServiceRemoveRequest) Header(key, value string) *GlusterBricksServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBricksServiceRemoveRequest) Query(key, value string) *GlusterBricksServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBricksServiceRemoveRequest) Async(async bool) *GlusterBricksServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *GlusterBricksServiceRemoveRequest) Bricks(bricks *GlusterBrickSlice) *GlusterBricksServiceRemoveRequest {
	p.bricks = bricks
	return p
}

func (p *GlusterBricksServiceRemoveRequest) BricksOfAny(anys ...*GlusterBrick) *GlusterBricksServiceRemoveRequest {
	if p.bricks == nil {
		p.bricks = new(GlusterBrickSlice)
	}
	p.bricks.slice = append(p.bricks.slice, anys...)
	return p
}

func (p *GlusterBricksServiceRemoveRequest) ReplicaCount(replicaCount int64) *GlusterBricksServiceRemoveRequest {
	p.replicaCount = &replicaCount
	return p
}

func (p *GlusterBricksServiceRemoveRequest) Send() (*GlusterBricksServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterBricksService.connection.URL(), p.GlusterBricksService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.replicaCount != nil {
		values["replica_count"] = []string{fmt.Sprintf("%v", *p.replicaCount)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBricksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBricksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBricksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBricksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBricksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(GlusterBricksServiceRemoveResponse), nil
}

func (p *GlusterBricksServiceRemoveRequest) MustSend() *GlusterBricksServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes bricks from gluster volume.
// The recommended way to remove bricks without data loss is to first migrate the data using
// <<services/gluster_bricks/methods/stop_migrate, stopmigrate>> and then removing them. If migrate was not called on
// bricks prior to remove, the bricks are removed without data migration which may lead to data loss.
// For example, to delete the bricks from gluster volume `123`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <name>host:brick_directory</name>
//   </brick>
// </bricks>
// ----
//
type GlusterBricksServiceRemoveResponse struct {
}

//
// Removes bricks from gluster volume.
// The recommended way to remove bricks without data loss is to first migrate the data using
// <<services/gluster_bricks/methods/stop_migrate, stopmigrate>> and then removing them. If migrate was not called on
// bricks prior to remove, the bricks are removed without data migration which may lead to data loss.
// For example, to delete the bricks from gluster volume `123`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <name>host:brick_directory</name>
//   </brick>
// </bricks>
// ----
//
func (p *GlusterBricksService) Remove() *GlusterBricksServiceRemoveRequest {
	return &GlusterBricksServiceRemoveRequest{GlusterBricksService: p}
}

//
// Stops migration of data from bricks for a remove brick operation.
// To cancel data migration that was started as part of the 2-step remove brick process in case the user wishes to
// continue using the bricks. The bricks that were marked for removal will function as normal bricks post this
// operation.
// For example, to stop migration of data from the bricks of gluster volume `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/stopmigrate
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <name>host:brick_directory</name>
//   </brick>
// </bricks>
// ----
//
type GlusterBricksServiceStopMigrateRequest struct {
	GlusterBricksService *GlusterBricksService
	header               map[string]string
	query                map[string]string
	async                *bool
	bricks               *GlusterBrickSlice
}

func (p *GlusterBricksServiceStopMigrateRequest) Header(key, value string) *GlusterBricksServiceStopMigrateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterBricksServiceStopMigrateRequest) Query(key, value string) *GlusterBricksServiceStopMigrateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterBricksServiceStopMigrateRequest) Async(async bool) *GlusterBricksServiceStopMigrateRequest {
	p.async = &async
	return p
}

func (p *GlusterBricksServiceStopMigrateRequest) Bricks(bricks *GlusterBrickSlice) *GlusterBricksServiceStopMigrateRequest {
	p.bricks = bricks
	return p
}

func (p *GlusterBricksServiceStopMigrateRequest) BricksOfAny(anys ...*GlusterBrick) *GlusterBricksServiceStopMigrateRequest {
	if p.bricks == nil {
		p.bricks = new(GlusterBrickSlice)
	}
	p.bricks.slice = append(p.bricks.slice, anys...)
	return p
}

func (p *GlusterBricksServiceStopMigrateRequest) Send() (*GlusterBricksServiceStopMigrateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/stopmigrate", p.GlusterBricksService.connection.URL(), p.GlusterBricksService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Bricks(p.bricks)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterBricksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterBricksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterBricksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterBricksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterBricksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterBricksServiceStopMigrateResponse), nil
}

func (p *GlusterBricksServiceStopMigrateRequest) MustSend() *GlusterBricksServiceStopMigrateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Stops migration of data from bricks for a remove brick operation.
// To cancel data migration that was started as part of the 2-step remove brick process in case the user wishes to
// continue using the bricks. The bricks that were marked for removal will function as normal bricks post this
// operation.
// For example, to stop migration of data from the bricks of gluster volume `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/stopmigrate
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <name>host:brick_directory</name>
//   </brick>
// </bricks>
// ----
//
type GlusterBricksServiceStopMigrateResponse struct {
}

//
// Stops migration of data from bricks for a remove brick operation.
// To cancel data migration that was started as part of the 2-step remove brick process in case the user wishes to
// continue using the bricks. The bricks that were marked for removal will function as normal bricks post this
// operation.
// For example, to stop migration of data from the bricks of gluster volume `123`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/567/glustervolumes/123/glusterbricks/stopmigrate
// ----
// With a request body like this:
// [source,xml]
// ----
// <bricks>
//   <brick>
//     <name>host:brick_directory</name>
//   </brick>
// </bricks>
// ----
//
func (p *GlusterBricksService) StopMigrate() *GlusterBricksServiceStopMigrateRequest {
	return &GlusterBricksServiceStopMigrateRequest{GlusterBricksService: p}
}

//
// Returns a reference to the service managing a single gluster brick.
//
func (op *GlusterBricksService) BrickService(id string) *GlusterBrickService {
	return NewGlusterBrickService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GlusterBricksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.BrickService(path), nil
	}
	return op.BrickService(path[:index]).Service(path[index+1:])
}

func (op *GlusterBricksService) String() string {
	return fmt.Sprintf("GlusterBricksService:%s", op.path)
}

//
//
type GlusterHookService struct {
	BaseService
}

func NewGlusterHookService(connection *Connection, path string) *GlusterHookService {
	var result GlusterHookService
	result.connection = connection
	result.path = path
	return &result
}

//
// Resolves status conflict of hook among servers in cluster by disabling Gluster hook in all servers of the
// cluster. This updates the hook status to `DISABLED` in database.
//
type GlusterHookServiceDisableRequest struct {
	GlusterHookService *GlusterHookService
	header             map[string]string
	query              map[string]string
	async              *bool
}

func (p *GlusterHookServiceDisableRequest) Header(key, value string) *GlusterHookServiceDisableRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterHookServiceDisableRequest) Query(key, value string) *GlusterHookServiceDisableRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterHookServiceDisableRequest) Async(async bool) *GlusterHookServiceDisableRequest {
	p.async = &async
	return p
}

func (p *GlusterHookServiceDisableRequest) Send() (*GlusterHookServiceDisableResponse, error) {
	rawURL := fmt.Sprintf("%s%s/disable", p.GlusterHookService.connection.URL(), p.GlusterHookService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterHookService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterHookService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterHookService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterHookService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterHookService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterHookServiceDisableResponse), nil
}

func (p *GlusterHookServiceDisableRequest) MustSend() *GlusterHookServiceDisableResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Resolves status conflict of hook among servers in cluster by disabling Gluster hook in all servers of the
// cluster. This updates the hook status to `DISABLED` in database.
//
type GlusterHookServiceDisableResponse struct {
}

//
// Resolves status conflict of hook among servers in cluster by disabling Gluster hook in all servers of the
// cluster. This updates the hook status to `DISABLED` in database.
//
func (p *GlusterHookService) Disable() *GlusterHookServiceDisableRequest {
	return &GlusterHookServiceDisableRequest{GlusterHookService: p}
}

//
// Resolves status conflict of hook among servers in cluster by disabling Gluster hook in all servers of the
// cluster. This updates the hook status to `DISABLED` in database.
//
type GlusterHookServiceEnableRequest struct {
	GlusterHookService *GlusterHookService
	header             map[string]string
	query              map[string]string
	async              *bool
}

func (p *GlusterHookServiceEnableRequest) Header(key, value string) *GlusterHookServiceEnableRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterHookServiceEnableRequest) Query(key, value string) *GlusterHookServiceEnableRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterHookServiceEnableRequest) Async(async bool) *GlusterHookServiceEnableRequest {
	p.async = &async
	return p
}

func (p *GlusterHookServiceEnableRequest) Send() (*GlusterHookServiceEnableResponse, error) {
	rawURL := fmt.Sprintf("%s%s/enable", p.GlusterHookService.connection.URL(), p.GlusterHookService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterHookService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterHookService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterHookService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterHookService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterHookService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterHookServiceEnableResponse), nil
}

func (p *GlusterHookServiceEnableRequest) MustSend() *GlusterHookServiceEnableResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Resolves status conflict of hook among servers in cluster by disabling Gluster hook in all servers of the
// cluster. This updates the hook status to `DISABLED` in database.
//
type GlusterHookServiceEnableResponse struct {
}

//
// Resolves status conflict of hook among servers in cluster by disabling Gluster hook in all servers of the
// cluster. This updates the hook status to `DISABLED` in database.
//
func (p *GlusterHookService) Enable() *GlusterHookServiceEnableRequest {
	return &GlusterHookServiceEnableRequest{GlusterHookService: p}
}

//
//
type GlusterHookServiceGetRequest struct {
	GlusterHookService *GlusterHookService
	header             map[string]string
	query              map[string]string
	follow             *string
}

func (p *GlusterHookServiceGetRequest) Header(key, value string) *GlusterHookServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterHookServiceGetRequest) Query(key, value string) *GlusterHookServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterHookServiceGetRequest) Follow(follow string) *GlusterHookServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *GlusterHookServiceGetRequest) Send() (*GlusterHookServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterHookService.connection.URL(), p.GlusterHookService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterHookService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterHookService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterHookService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterHookService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterHookService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterHookReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &GlusterHookServiceGetResponse{hook: result}, nil
}

func (p *GlusterHookServiceGetRequest) MustSend() *GlusterHookServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type GlusterHookServiceGetResponse struct {
	hook *GlusterHook
}

func (p *GlusterHookServiceGetResponse) Hook() (*GlusterHook, bool) {
	if p.hook != nil {
		return p.hook, true
	}
	return nil, false
}

func (p *GlusterHookServiceGetResponse) MustHook() *GlusterHook {
	if p.hook == nil {
		panic("hook in response does not exist")
	}
	return p.hook
}

//
//
func (p *GlusterHookService) Get() *GlusterHookServiceGetRequest {
	return &GlusterHookServiceGetRequest{GlusterHookService: p}
}

//
// Removes the this Gluster hook from all servers in cluster and deletes it from the database.
//
type GlusterHookServiceRemoveRequest struct {
	GlusterHookService *GlusterHookService
	header             map[string]string
	query              map[string]string
	async              *bool
}

func (p *GlusterHookServiceRemoveRequest) Header(key, value string) *GlusterHookServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterHookServiceRemoveRequest) Query(key, value string) *GlusterHookServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterHookServiceRemoveRequest) Async(async bool) *GlusterHookServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *GlusterHookServiceRemoveRequest) Send() (*GlusterHookServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterHookService.connection.URL(), p.GlusterHookService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterHookService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterHookService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterHookService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterHookService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterHookService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(GlusterHookServiceRemoveResponse), nil
}

func (p *GlusterHookServiceRemoveRequest) MustSend() *GlusterHookServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the this Gluster hook from all servers in cluster and deletes it from the database.
//
type GlusterHookServiceRemoveResponse struct {
}

//
// Removes the this Gluster hook from all servers in cluster and deletes it from the database.
//
func (p *GlusterHookService) Remove() *GlusterHookServiceRemoveRequest {
	return &GlusterHookServiceRemoveRequest{GlusterHookService: p}
}

//
// Resolves missing hook conflict depending on the resolution type.
// For `ADD` resolves by copying hook stored in engine database to all servers where the hook is missing. The
// engine maintains a list of all servers where hook is missing.
// For `COPY` resolves conflict in hook content by copying hook stored in engine database to all servers where
// the hook is missing. The engine maintains a list of all servers where the content is conflicting. If a host
// id is passed as parameter, the hook content from the server is used as the master to copy to other servers
// in cluster.
//
type GlusterHookServiceResolveRequest struct {
	GlusterHookService *GlusterHookService
	header             map[string]string
	query              map[string]string
	async              *bool
	host               *Host
	resolutionType     *string
}

func (p *GlusterHookServiceResolveRequest) Header(key, value string) *GlusterHookServiceResolveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterHookServiceResolveRequest) Query(key, value string) *GlusterHookServiceResolveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterHookServiceResolveRequest) Async(async bool) *GlusterHookServiceResolveRequest {
	p.async = &async
	return p
}

func (p *GlusterHookServiceResolveRequest) Host(host *Host) *GlusterHookServiceResolveRequest {
	p.host = host
	return p
}

func (p *GlusterHookServiceResolveRequest) ResolutionType(resolutionType string) *GlusterHookServiceResolveRequest {
	p.resolutionType = &resolutionType
	return p
}

func (p *GlusterHookServiceResolveRequest) Send() (*GlusterHookServiceResolveResponse, error) {
	rawURL := fmt.Sprintf("%s%s/resolve", p.GlusterHookService.connection.URL(), p.GlusterHookService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Host(p.host)
	if p.resolutionType != nil {
		actionBuilder.ResolutionType(*p.resolutionType)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterHookService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterHookService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterHookService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterHookService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterHookService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterHookServiceResolveResponse), nil
}

func (p *GlusterHookServiceResolveRequest) MustSend() *GlusterHookServiceResolveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Resolves missing hook conflict depending on the resolution type.
// For `ADD` resolves by copying hook stored in engine database to all servers where the hook is missing. The
// engine maintains a list of all servers where hook is missing.
// For `COPY` resolves conflict in hook content by copying hook stored in engine database to all servers where
// the hook is missing. The engine maintains a list of all servers where the content is conflicting. If a host
// id is passed as parameter, the hook content from the server is used as the master to copy to other servers
// in cluster.
//
type GlusterHookServiceResolveResponse struct {
}

//
// Resolves missing hook conflict depending on the resolution type.
// For `ADD` resolves by copying hook stored in engine database to all servers where the hook is missing. The
// engine maintains a list of all servers where hook is missing.
// For `COPY` resolves conflict in hook content by copying hook stored in engine database to all servers where
// the hook is missing. The engine maintains a list of all servers where the content is conflicting. If a host
// id is passed as parameter, the hook content from the server is used as the master to copy to other servers
// in cluster.
//
func (p *GlusterHookService) Resolve() *GlusterHookServiceResolveRequest {
	return &GlusterHookServiceResolveRequest{GlusterHookService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GlusterHookService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *GlusterHookService) String() string {
	return fmt.Sprintf("GlusterHookService:%s", op.path)
}

//
// This service manages a single gluster volume.
//
type GlusterVolumeService struct {
	BaseService
}

func NewGlusterVolumeService(connection *Connection, path string) *GlusterVolumeService {
	var result GlusterVolumeService
	result.connection = connection
	result.path = path
	return &result
}

//
// Get the gluster volume details.
// For example, to get details of a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/456/glustervolumes/123
// ----
// This GET request will return the following output:
// [source,xml]
// ----
// <gluster_volume id="123">
//  <name>data</name>
//  <link href="/ovirt-engine/api/clusters/456/glustervolumes/123/glusterbricks" rel="glusterbricks"/>
//  <disperse_count>0</disperse_count>
//  <options>
//    <option>
//      <name>storage.owner-gid</name>
//      <value>36</value>
//    </option>
//    <option>
//      <name>performance.io-cache</name>
//      <value>off</value>
//    </option>
//    <option>
//      <name>cluster.data-self-heal-algorithm</name>
//      <value>full</value>
//    </option>
//  </options>
//  <redundancy_count>0</redundancy_count>
//  <replica_count>3</replica_count>
//  <status>up</status>
//  <stripe_count>0</stripe_count>
//  <transport_types>
//    <transport_type>tcp</transport_type>
//  </transport_types>
//  <volume_type>replicate</volume_type>
//  </gluster_volume>
// ----
//
type GlusterVolumeServiceGetRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	follow               *string
}

func (p *GlusterVolumeServiceGetRequest) Header(key, value string) *GlusterVolumeServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceGetRequest) Query(key, value string) *GlusterVolumeServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceGetRequest) Follow(follow string) *GlusterVolumeServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *GlusterVolumeServiceGetRequest) Send() (*GlusterVolumeServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLGlusterVolumeReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &GlusterVolumeServiceGetResponse{volume: result}, nil
}

func (p *GlusterVolumeServiceGetRequest) MustSend() *GlusterVolumeServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get the gluster volume details.
// For example, to get details of a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/456/glustervolumes/123
// ----
// This GET request will return the following output:
// [source,xml]
// ----
// <gluster_volume id="123">
//  <name>data</name>
//  <link href="/ovirt-engine/api/clusters/456/glustervolumes/123/glusterbricks" rel="glusterbricks"/>
//  <disperse_count>0</disperse_count>
//  <options>
//    <option>
//      <name>storage.owner-gid</name>
//      <value>36</value>
//    </option>
//    <option>
//      <name>performance.io-cache</name>
//      <value>off</value>
//    </option>
//    <option>
//      <name>cluster.data-self-heal-algorithm</name>
//      <value>full</value>
//    </option>
//  </options>
//  <redundancy_count>0</redundancy_count>
//  <replica_count>3</replica_count>
//  <status>up</status>
//  <stripe_count>0</stripe_count>
//  <transport_types>
//    <transport_type>tcp</transport_type>
//  </transport_types>
//  <volume_type>replicate</volume_type>
//  </gluster_volume>
// ----
//
type GlusterVolumeServiceGetResponse struct {
	volume *GlusterVolume
}

func (p *GlusterVolumeServiceGetResponse) Volume() (*GlusterVolume, bool) {
	if p.volume != nil {
		return p.volume, true
	}
	return nil, false
}

func (p *GlusterVolumeServiceGetResponse) MustVolume() *GlusterVolume {
	if p.volume == nil {
		panic("volume in response does not exist")
	}
	return p.volume
}

//
// Get the gluster volume details.
// For example, to get details of a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// GET /ovirt-engine/api/clusters/456/glustervolumes/123
// ----
// This GET request will return the following output:
// [source,xml]
// ----
// <gluster_volume id="123">
//  <name>data</name>
//  <link href="/ovirt-engine/api/clusters/456/glustervolumes/123/glusterbricks" rel="glusterbricks"/>
//  <disperse_count>0</disperse_count>
//  <options>
//    <option>
//      <name>storage.owner-gid</name>
//      <value>36</value>
//    </option>
//    <option>
//      <name>performance.io-cache</name>
//      <value>off</value>
//    </option>
//    <option>
//      <name>cluster.data-self-heal-algorithm</name>
//      <value>full</value>
//    </option>
//  </options>
//  <redundancy_count>0</redundancy_count>
//  <replica_count>3</replica_count>
//  <status>up</status>
//  <stripe_count>0</stripe_count>
//  <transport_types>
//    <transport_type>tcp</transport_type>
//  </transport_types>
//  <volume_type>replicate</volume_type>
//  </gluster_volume>
// ----
//
func (p *GlusterVolumeService) Get() *GlusterVolumeServiceGetRequest {
	return &GlusterVolumeServiceGetRequest{GlusterVolumeService: p}
}

//
// Get gluster volume profile statistics.
// For example, to get profile statistics for a gluster volume with identifier `123` in cluster `456`, send a
// request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/getprofilestatistics
// ----
//
type GlusterVolumeServiceGetProfileStatisticsRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
}

func (p *GlusterVolumeServiceGetProfileStatisticsRequest) Header(key, value string) *GlusterVolumeServiceGetProfileStatisticsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceGetProfileStatisticsRequest) Query(key, value string) *GlusterVolumeServiceGetProfileStatisticsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceGetProfileStatisticsRequest) Send() (*GlusterVolumeServiceGetProfileStatisticsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/getprofilestatistics", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustDetails()
	return &GlusterVolumeServiceGetProfileStatisticsResponse{details: result}, nil
}

func (p *GlusterVolumeServiceGetProfileStatisticsRequest) MustSend() *GlusterVolumeServiceGetProfileStatisticsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get gluster volume profile statistics.
// For example, to get profile statistics for a gluster volume with identifier `123` in cluster `456`, send a
// request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/getprofilestatistics
// ----
//
type GlusterVolumeServiceGetProfileStatisticsResponse struct {
	details *GlusterVolumeProfileDetails
}

func (p *GlusterVolumeServiceGetProfileStatisticsResponse) Details() (*GlusterVolumeProfileDetails, bool) {
	if p.details != nil {
		return p.details, true
	}
	return nil, false
}

func (p *GlusterVolumeServiceGetProfileStatisticsResponse) MustDetails() *GlusterVolumeProfileDetails {
	if p.details == nil {
		panic("details in response does not exist")
	}
	return p.details
}

//
// Get gluster volume profile statistics.
// For example, to get profile statistics for a gluster volume with identifier `123` in cluster `456`, send a
// request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/getprofilestatistics
// ----
//
func (p *GlusterVolumeService) GetProfileStatistics() *GlusterVolumeServiceGetProfileStatisticsRequest {
	return &GlusterVolumeServiceGetProfileStatisticsRequest{GlusterVolumeService: p}
}

//
// Rebalance the gluster volume.
// Rebalancing a gluster volume helps to distribute the data evenly across all the bricks. After expanding or
// shrinking a gluster volume (without migrating data), we need to rebalance the data among the bricks. In a
// non-replicated volume, all bricks should be online to perform the rebalance operation. In a replicated volume, at
// least one of the bricks in the replica should be online.
// For example, to rebalance a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/rebalance
// ----
//
type GlusterVolumeServiceRebalanceRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
	fixLayout            *bool
	force                *bool
}

func (p *GlusterVolumeServiceRebalanceRequest) Header(key, value string) *GlusterVolumeServiceRebalanceRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceRebalanceRequest) Query(key, value string) *GlusterVolumeServiceRebalanceRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceRebalanceRequest) Async(async bool) *GlusterVolumeServiceRebalanceRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceRebalanceRequest) FixLayout(fixLayout bool) *GlusterVolumeServiceRebalanceRequest {
	p.fixLayout = &fixLayout
	return p
}

func (p *GlusterVolumeServiceRebalanceRequest) Force(force bool) *GlusterVolumeServiceRebalanceRequest {
	p.force = &force
	return p
}

func (p *GlusterVolumeServiceRebalanceRequest) Send() (*GlusterVolumeServiceRebalanceResponse, error) {
	rawURL := fmt.Sprintf("%s%s/rebalance", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.fixLayout != nil {
		actionBuilder.FixLayout(*p.fixLayout)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceRebalanceResponse), nil
}

func (p *GlusterVolumeServiceRebalanceRequest) MustSend() *GlusterVolumeServiceRebalanceResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Rebalance the gluster volume.
// Rebalancing a gluster volume helps to distribute the data evenly across all the bricks. After expanding or
// shrinking a gluster volume (without migrating data), we need to rebalance the data among the bricks. In a
// non-replicated volume, all bricks should be online to perform the rebalance operation. In a replicated volume, at
// least one of the bricks in the replica should be online.
// For example, to rebalance a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/rebalance
// ----
//
type GlusterVolumeServiceRebalanceResponse struct {
}

//
// Rebalance the gluster volume.
// Rebalancing a gluster volume helps to distribute the data evenly across all the bricks. After expanding or
// shrinking a gluster volume (without migrating data), we need to rebalance the data among the bricks. In a
// non-replicated volume, all bricks should be online to perform the rebalance operation. In a replicated volume, at
// least one of the bricks in the replica should be online.
// For example, to rebalance a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/rebalance
// ----
//
func (p *GlusterVolumeService) Rebalance() *GlusterVolumeServiceRebalanceRequest {
	return &GlusterVolumeServiceRebalanceRequest{GlusterVolumeService: p}
}

//
// Removes the gluster volume.
// For example, to remove a volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/456/glustervolumes/123
// ----
//
type GlusterVolumeServiceRemoveRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
}

func (p *GlusterVolumeServiceRemoveRequest) Header(key, value string) *GlusterVolumeServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceRemoveRequest) Query(key, value string) *GlusterVolumeServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceRemoveRequest) Async(async bool) *GlusterVolumeServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceRemoveRequest) Send() (*GlusterVolumeServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(GlusterVolumeServiceRemoveResponse), nil
}

func (p *GlusterVolumeServiceRemoveRequest) MustSend() *GlusterVolumeServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the gluster volume.
// For example, to remove a volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/456/glustervolumes/123
// ----
//
type GlusterVolumeServiceRemoveResponse struct {
}

//
// Removes the gluster volume.
// For example, to remove a volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// DELETE /ovirt-engine/api/clusters/456/glustervolumes/123
// ----
//
func (p *GlusterVolumeService) Remove() *GlusterVolumeServiceRemoveRequest {
	return &GlusterVolumeServiceRemoveRequest{GlusterVolumeService: p}
}

//
// Resets all the options set in the gluster volume.
// For example, to reset all options in a gluster volume with identifier `123` in cluster `456`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/resetalloptions
// ----
//
type GlusterVolumeServiceResetAllOptionsRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
}

func (p *GlusterVolumeServiceResetAllOptionsRequest) Header(key, value string) *GlusterVolumeServiceResetAllOptionsRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceResetAllOptionsRequest) Query(key, value string) *GlusterVolumeServiceResetAllOptionsRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceResetAllOptionsRequest) Async(async bool) *GlusterVolumeServiceResetAllOptionsRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceResetAllOptionsRequest) Send() (*GlusterVolumeServiceResetAllOptionsResponse, error) {
	rawURL := fmt.Sprintf("%s%s/resetalloptions", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceResetAllOptionsResponse), nil
}

func (p *GlusterVolumeServiceResetAllOptionsRequest) MustSend() *GlusterVolumeServiceResetAllOptionsResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Resets all the options set in the gluster volume.
// For example, to reset all options in a gluster volume with identifier `123` in cluster `456`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/resetalloptions
// ----
//
type GlusterVolumeServiceResetAllOptionsResponse struct {
}

//
// Resets all the options set in the gluster volume.
// For example, to reset all options in a gluster volume with identifier `123` in cluster `456`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/resetalloptions
// ----
//
func (p *GlusterVolumeService) ResetAllOptions() *GlusterVolumeServiceResetAllOptionsRequest {
	return &GlusterVolumeServiceResetAllOptionsRequest{GlusterVolumeService: p}
}

//
// Resets a particular option in the gluster volume.
// For example, to reset a particular option `option1` in a gluster volume with identifier `123` in cluster `456`,
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/resetoption
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//  <option name="option1"/>
// </action>
// ----
//
type GlusterVolumeServiceResetOptionRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
	force                *bool
	option               *Option
}

func (p *GlusterVolumeServiceResetOptionRequest) Header(key, value string) *GlusterVolumeServiceResetOptionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceResetOptionRequest) Query(key, value string) *GlusterVolumeServiceResetOptionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceResetOptionRequest) Async(async bool) *GlusterVolumeServiceResetOptionRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceResetOptionRequest) Force(force bool) *GlusterVolumeServiceResetOptionRequest {
	p.force = &force
	return p
}

func (p *GlusterVolumeServiceResetOptionRequest) Option(option *Option) *GlusterVolumeServiceResetOptionRequest {
	p.option = option
	return p
}

func (p *GlusterVolumeServiceResetOptionRequest) Send() (*GlusterVolumeServiceResetOptionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/resetoption", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	actionBuilder.Option(p.option)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceResetOptionResponse), nil
}

func (p *GlusterVolumeServiceResetOptionRequest) MustSend() *GlusterVolumeServiceResetOptionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Resets a particular option in the gluster volume.
// For example, to reset a particular option `option1` in a gluster volume with identifier `123` in cluster `456`,
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/resetoption
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//  <option name="option1"/>
// </action>
// ----
//
type GlusterVolumeServiceResetOptionResponse struct {
}

//
// Resets a particular option in the gluster volume.
// For example, to reset a particular option `option1` in a gluster volume with identifier `123` in cluster `456`,
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/resetoption
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//  <option name="option1"/>
// </action>
// ----
//
func (p *GlusterVolumeService) ResetOption() *GlusterVolumeServiceResetOptionRequest {
	return &GlusterVolumeServiceResetOptionRequest{GlusterVolumeService: p}
}

//
// Sets a particular option in the gluster volume.
// For example, to set `option1` with value `value1` in a gluster volume with identifier `123` in cluster `456`,
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/setoption
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//  <option name="option1" value="value1"/>
// </action>
// ----
//
type GlusterVolumeServiceSetOptionRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
	option               *Option
}

func (p *GlusterVolumeServiceSetOptionRequest) Header(key, value string) *GlusterVolumeServiceSetOptionRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceSetOptionRequest) Query(key, value string) *GlusterVolumeServiceSetOptionRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceSetOptionRequest) Async(async bool) *GlusterVolumeServiceSetOptionRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceSetOptionRequest) Option(option *Option) *GlusterVolumeServiceSetOptionRequest {
	p.option = option
	return p
}

func (p *GlusterVolumeServiceSetOptionRequest) Send() (*GlusterVolumeServiceSetOptionResponse, error) {
	rawURL := fmt.Sprintf("%s%s/setoption", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Option(p.option)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceSetOptionResponse), nil
}

func (p *GlusterVolumeServiceSetOptionRequest) MustSend() *GlusterVolumeServiceSetOptionResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Sets a particular option in the gluster volume.
// For example, to set `option1` with value `value1` in a gluster volume with identifier `123` in cluster `456`,
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/setoption
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//  <option name="option1" value="value1"/>
// </action>
// ----
//
type GlusterVolumeServiceSetOptionResponse struct {
}

//
// Sets a particular option in the gluster volume.
// For example, to set `option1` with value `value1` in a gluster volume with identifier `123` in cluster `456`,
// send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/setoption
// ----
// With the following request body:
// [source,xml]
// ----
// <action>
//  <option name="option1" value="value1"/>
// </action>
// ----
//
func (p *GlusterVolumeService) SetOption() *GlusterVolumeServiceSetOptionRequest {
	return &GlusterVolumeServiceSetOptionRequest{GlusterVolumeService: p}
}

//
// Starts the gluster volume.
// A Gluster Volume should be started to read/write data. For example, to start a gluster volume with identifier
// `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/start
// ----
//
type GlusterVolumeServiceStartRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
	force                *bool
}

func (p *GlusterVolumeServiceStartRequest) Header(key, value string) *GlusterVolumeServiceStartRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceStartRequest) Query(key, value string) *GlusterVolumeServiceStartRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceStartRequest) Async(async bool) *GlusterVolumeServiceStartRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceStartRequest) Force(force bool) *GlusterVolumeServiceStartRequest {
	p.force = &force
	return p
}

func (p *GlusterVolumeServiceStartRequest) Send() (*GlusterVolumeServiceStartResponse, error) {
	rawURL := fmt.Sprintf("%s%s/start", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceStartResponse), nil
}

func (p *GlusterVolumeServiceStartRequest) MustSend() *GlusterVolumeServiceStartResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Starts the gluster volume.
// A Gluster Volume should be started to read/write data. For example, to start a gluster volume with identifier
// `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/start
// ----
//
type GlusterVolumeServiceStartResponse struct {
}

//
// Starts the gluster volume.
// A Gluster Volume should be started to read/write data. For example, to start a gluster volume with identifier
// `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/start
// ----
//
func (p *GlusterVolumeService) Start() *GlusterVolumeServiceStartRequest {
	return &GlusterVolumeServiceStartRequest{GlusterVolumeService: p}
}

//
// Start profiling the gluster volume.
// For example, to start profiling a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/startprofile
// ----
//
type GlusterVolumeServiceStartProfileRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
}

func (p *GlusterVolumeServiceStartProfileRequest) Header(key, value string) *GlusterVolumeServiceStartProfileRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceStartProfileRequest) Query(key, value string) *GlusterVolumeServiceStartProfileRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceStartProfileRequest) Async(async bool) *GlusterVolumeServiceStartProfileRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceStartProfileRequest) Send() (*GlusterVolumeServiceStartProfileResponse, error) {
	rawURL := fmt.Sprintf("%s%s/startprofile", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceStartProfileResponse), nil
}

func (p *GlusterVolumeServiceStartProfileRequest) MustSend() *GlusterVolumeServiceStartProfileResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Start profiling the gluster volume.
// For example, to start profiling a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/startprofile
// ----
//
type GlusterVolumeServiceStartProfileResponse struct {
}

//
// Start profiling the gluster volume.
// For example, to start profiling a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/startprofile
// ----
//
func (p *GlusterVolumeService) StartProfile() *GlusterVolumeServiceStartProfileRequest {
	return &GlusterVolumeServiceStartProfileRequest{GlusterVolumeService: p}
}

//
// Stops the gluster volume.
// Stopping a volume will make its data inaccessible.
// For example, to stop a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stop
// ----
//
type GlusterVolumeServiceStopRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
	force                *bool
}

func (p *GlusterVolumeServiceStopRequest) Header(key, value string) *GlusterVolumeServiceStopRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceStopRequest) Query(key, value string) *GlusterVolumeServiceStopRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceStopRequest) Async(async bool) *GlusterVolumeServiceStopRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceStopRequest) Force(force bool) *GlusterVolumeServiceStopRequest {
	p.force = &force
	return p
}

func (p *GlusterVolumeServiceStopRequest) Send() (*GlusterVolumeServiceStopResponse, error) {
	rawURL := fmt.Sprintf("%s%s/stop", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceStopResponse), nil
}

func (p *GlusterVolumeServiceStopRequest) MustSend() *GlusterVolumeServiceStopResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Stops the gluster volume.
// Stopping a volume will make its data inaccessible.
// For example, to stop a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stop
// ----
//
type GlusterVolumeServiceStopResponse struct {
}

//
// Stops the gluster volume.
// Stopping a volume will make its data inaccessible.
// For example, to stop a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stop
// ----
//
func (p *GlusterVolumeService) Stop() *GlusterVolumeServiceStopRequest {
	return &GlusterVolumeServiceStopRequest{GlusterVolumeService: p}
}

//
// Stop profiling the gluster volume.
// For example, to stop profiling a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stopprofile
// ----
//
type GlusterVolumeServiceStopProfileRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
}

func (p *GlusterVolumeServiceStopProfileRequest) Header(key, value string) *GlusterVolumeServiceStopProfileRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceStopProfileRequest) Query(key, value string) *GlusterVolumeServiceStopProfileRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceStopProfileRequest) Async(async bool) *GlusterVolumeServiceStopProfileRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceStopProfileRequest) Send() (*GlusterVolumeServiceStopProfileResponse, error) {
	rawURL := fmt.Sprintf("%s%s/stopprofile", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceStopProfileResponse), nil
}

func (p *GlusterVolumeServiceStopProfileRequest) MustSend() *GlusterVolumeServiceStopProfileResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Stop profiling the gluster volume.
// For example, to stop profiling a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stopprofile
// ----
//
type GlusterVolumeServiceStopProfileResponse struct {
}

//
// Stop profiling the gluster volume.
// For example, to stop profiling a gluster volume with identifier `123` in cluster `456`, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stopprofile
// ----
//
func (p *GlusterVolumeService) StopProfile() *GlusterVolumeServiceStopProfileRequest {
	return &GlusterVolumeServiceStopProfileRequest{GlusterVolumeService: p}
}

//
// Stop rebalancing the gluster volume.
// For example, to stop rebalancing a gluster volume with identifier `123` in cluster `456`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stoprebalance
// ----
//
type GlusterVolumeServiceStopRebalanceRequest struct {
	GlusterVolumeService *GlusterVolumeService
	header               map[string]string
	query                map[string]string
	async                *bool
}

func (p *GlusterVolumeServiceStopRebalanceRequest) Header(key, value string) *GlusterVolumeServiceStopRebalanceRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *GlusterVolumeServiceStopRebalanceRequest) Query(key, value string) *GlusterVolumeServiceStopRebalanceRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *GlusterVolumeServiceStopRebalanceRequest) Async(async bool) *GlusterVolumeServiceStopRebalanceRequest {
	p.async = &async
	return p
}

func (p *GlusterVolumeServiceStopRebalanceRequest) Send() (*GlusterVolumeServiceStopRebalanceResponse, error) {
	rawURL := fmt.Sprintf("%s%s/stoprebalance", p.GlusterVolumeService.connection.URL(), p.GlusterVolumeService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.GlusterVolumeService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.GlusterVolumeService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.GlusterVolumeService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.GlusterVolumeService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.GlusterVolumeService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(GlusterVolumeServiceStopRebalanceResponse), nil
}

func (p *GlusterVolumeServiceStopRebalanceRequest) MustSend() *GlusterVolumeServiceStopRebalanceResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Stop rebalancing the gluster volume.
// For example, to stop rebalancing a gluster volume with identifier `123` in cluster `456`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stoprebalance
// ----
//
type GlusterVolumeServiceStopRebalanceResponse struct {
}

//
// Stop rebalancing the gluster volume.
// For example, to stop rebalancing a gluster volume with identifier `123` in cluster `456`, send a request like
// this:
// [source]
// ----
// POST /ovirt-engine/api/clusters/456/glustervolumes/123/stoprebalance
// ----
//
func (p *GlusterVolumeService) StopRebalance() *GlusterVolumeServiceStopRebalanceRequest {
	return &GlusterVolumeServiceStopRebalanceRequest{GlusterVolumeService: p}
}

//
// Reference to a service managing gluster bricks.
//
func (op *GlusterVolumeService) GlusterBricksService() *GlusterBricksService {
	return NewGlusterBricksService(op.connection, fmt.Sprintf("%s/glusterbricks", op.path))
}

//
//
func (op *GlusterVolumeService) StatisticsService() *StatisticsService {
	return NewStatisticsService(op.connection, fmt.Sprintf("%s/statistics", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *GlusterVolumeService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "glusterbricks" {
		return op.GlusterBricksService(), nil
	}
	if strings.HasPrefix(path, "glusterbricks/") {
		return op.GlusterBricksService().Service(path[14:])
	}
	if path == "statistics" {
		return op.StatisticsService(), nil
	}
	if strings.HasPrefix(path, "statistics/") {
		return op.StatisticsService().Service(path[11:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *GlusterVolumeService) String() string {
	return fmt.Sprintf("GlusterVolumeService:%s", op.path)
}

//
//
type SnapshotService struct {
	BaseService
}

func NewSnapshotService(connection *Connection, path string) *SnapshotService {
	var result SnapshotService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type SnapshotServiceGetRequest struct {
	SnapshotService *SnapshotService
	header          map[string]string
	query           map[string]string
	follow          *string
}

func (p *SnapshotServiceGetRequest) Header(key, value string) *SnapshotServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SnapshotServiceGetRequest) Query(key, value string) *SnapshotServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SnapshotServiceGetRequest) Follow(follow string) *SnapshotServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *SnapshotServiceGetRequest) Send() (*SnapshotServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SnapshotService.connection.URL(), p.SnapshotService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SnapshotService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SnapshotService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SnapshotService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SnapshotService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SnapshotService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLSnapshotReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &SnapshotServiceGetResponse{snapshot: result}, nil
}

func (p *SnapshotServiceGetRequest) MustSend() *SnapshotServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SnapshotServiceGetResponse struct {
	snapshot *Snapshot
}

func (p *SnapshotServiceGetResponse) Snapshot() (*Snapshot, bool) {
	if p.snapshot != nil {
		return p.snapshot, true
	}
	return nil, false
}

func (p *SnapshotServiceGetResponse) MustSnapshot() *Snapshot {
	if p.snapshot == nil {
		panic("snapshot in response does not exist")
	}
	return p.snapshot
}

//
//
func (p *SnapshotService) Get() *SnapshotServiceGetRequest {
	return &SnapshotServiceGetRequest{SnapshotService: p}
}

//
//
type SnapshotServiceRemoveRequest struct {
	SnapshotService *SnapshotService
	header          map[string]string
	query           map[string]string
	allContent      *bool
	async           *bool
}

func (p *SnapshotServiceRemoveRequest) Header(key, value string) *SnapshotServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SnapshotServiceRemoveRequest) Query(key, value string) *SnapshotServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SnapshotServiceRemoveRequest) AllContent(allContent bool) *SnapshotServiceRemoveRequest {
	p.allContent = &allContent
	return p
}

func (p *SnapshotServiceRemoveRequest) Async(async bool) *SnapshotServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *SnapshotServiceRemoveRequest) Send() (*SnapshotServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.SnapshotService.connection.URL(), p.SnapshotService.path)
	values := make(url.Values)
	if p.allContent != nil {
		values["all_content"] = []string{fmt.Sprintf("%v", *p.allContent)}
	}

	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SnapshotService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SnapshotService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SnapshotService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SnapshotService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SnapshotService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(SnapshotServiceRemoveResponse), nil
}

func (p *SnapshotServiceRemoveRequest) MustSend() *SnapshotServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type SnapshotServiceRemoveResponse struct {
}

//
//
func (p *SnapshotService) Remove() *SnapshotServiceRemoveRequest {
	return &SnapshotServiceRemoveRequest{SnapshotService: p}
}

//
// Restores a virtual machine snapshot.
// For example, to restore the snapshot with identifier `456` of virtual machine with identifier `123` send a
// request like this:
// [source]
// ----
// POST /ovirt-engine/api/vms/123/snapshots/456/restore
// ----
// With an empty `action` in the body:
// [source,xml]
// ----
// <action/>
// ----
//
type SnapshotServiceRestoreRequest struct {
	SnapshotService *SnapshotService
	header          map[string]string
	query           map[string]string
	async           *bool
	disks           *DiskSlice
	restoreMemory   *bool
}

func (p *SnapshotServiceRestoreRequest) Header(key, value string) *SnapshotServiceRestoreRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *SnapshotServiceRestoreRequest) Query(key, value string) *SnapshotServiceRestoreRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *SnapshotServiceRestoreRequest) Async(async bool) *SnapshotServiceRestoreRequest {
	p.async = &async
	return p
}

func (p *SnapshotServiceRestoreRequest) Disks(disks *DiskSlice) *SnapshotServiceRestoreRequest {
	p.disks = disks
	return p
}

func (p *SnapshotServiceRestoreRequest) DisksOfAny(anys ...*Disk) *SnapshotServiceRestoreRequest {
	if p.disks == nil {
		p.disks = new(DiskSlice)
	}
	p.disks.slice = append(p.disks.slice, anys...)
	return p
}

func (p *SnapshotServiceRestoreRequest) RestoreMemory(restoreMemory bool) *SnapshotServiceRestoreRequest {
	p.restoreMemory = &restoreMemory
	return p
}

func (p *SnapshotServiceRestoreRequest) Send() (*SnapshotServiceRestoreResponse, error) {
	rawURL := fmt.Sprintf("%s%s/restore", p.SnapshotService.connection.URL(), p.SnapshotService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	actionBuilder.Disks(p.disks)
	if p.restoreMemory != nil {
		actionBuilder.RestoreMemory(*p.restoreMemory)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.SnapshotService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.SnapshotService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.SnapshotService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.SnapshotService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.SnapshotService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(SnapshotServiceRestoreResponse), nil
}

func (p *SnapshotServiceRestoreRequest) MustSend() *SnapshotServiceRestoreResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Restores a virtual machine snapshot.
// For example, to restore the snapshot with identifier `456` of virtual machine with identifier `123` send a
// request like this:
// [source]
// ----
// POST /ovirt-engine/api/vms/123/snapshots/456/restore
// ----
// With an empty `action` in the body:
// [source,xml]
// ----
// <action/>
// ----
//
type SnapshotServiceRestoreResponse struct {
}

//
// Restores a virtual machine snapshot.
// For example, to restore the snapshot with identifier `456` of virtual machine with identifier `123` send a
// request like this:
// [source]
// ----
// POST /ovirt-engine/api/vms/123/snapshots/456/restore
// ----
// With an empty `action` in the body:
// [source,xml]
// ----
// <action/>
// ----
//
func (p *SnapshotService) Restore() *SnapshotServiceRestoreRequest {
	return &SnapshotServiceRestoreRequest{SnapshotService: p}
}

//
//
func (op *SnapshotService) CdromsService() *SnapshotCdromsService {
	return NewSnapshotCdromsService(op.connection, fmt.Sprintf("%s/cdroms", op.path))
}

//
//
func (op *SnapshotService) DisksService() *SnapshotDisksService {
	return NewSnapshotDisksService(op.connection, fmt.Sprintf("%s/disks", op.path))
}

//
//
func (op *SnapshotService) NicsService() *SnapshotNicsService {
	return NewSnapshotNicsService(op.connection, fmt.Sprintf("%s/nics", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *SnapshotService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "cdroms" {
		return op.CdromsService(), nil
	}
	if strings.HasPrefix(path, "cdroms/") {
		return op.CdromsService().Service(path[7:])
	}
	if path == "disks" {
		return op.DisksService(), nil
	}
	if strings.HasPrefix(path, "disks/") {
		return op.DisksService().Service(path[6:])
	}
	if path == "nics" {
		return op.NicsService(), nil
	}
	if strings.HasPrefix(path, "nics/") {
		return op.NicsService().Service(path[5:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *SnapshotService) String() string {
	return fmt.Sprintf("SnapshotService:%s", op.path)
}

//
//
type StorageDomainServerConnectionService struct {
	BaseService
}

func NewStorageDomainServerConnectionService(connection *Connection, path string) *StorageDomainServerConnectionService {
	var result StorageDomainServerConnectionService
	result.connection = connection
	result.path = path
	return &result
}

//
//
type StorageDomainServerConnectionServiceGetRequest struct {
	StorageDomainServerConnectionService *StorageDomainServerConnectionService
	header                               map[string]string
	query                                map[string]string
	follow                               *string
}

func (p *StorageDomainServerConnectionServiceGetRequest) Header(key, value string) *StorageDomainServerConnectionServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServerConnectionServiceGetRequest) Query(key, value string) *StorageDomainServerConnectionServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServerConnectionServiceGetRequest) Follow(follow string) *StorageDomainServerConnectionServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *StorageDomainServerConnectionServiceGetRequest) Send() (*StorageDomainServerConnectionServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainServerConnectionService.connection.URL(), p.StorageDomainServerConnectionService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainServerConnectionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainServerConnectionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainServerConnectionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainServerConnectionService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainServerConnectionService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageConnectionReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &StorageDomainServerConnectionServiceGetResponse{connection: result}, nil
}

func (p *StorageDomainServerConnectionServiceGetRequest) MustSend() *StorageDomainServerConnectionServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type StorageDomainServerConnectionServiceGetResponse struct {
	connection *StorageConnection
}

func (p *StorageDomainServerConnectionServiceGetResponse) Connection() (*StorageConnection, bool) {
	if p.connection != nil {
		return p.connection, true
	}
	return nil, false
}

func (p *StorageDomainServerConnectionServiceGetResponse) MustConnection() *StorageConnection {
	if p.connection == nil {
		panic("connection in response does not exist")
	}
	return p.connection
}

//
//
func (p *StorageDomainServerConnectionService) Get() *StorageDomainServerConnectionServiceGetRequest {
	return &StorageDomainServerConnectionServiceGetRequest{StorageDomainServerConnectionService: p}
}

//
// Detaches a storage connection from storage.
//
type StorageDomainServerConnectionServiceRemoveRequest struct {
	StorageDomainServerConnectionService *StorageDomainServerConnectionService
	header                               map[string]string
	query                                map[string]string
	async                                *bool
}

func (p *StorageDomainServerConnectionServiceRemoveRequest) Header(key, value string) *StorageDomainServerConnectionServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *StorageDomainServerConnectionServiceRemoveRequest) Query(key, value string) *StorageDomainServerConnectionServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *StorageDomainServerConnectionServiceRemoveRequest) Async(async bool) *StorageDomainServerConnectionServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *StorageDomainServerConnectionServiceRemoveRequest) Send() (*StorageDomainServerConnectionServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.StorageDomainServerConnectionService.connection.URL(), p.StorageDomainServerConnectionService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.StorageDomainServerConnectionService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.StorageDomainServerConnectionService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.StorageDomainServerConnectionService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.StorageDomainServerConnectionService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.StorageDomainServerConnectionService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(StorageDomainServerConnectionServiceRemoveResponse), nil
}

func (p *StorageDomainServerConnectionServiceRemoveRequest) MustSend() *StorageDomainServerConnectionServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Detaches a storage connection from storage.
//
type StorageDomainServerConnectionServiceRemoveResponse struct {
}

//
// Detaches a storage connection from storage.
//
func (p *StorageDomainServerConnectionService) Remove() *StorageDomainServerConnectionServiceRemoveRequest {
	return &StorageDomainServerConnectionServiceRemoveRequest{StorageDomainServerConnectionService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *StorageDomainServerConnectionService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *StorageDomainServerConnectionService) String() string {
	return fmt.Sprintf("StorageDomainServerConnectionService:%s", op.path)
}

//
//
type AttachedStorageDomainService struct {
	BaseService
}

func NewAttachedStorageDomainService(connection *Connection, path string) *AttachedStorageDomainService {
	var result AttachedStorageDomainService
	result.connection = connection
	result.path = path
	return &result
}

//
// This operation activates an attached storage domain.
// Once the storage domain is activated it is ready for use with the data center.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/activate
// ----
// The activate action does not take any action specific parameters,
// so the request body should contain an empty `action`:
// [source,xml]
// ----
// <action/>
// ----
//
type AttachedStorageDomainServiceActivateRequest struct {
	AttachedStorageDomainService *AttachedStorageDomainService
	header                       map[string]string
	query                        map[string]string
	async                        *bool
}

func (p *AttachedStorageDomainServiceActivateRequest) Header(key, value string) *AttachedStorageDomainServiceActivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainServiceActivateRequest) Query(key, value string) *AttachedStorageDomainServiceActivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainServiceActivateRequest) Async(async bool) *AttachedStorageDomainServiceActivateRequest {
	p.async = &async
	return p
}

func (p *AttachedStorageDomainServiceActivateRequest) Send() (*AttachedStorageDomainServiceActivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/activate", p.AttachedStorageDomainService.connection.URL(), p.AttachedStorageDomainService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainServiceActivateResponse), nil
}

func (p *AttachedStorageDomainServiceActivateRequest) MustSend() *AttachedStorageDomainServiceActivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation activates an attached storage domain.
// Once the storage domain is activated it is ready for use with the data center.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/activate
// ----
// The activate action does not take any action specific parameters,
// so the request body should contain an empty `action`:
// [source,xml]
// ----
// <action/>
// ----
//
type AttachedStorageDomainServiceActivateResponse struct {
}

//
// This operation activates an attached storage domain.
// Once the storage domain is activated it is ready for use with the data center.
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/activate
// ----
// The activate action does not take any action specific parameters,
// so the request body should contain an empty `action`:
// [source,xml]
// ----
// <action/>
// ----
//
func (p *AttachedStorageDomainService) Activate() *AttachedStorageDomainServiceActivateRequest {
	return &AttachedStorageDomainServiceActivateRequest{AttachedStorageDomainService: p}
}

//
// This operation deactivates an attached storage domain.
// Once the storage domain is deactivated it will not be used with the data center.
// For example, to deactivate storage domain `456`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/deactivate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// If the `force` parameter is `true` then the operation will succeed, even if the OVF update which takes place
// before the deactivation of the storage domain failed. If the `force` parameter is `false` and the OVF update failed,
// the deactivation of the storage domain will also fail.
//
type AttachedStorageDomainServiceDeactivateRequest struct {
	AttachedStorageDomainService *AttachedStorageDomainService
	header                       map[string]string
	query                        map[string]string
	async                        *bool
	force                        *bool
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Header(key, value string) *AttachedStorageDomainServiceDeactivateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Query(key, value string) *AttachedStorageDomainServiceDeactivateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Async(async bool) *AttachedStorageDomainServiceDeactivateRequest {
	p.async = &async
	return p
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Force(force bool) *AttachedStorageDomainServiceDeactivateRequest {
	p.force = &force
	return p
}

func (p *AttachedStorageDomainServiceDeactivateRequest) Send() (*AttachedStorageDomainServiceDeactivateResponse, error) {
	rawURL := fmt.Sprintf("%s%s/deactivate", p.AttachedStorageDomainService.connection.URL(), p.AttachedStorageDomainService.path)
	actionBuilder := NewActionBuilder()
	if p.async != nil {
		actionBuilder.Async(*p.async)
	}
	if p.force != nil {
		actionBuilder.Force(*p.force)
	}
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	_, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	return new(AttachedStorageDomainServiceDeactivateResponse), nil
}

func (p *AttachedStorageDomainServiceDeactivateRequest) MustSend() *AttachedStorageDomainServiceDeactivateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// This operation deactivates an attached storage domain.
// Once the storage domain is deactivated it will not be used with the data center.
// For example, to deactivate storage domain `456`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/deactivate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// If the `force` parameter is `true` then the operation will succeed, even if the OVF update which takes place
// before the deactivation of the storage domain failed. If the `force` parameter is `false` and the OVF update failed,
// the deactivation of the storage domain will also fail.
//
type AttachedStorageDomainServiceDeactivateResponse struct {
}

//
// This operation deactivates an attached storage domain.
// Once the storage domain is deactivated it will not be used with the data center.
// For example, to deactivate storage domain `456`, send the following request:
// [source]
// ----
// POST /ovirt-engine/api/datacenters/123/storagedomains/456/deactivate
// ----
// With a request body like this:
// [source,xml]
// ----
// <action/>
// ----
// If the `force` parameter is `true` then the operation will succeed, even if the OVF update which takes place
// before the deactivation of the storage domain failed. If the `force` parameter is `false` and the OVF update failed,
// the deactivation of the storage domain will also fail.
//
func (p *AttachedStorageDomainService) Deactivate() *AttachedStorageDomainServiceDeactivateRequest {
	return &AttachedStorageDomainServiceDeactivateRequest{AttachedStorageDomainService: p}
}

//
//
type AttachedStorageDomainServiceGetRequest struct {
	AttachedStorageDomainService *AttachedStorageDomainService
	header                       map[string]string
	query                        map[string]string
	follow                       *string
}

func (p *AttachedStorageDomainServiceGetRequest) Header(key, value string) *AttachedStorageDomainServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainServiceGetRequest) Query(key, value string) *AttachedStorageDomainServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainServiceGetRequest) Follow(follow string) *AttachedStorageDomainServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AttachedStorageDomainServiceGetRequest) Send() (*AttachedStorageDomainServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainService.connection.URL(), p.AttachedStorageDomainService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLStorageDomainReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AttachedStorageDomainServiceGetResponse{storageDomain: result}, nil
}

func (p *AttachedStorageDomainServiceGetRequest) MustSend() *AttachedStorageDomainServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AttachedStorageDomainServiceGetResponse struct {
	storageDomain *StorageDomain
}

func (p *AttachedStorageDomainServiceGetResponse) StorageDomain() (*StorageDomain, bool) {
	if p.storageDomain != nil {
		return p.storageDomain, true
	}
	return nil, false
}

func (p *AttachedStorageDomainServiceGetResponse) MustStorageDomain() *StorageDomain {
	if p.storageDomain == nil {
		panic("storageDomain in response does not exist")
	}
	return p.storageDomain
}

//
//
func (p *AttachedStorageDomainService) Get() *AttachedStorageDomainServiceGetRequest {
	return &AttachedStorageDomainServiceGetRequest{AttachedStorageDomainService: p}
}

//
//
type AttachedStorageDomainServiceRemoveRequest struct {
	AttachedStorageDomainService *AttachedStorageDomainService
	header                       map[string]string
	query                        map[string]string
	async                        *bool
}

func (p *AttachedStorageDomainServiceRemoveRequest) Header(key, value string) *AttachedStorageDomainServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AttachedStorageDomainServiceRemoveRequest) Query(key, value string) *AttachedStorageDomainServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AttachedStorageDomainServiceRemoveRequest) Async(async bool) *AttachedStorageDomainServiceRemoveRequest {
	p.async = &async
	return p
}

func (p *AttachedStorageDomainServiceRemoveRequest) Send() (*AttachedStorageDomainServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AttachedStorageDomainService.connection.URL(), p.AttachedStorageDomainService.path)
	values := make(url.Values)
	if p.async != nil {
		values["async"] = []string{fmt.Sprintf("%v", *p.async)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AttachedStorageDomainService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AttachedStorageDomainService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AttachedStorageDomainService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AttachedStorageDomainService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AttachedStorageDomainService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AttachedStorageDomainServiceRemoveResponse), nil
}

func (p *AttachedStorageDomainServiceRemoveRequest) MustSend() *AttachedStorageDomainServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
//
type AttachedStorageDomainServiceRemoveResponse struct {
}

//
//
func (p *AttachedStorageDomainService) Remove() *AttachedStorageDomainServiceRemoveRequest {
	return &AttachedStorageDomainServiceRemoveRequest{AttachedStorageDomainService: p}
}

//
//
func (op *AttachedStorageDomainService) DisksService() *AttachedStorageDomainDisksService {
	return NewAttachedStorageDomainDisksService(op.connection, fmt.Sprintf("%s/disks", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AttachedStorageDomainService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "disks" {
		return op.DisksService(), nil
	}
	if strings.HasPrefix(path, "disks/") {
		return op.DisksService().Service(path[6:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AttachedStorageDomainService) String() string {
	return fmt.Sprintf("AttachedStorageDomainService:%s", op.path)
}

//
// Manages the collection of disks available in the system.
//
type DisksService struct {
	BaseService
}

func NewDisksService(connection *Connection, path string) *DisksService {
	var result DisksService
	result.connection = connection
	result.path = path
	return &result
}

//
// Adds a new floating disk.
// There are three types of disks that can be added - disk image, direct LUN and
//  https://wiki.openstack.org/wiki/Cinder[Cinder] disk.
// *Adding a new image disk:*
// When creating a new floating image <<types/disk,Disk>>, the API requires the `storage_domain`, `provisioned_size`
// and `format` attributes.
// Note that block storage domains (i.e., storage domains with the <<types/storage_type, storage type>> of iSCSI or
// FCP) don't support the combination of the raw `format` with `sparse=true`, so `sparse=false` must be stated
// explicitly.
// To create a new floating image disk with specified `provisioned_size`, `format` and `name` on a storage domain
// with an id `123`, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
// *Adding a new direct LUN disk:*
// When adding a new floating direct LUN via the API, there are two flavors that can be used:
// . With a `host` element - in this case, the host is used for sanity checks (e.g., that the LUN is visible) and
// to retrieve basic information about the LUN (e.g., size and serial).
// . Without a `host` element - in this case, the operation is a database-only operation, and the storage is never
// accessed.
// To create a new floating direct LUN disk with a `host` element with an id `123`, specified `alias`, `type` and
// `logical_unit` with an id `456` (that has the attributes `address`, `port` and `target`),
// send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <alias>mylun</alias>
//   <lun_storage>
//     <host id="123"/>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="456">
//         <address>10.35.10.20</address>
//         <port>3260</port>
//         <target>iqn.2017-01.com.myhost:444</target>
//       </logical_unit>
//     </logical_units>
//   </lun_storage>
// </disk>
// ----
// To create a new floating direct LUN disk without using a host, remove the `host` element.
// *Adding a new Cinder disk:*
// To create a new floating Cinder disk, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <openstack_volume_type>
//     <name>myceph</name>
//   </openstack_volume_type>
//   <storage_domains>
//     <storage_domain>
//       <name>cinderDomain</name>
//     </storage_domain>
//   </storage_domains>
//   <provisioned_size>1073741824</provisioned_size>
//   <interface>virtio</interface>
//   <format>raw</format>
// </disk>
// ----
// *Adding a floating disks in order to upload disk snapshots:*
// Since version 4.2 of the engine it is possible to upload disks with
// snapshots. This request should be used to create the base image of the
// images chain (The consecutive disk snapshots (images), should be created
// using `disk-attachments` element when creating a snapshot).
// The disk has to be created with the same disk identifier and image identifier
// of the uploaded image. I.e. the identifiers should be saved as part of the
// backup process. The image identifier can be also fetched using the
// `qemu-img info` command. For example, if the disk image is stored into
// a file named `b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img`:
// [source,shell]
// ----
// $ qemu-img info b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img
// image: b548366b-fb51-4b41-97be-733c887fe305
// file format: qcow2
// virtual size: 1.0G (1073741824 bytes)
// disk size: 196K
// cluster_size: 65536
// backing file: ad58716a-1fe9-481f-815e-664de1df04eb
// backing file format: raw
// ----
// To create a disk with with the disk identifier and image identifier obtained
// with the `qemu-img info` command shown above, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk id="b7a4c6c5-443b-47c5-967f-6abc79675e8b">
//   <image_id>b548366b-fb51-4b41-97be-733c887fe305</image_id>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
//
type DisksServiceAddRequest struct {
	DisksService *DisksService
	header       map[string]string
	query        map[string]string
	disk         *Disk
}

func (p *DisksServiceAddRequest) Header(key, value string) *DisksServiceAddRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DisksServiceAddRequest) Query(key, value string) *DisksServiceAddRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DisksServiceAddRequest) Disk(disk *Disk) *DisksServiceAddRequest {
	p.disk = disk
	return p
}

func (p *DisksServiceAddRequest) Send() (*DisksServiceAddResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DisksService.connection.URL(), p.DisksService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLDiskWriteOne(writer, p.disk, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200, 201, 202}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &DisksServiceAddResponse{disk: result}, nil
}

func (p *DisksServiceAddRequest) MustSend() *DisksServiceAddResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Adds a new floating disk.
// There are three types of disks that can be added - disk image, direct LUN and
//  https://wiki.openstack.org/wiki/Cinder[Cinder] disk.
// *Adding a new image disk:*
// When creating a new floating image <<types/disk,Disk>>, the API requires the `storage_domain`, `provisioned_size`
// and `format` attributes.
// Note that block storage domains (i.e., storage domains with the <<types/storage_type, storage type>> of iSCSI or
// FCP) don't support the combination of the raw `format` with `sparse=true`, so `sparse=false` must be stated
// explicitly.
// To create a new floating image disk with specified `provisioned_size`, `format` and `name` on a storage domain
// with an id `123`, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
// *Adding a new direct LUN disk:*
// When adding a new floating direct LUN via the API, there are two flavors that can be used:
// . With a `host` element - in this case, the host is used for sanity checks (e.g., that the LUN is visible) and
// to retrieve basic information about the LUN (e.g., size and serial).
// . Without a `host` element - in this case, the operation is a database-only operation, and the storage is never
// accessed.
// To create a new floating direct LUN disk with a `host` element with an id `123`, specified `alias`, `type` and
// `logical_unit` with an id `456` (that has the attributes `address`, `port` and `target`),
// send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <alias>mylun</alias>
//   <lun_storage>
//     <host id="123"/>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="456">
//         <address>10.35.10.20</address>
//         <port>3260</port>
//         <target>iqn.2017-01.com.myhost:444</target>
//       </logical_unit>
//     </logical_units>
//   </lun_storage>
// </disk>
// ----
// To create a new floating direct LUN disk without using a host, remove the `host` element.
// *Adding a new Cinder disk:*
// To create a new floating Cinder disk, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <openstack_volume_type>
//     <name>myceph</name>
//   </openstack_volume_type>
//   <storage_domains>
//     <storage_domain>
//       <name>cinderDomain</name>
//     </storage_domain>
//   </storage_domains>
//   <provisioned_size>1073741824</provisioned_size>
//   <interface>virtio</interface>
//   <format>raw</format>
// </disk>
// ----
// *Adding a floating disks in order to upload disk snapshots:*
// Since version 4.2 of the engine it is possible to upload disks with
// snapshots. This request should be used to create the base image of the
// images chain (The consecutive disk snapshots (images), should be created
// using `disk-attachments` element when creating a snapshot).
// The disk has to be created with the same disk identifier and image identifier
// of the uploaded image. I.e. the identifiers should be saved as part of the
// backup process. The image identifier can be also fetched using the
// `qemu-img info` command. For example, if the disk image is stored into
// a file named `b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img`:
// [source,shell]
// ----
// $ qemu-img info b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img
// image: b548366b-fb51-4b41-97be-733c887fe305
// file format: qcow2
// virtual size: 1.0G (1073741824 bytes)
// disk size: 196K
// cluster_size: 65536
// backing file: ad58716a-1fe9-481f-815e-664de1df04eb
// backing file format: raw
// ----
// To create a disk with with the disk identifier and image identifier obtained
// with the `qemu-img info` command shown above, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk id="b7a4c6c5-443b-47c5-967f-6abc79675e8b">
//   <image_id>b548366b-fb51-4b41-97be-733c887fe305</image_id>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
//
type DisksServiceAddResponse struct {
	disk *Disk
}

func (p *DisksServiceAddResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *DisksServiceAddResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Adds a new floating disk.
// There are three types of disks that can be added - disk image, direct LUN and
//  https://wiki.openstack.org/wiki/Cinder[Cinder] disk.
// *Adding a new image disk:*
// When creating a new floating image <<types/disk,Disk>>, the API requires the `storage_domain`, `provisioned_size`
// and `format` attributes.
// Note that block storage domains (i.e., storage domains with the <<types/storage_type, storage type>> of iSCSI or
// FCP) don't support the combination of the raw `format` with `sparse=true`, so `sparse=false` must be stated
// explicitly.
// To create a new floating image disk with specified `provisioned_size`, `format` and `name` on a storage domain
// with an id `123`, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
// *Adding a new direct LUN disk:*
// When adding a new floating direct LUN via the API, there are two flavors that can be used:
// . With a `host` element - in this case, the host is used for sanity checks (e.g., that the LUN is visible) and
// to retrieve basic information about the LUN (e.g., size and serial).
// . Without a `host` element - in this case, the operation is a database-only operation, and the storage is never
// accessed.
// To create a new floating direct LUN disk with a `host` element with an id `123`, specified `alias`, `type` and
// `logical_unit` with an id `456` (that has the attributes `address`, `port` and `target`),
// send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <alias>mylun</alias>
//   <lun_storage>
//     <host id="123"/>
//     <type>iscsi</type>
//     <logical_units>
//       <logical_unit id="456">
//         <address>10.35.10.20</address>
//         <port>3260</port>
//         <target>iqn.2017-01.com.myhost:444</target>
//       </logical_unit>
//     </logical_units>
//   </lun_storage>
// </disk>
// ----
// To create a new floating direct LUN disk without using a host, remove the `host` element.
// *Adding a new Cinder disk:*
// To create a new floating Cinder disk, send a request as follows:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk>
//   <openstack_volume_type>
//     <name>myceph</name>
//   </openstack_volume_type>
//   <storage_domains>
//     <storage_domain>
//       <name>cinderDomain</name>
//     </storage_domain>
//   </storage_domains>
//   <provisioned_size>1073741824</provisioned_size>
//   <interface>virtio</interface>
//   <format>raw</format>
// </disk>
// ----
// *Adding a floating disks in order to upload disk snapshots:*
// Since version 4.2 of the engine it is possible to upload disks with
// snapshots. This request should be used to create the base image of the
// images chain (The consecutive disk snapshots (images), should be created
// using `disk-attachments` element when creating a snapshot).
// The disk has to be created with the same disk identifier and image identifier
// of the uploaded image. I.e. the identifiers should be saved as part of the
// backup process. The image identifier can be also fetched using the
// `qemu-img info` command. For example, if the disk image is stored into
// a file named `b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img`:
// [source,shell]
// ----
// $ qemu-img info b7a4c6c5-443b-47c5-967f-6abc79675e8b/myimage.img
// image: b548366b-fb51-4b41-97be-733c887fe305
// file format: qcow2
// virtual size: 1.0G (1073741824 bytes)
// disk size: 196K
// cluster_size: 65536
// backing file: ad58716a-1fe9-481f-815e-664de1df04eb
// backing file format: raw
// ----
// To create a disk with with the disk identifier and image identifier obtained
// with the `qemu-img info` command shown above, send a request like this:
// [source]
// ----
// POST /ovirt-engine/api/disks
// ----
// With a request body as follows:
// [source,xml]
// ----
// <disk id="b7a4c6c5-443b-47c5-967f-6abc79675e8b">
//   <image_id>b548366b-fb51-4b41-97be-733c887fe305</image_id>
//   <storage_domains>
//     <storage_domain id="123"/>
//   </storage_domains>
//   <name>mydisk</name>
//   <provisioned_size>1048576</provisioned_size>
//   <format>cow</format>
// </disk>
// ----
//
func (p *DisksService) Add() *DisksServiceAddRequest {
	return &DisksServiceAddRequest{DisksService: p}
}

//
// Get list of disks.
// [source]
// ----
// GET /ovirt-engine/api/disks
// ----
// You will get a XML response which will look like this one:
// [source,xml]
// ----
// <disks>
//   <disk id="123">
//     <actions>...</actions>
//     <name>MyDisk</name>
//     <description>MyDisk description</description>
//     <link href="/ovirt-engine/api/disks/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/disks/123/statistics" rel="statistics"/>
//     <actual_size>5345845248</actual_size>
//     <alias>MyDisk alias</alias>
//     ...
//     <status>ok</status>
//     <storage_type>image</storage_type>
//     <wipe_after_delete>false</wipe_after_delete>
//     <disk_profile id="123"/>
//     <quota id="123"/>
//     <storage_domains>...</storage_domains>
//   </disk>
//   ...
// </disks>
// ----
// The order of the returned list of disks is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type DisksServiceListRequest struct {
	DisksService  *DisksService
	header        map[string]string
	query         map[string]string
	caseSensitive *bool
	follow        *string
	max           *int64
	search        *string
}

func (p *DisksServiceListRequest) Header(key, value string) *DisksServiceListRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DisksServiceListRequest) Query(key, value string) *DisksServiceListRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DisksServiceListRequest) CaseSensitive(caseSensitive bool) *DisksServiceListRequest {
	p.caseSensitive = &caseSensitive
	return p
}

func (p *DisksServiceListRequest) Follow(follow string) *DisksServiceListRequest {
	p.follow = &follow
	return p
}

func (p *DisksServiceListRequest) Max(max int64) *DisksServiceListRequest {
	p.max = &max
	return p
}

func (p *DisksServiceListRequest) Search(search string) *DisksServiceListRequest {
	p.search = &search
	return p
}

func (p *DisksServiceListRequest) Send() (*DisksServiceListResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.DisksService.connection.URL(), p.DisksService.path)
	values := make(url.Values)
	if p.caseSensitive != nil {
		values["case_sensitive"] = []string{fmt.Sprintf("%v", *p.caseSensitive)}
	}

	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.max != nil {
		values["max"] = []string{fmt.Sprintf("%v", *p.max)}
	}

	if p.search != nil {
		values["search"] = []string{fmt.Sprintf("%v", *p.search)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLDiskReadMany(reader, nil)
	if err != nil {
		return nil, err
	}
	return &DisksServiceListResponse{disks: result}, nil
}

func (p *DisksServiceListRequest) MustSend() *DisksServiceListResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Get list of disks.
// [source]
// ----
// GET /ovirt-engine/api/disks
// ----
// You will get a XML response which will look like this one:
// [source,xml]
// ----
// <disks>
//   <disk id="123">
//     <actions>...</actions>
//     <name>MyDisk</name>
//     <description>MyDisk description</description>
//     <link href="/ovirt-engine/api/disks/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/disks/123/statistics" rel="statistics"/>
//     <actual_size>5345845248</actual_size>
//     <alias>MyDisk alias</alias>
//     ...
//     <status>ok</status>
//     <storage_type>image</storage_type>
//     <wipe_after_delete>false</wipe_after_delete>
//     <disk_profile id="123"/>
//     <quota id="123"/>
//     <storage_domains>...</storage_domains>
//   </disk>
//   ...
// </disks>
// ----
// The order of the returned list of disks is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
type DisksServiceListResponse struct {
	disks *DiskSlice
}

func (p *DisksServiceListResponse) Disks() (*DiskSlice, bool) {
	if p.disks != nil {
		return p.disks, true
	}
	return nil, false
}

func (p *DisksServiceListResponse) MustDisks() *DiskSlice {
	if p.disks == nil {
		panic("disks in response does not exist")
	}
	return p.disks
}

//
// Get list of disks.
// [source]
// ----
// GET /ovirt-engine/api/disks
// ----
// You will get a XML response which will look like this one:
// [source,xml]
// ----
// <disks>
//   <disk id="123">
//     <actions>...</actions>
//     <name>MyDisk</name>
//     <description>MyDisk description</description>
//     <link href="/ovirt-engine/api/disks/123/permissions" rel="permissions"/>
//     <link href="/ovirt-engine/api/disks/123/statistics" rel="statistics"/>
//     <actual_size>5345845248</actual_size>
//     <alias>MyDisk alias</alias>
//     ...
//     <status>ok</status>
//     <storage_type>image</storage_type>
//     <wipe_after_delete>false</wipe_after_delete>
//     <disk_profile id="123"/>
//     <quota id="123"/>
//     <storage_domains>...</storage_domains>
//   </disk>
//   ...
// </disks>
// ----
// The order of the returned list of disks is guaranteed only if the `sortby` clause is included in the
// `search` parameter.
//
func (p *DisksService) List() *DisksServiceListRequest {
	return &DisksServiceListRequest{DisksService: p}
}

//
// Add a new lun disk to the storage domain.
//
type DisksServiceAddLunRequest struct {
	DisksService *DisksService
	header       map[string]string
	query        map[string]string
	disk         *Disk
}

func (p *DisksServiceAddLunRequest) Header(key, value string) *DisksServiceAddLunRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DisksServiceAddLunRequest) Query(key, value string) *DisksServiceAddLunRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DisksServiceAddLunRequest) Disk(disk *Disk) *DisksServiceAddLunRequest {
	p.disk = disk
	return p
}

func (p *DisksServiceAddLunRequest) Send() (*DisksServiceAddLunResponse, error) {
	rawURL := fmt.Sprintf("%s%s/lun", p.DisksService.connection.URL(), p.DisksService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Disk(p.disk)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustDisk()
	return &DisksServiceAddLunResponse{disk: result}, nil
}

func (p *DisksServiceAddLunRequest) MustSend() *DisksServiceAddLunResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new lun disk to the storage domain.
//
type DisksServiceAddLunResponse struct {
	disk *Disk
}

func (p *DisksServiceAddLunResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *DisksServiceAddLunResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Add a new lun disk to the storage domain.
//
func (p *DisksService) AddLun() *DisksServiceAddLunRequest {
	return &DisksServiceAddLunRequest{DisksService: p}
}

//
// Add a new disk to the storage domain with the specified size allocating space from the storage domain.
//
type DisksServiceAddOnStorageDomainRequest struct {
	DisksService *DisksService
	header       map[string]string
	query        map[string]string
	disk         *Disk
}

func (p *DisksServiceAddOnStorageDomainRequest) Header(key, value string) *DisksServiceAddOnStorageDomainRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *DisksServiceAddOnStorageDomainRequest) Query(key, value string) *DisksServiceAddOnStorageDomainRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *DisksServiceAddOnStorageDomainRequest) Disk(disk *Disk) *DisksServiceAddOnStorageDomainRequest {
	p.disk = disk
	return p
}

func (p *DisksServiceAddOnStorageDomainRequest) Send() (*DisksServiceAddOnStorageDomainResponse, error) {
	rawURL := fmt.Sprintf("%s%s/onstoragedomain", p.DisksService.connection.URL(), p.DisksService.path)
	actionBuilder := NewActionBuilder()
	actionBuilder.Disk(p.disk)
	action, err := actionBuilder.Build()
	if err != nil {
		return nil, err
	}
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err = XMLActionWriteOne(writer, action, "")
	writer.Flush()
	req, err := http.NewRequest("POST", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.DisksService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.DisksService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.DisksService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.DisksService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.DisksService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	action, errCheckAction := CheckAction(resp)
	if errCheckAction != nil {
		return nil, errCheckAction
	}
	result := action.MustDisk()
	return &DisksServiceAddOnStorageDomainResponse{disk: result}, nil
}

func (p *DisksServiceAddOnStorageDomainRequest) MustSend() *DisksServiceAddOnStorageDomainResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Add a new disk to the storage domain with the specified size allocating space from the storage domain.
//
type DisksServiceAddOnStorageDomainResponse struct {
	disk *Disk
}

func (p *DisksServiceAddOnStorageDomainResponse) Disk() (*Disk, bool) {
	if p.disk != nil {
		return p.disk, true
	}
	return nil, false
}

func (p *DisksServiceAddOnStorageDomainResponse) MustDisk() *Disk {
	if p.disk == nil {
		panic("disk in response does not exist")
	}
	return p.disk
}

//
// Add a new disk to the storage domain with the specified size allocating space from the storage domain.
//
func (p *DisksService) AddOnStorageDomain() *DisksServiceAddOnStorageDomainRequest {
	return &DisksServiceAddOnStorageDomainRequest{DisksService: p}
}

//
// Reference to a service managing a specific disk.
//
func (op *DisksService) DiskService(id string) *DiskService {
	return NewDiskService(op.connection, fmt.Sprintf("%s/%s", op.path, id))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *DisksService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	index := strings.Index(path, "/")
	if index == -1 {
		return op.DiskService(path), nil
	}
	return op.DiskService(path[:index]).Service(path[index+1:])
}

func (op *DisksService) String() string {
	return fmt.Sprintf("DisksService:%s", op.path)
}

//
// The details of a single affinity label.
//
type AffinityLabelService struct {
	BaseService
}

func NewAffinityLabelService(connection *Connection, path string) *AffinityLabelService {
	var result AffinityLabelService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the details of a label.
//
type AffinityLabelServiceGetRequest struct {
	AffinityLabelService *AffinityLabelService
	header               map[string]string
	query                map[string]string
	follow               *string
}

func (p *AffinityLabelServiceGetRequest) Header(key, value string) *AffinityLabelServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelServiceGetRequest) Query(key, value string) *AffinityLabelServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelServiceGetRequest) Follow(follow string) *AffinityLabelServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AffinityLabelServiceGetRequest) Send() (*AffinityLabelServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelService.connection.URL(), p.AffinityLabelService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelServiceGetResponse{label: result}, nil
}

func (p *AffinityLabelServiceGetRequest) MustSend() *AffinityLabelServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves the details of a label.
//
type AffinityLabelServiceGetResponse struct {
	label *AffinityLabel
}

func (p *AffinityLabelServiceGetResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AffinityLabelServiceGetResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Retrieves the details of a label.
//
func (p *AffinityLabelService) Get() *AffinityLabelServiceGetRequest {
	return &AffinityLabelServiceGetRequest{AffinityLabelService: p}
}

//
// Removes a label from the system and clears all assignments
// of the removed label.
//
type AffinityLabelServiceRemoveRequest struct {
	AffinityLabelService *AffinityLabelService
	header               map[string]string
	query                map[string]string
}

func (p *AffinityLabelServiceRemoveRequest) Header(key, value string) *AffinityLabelServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelServiceRemoveRequest) Query(key, value string) *AffinityLabelServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelServiceRemoveRequest) Send() (*AffinityLabelServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelService.connection.URL(), p.AffinityLabelService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AffinityLabelServiceRemoveResponse), nil
}

func (p *AffinityLabelServiceRemoveRequest) MustSend() *AffinityLabelServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes a label from the system and clears all assignments
// of the removed label.
//
type AffinityLabelServiceRemoveResponse struct {
}

//
// Removes a label from the system and clears all assignments
// of the removed label.
//
func (p *AffinityLabelService) Remove() *AffinityLabelServiceRemoveRequest {
	return &AffinityLabelServiceRemoveRequest{AffinityLabelService: p}
}

//
// Updates a label. This call will update all metadata, such as the name
// or description.
//
type AffinityLabelServiceUpdateRequest struct {
	AffinityLabelService *AffinityLabelService
	header               map[string]string
	query                map[string]string
	label                *AffinityLabel
}

func (p *AffinityLabelServiceUpdateRequest) Header(key, value string) *AffinityLabelServiceUpdateRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AffinityLabelServiceUpdateRequest) Query(key, value string) *AffinityLabelServiceUpdateRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AffinityLabelServiceUpdateRequest) Label(label *AffinityLabel) *AffinityLabelServiceUpdateRequest {
	p.label = label
	return p
}

func (p *AffinityLabelServiceUpdateRequest) Send() (*AffinityLabelServiceUpdateResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AffinityLabelService.connection.URL(), p.AffinityLabelService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	var body bytes.Buffer
	writer := NewXMLWriter(&body)
	err := XMLAffinityLabelWriteOne(writer, p.label, "")
	if err != nil {
		return nil, err
	}
	writer.Flush()
	req, err := http.NewRequest("PUT", rawURL, &body)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AffinityLabelServiceUpdateResponse{label: result}, nil
}

func (p *AffinityLabelServiceUpdateRequest) MustSend() *AffinityLabelServiceUpdateResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Updates a label. This call will update all metadata, such as the name
// or description.
//
type AffinityLabelServiceUpdateResponse struct {
	label *AffinityLabel
}

func (p *AffinityLabelServiceUpdateResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AffinityLabelServiceUpdateResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Updates a label. This call will update all metadata, such as the name
// or description.
//
func (p *AffinityLabelService) Update() *AffinityLabelServiceUpdateRequest {
	return &AffinityLabelServiceUpdateRequest{AffinityLabelService: p}
}

//
// List all hosts with this label.
//
func (op *AffinityLabelService) HostsService() *AffinityLabelHostsService {
	return NewAffinityLabelHostsService(op.connection, fmt.Sprintf("%s/hosts", op.path))
}

//
// List all virtual machines with this label.
//
func (op *AffinityLabelService) VmsService() *AffinityLabelVmsService {
	return NewAffinityLabelVmsService(op.connection, fmt.Sprintf("%s/vms", op.path))
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AffinityLabelService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	if path == "hosts" {
		return op.HostsService(), nil
	}
	if strings.HasPrefix(path, "hosts/") {
		return op.HostsService().Service(path[6:])
	}
	if path == "vms" {
		return op.VmsService(), nil
	}
	if strings.HasPrefix(path, "vms/") {
		return op.VmsService().Service(path[4:])
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AffinityLabelService) String() string {
	return fmt.Sprintf("AffinityLabelService:%s", op.path)
}

//
// This service represents one label to entity assignment
// when accessed using the entities/affinitylabels subcollection.
//
type AssignedAffinityLabelService struct {
	BaseService
}

func NewAssignedAffinityLabelService(connection *Connection, path string) *AssignedAffinityLabelService {
	var result AssignedAffinityLabelService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves details about the attached label.
//
type AssignedAffinityLabelServiceGetRequest struct {
	AssignedAffinityLabelService *AssignedAffinityLabelService
	header                       map[string]string
	query                        map[string]string
	follow                       *string
}

func (p *AssignedAffinityLabelServiceGetRequest) Header(key, value string) *AssignedAffinityLabelServiceGetRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedAffinityLabelServiceGetRequest) Query(key, value string) *AssignedAffinityLabelServiceGetRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedAffinityLabelServiceGetRequest) Follow(follow string) *AssignedAffinityLabelServiceGetRequest {
	p.follow = &follow
	return p
}

func (p *AssignedAffinityLabelServiceGetRequest) Send() (*AssignedAffinityLabelServiceGetResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedAffinityLabelService.connection.URL(), p.AssignedAffinityLabelService.path)
	values := make(url.Values)
	if p.follow != nil {
		values["follow"] = []string{fmt.Sprintf("%v", *p.follow)}
	}

	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("GET", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedAffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedAffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedAffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedAffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedAffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	respBodyBytes, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	reader := NewXMLReader(respBodyBytes)
	result, err := XMLAffinityLabelReadOne(reader, nil, "")
	if err != nil {
		return nil, err
	}
	return &AssignedAffinityLabelServiceGetResponse{label: result}, nil
}

func (p *AssignedAffinityLabelServiceGetRequest) MustSend() *AssignedAffinityLabelServiceGetResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Retrieves details about the attached label.
//
type AssignedAffinityLabelServiceGetResponse struct {
	label *AffinityLabel
}

func (p *AssignedAffinityLabelServiceGetResponse) Label() (*AffinityLabel, bool) {
	if p.label != nil {
		return p.label, true
	}
	return nil, false
}

func (p *AssignedAffinityLabelServiceGetResponse) MustLabel() *AffinityLabel {
	if p.label == nil {
		panic("label in response does not exist")
	}
	return p.label
}

//
// Retrieves details about the attached label.
//
func (p *AssignedAffinityLabelService) Get() *AssignedAffinityLabelServiceGetRequest {
	return &AssignedAffinityLabelServiceGetRequest{AssignedAffinityLabelService: p}
}

//
// Removes the label from an entity. Does not touch the label itself.
//
type AssignedAffinityLabelServiceRemoveRequest struct {
	AssignedAffinityLabelService *AssignedAffinityLabelService
	header                       map[string]string
	query                        map[string]string
}

func (p *AssignedAffinityLabelServiceRemoveRequest) Header(key, value string) *AssignedAffinityLabelServiceRemoveRequest {
	if p.header == nil {
		p.header = make(map[string]string)
	}
	p.header[key] = value
	return p
}

func (p *AssignedAffinityLabelServiceRemoveRequest) Query(key, value string) *AssignedAffinityLabelServiceRemoveRequest {
	if p.query == nil {
		p.query = make(map[string]string)
	}
	p.query[key] = value
	return p
}

func (p *AssignedAffinityLabelServiceRemoveRequest) Send() (*AssignedAffinityLabelServiceRemoveResponse, error) {
	rawURL := fmt.Sprintf("%s%s", p.AssignedAffinityLabelService.connection.URL(), p.AssignedAffinityLabelService.path)
	values := make(url.Values)
	if p.query != nil {
		for k, v := range p.query {
			values[k] = []string{v}
		}
	}
	if len(values) > 0 {
		rawURL = fmt.Sprintf("%s?%s", rawURL, values.Encode())
	}
	req, err := http.NewRequest("DELETE", rawURL, nil)
	if err != nil {
		return nil, err
	}

	for hk, hv := range p.AssignedAffinityLabelService.connection.headers {
		req.Header.Add(hk, hv)
	}

	if p.header != nil {
		for hk, hv := range p.header {
			req.Header.Add(hk, hv)
		}
	}

	req.Header.Add("User-Agent", fmt.Sprintf("GoSDK/%s", SDK_VERSION))
	req.Header.Add("Version", "4")
	req.Header.Add("Content-Type", "application/xml")
	req.Header.Add("Accept", "application/xml")
	// get OAuth access token
	token, err := p.AssignedAffinityLabelService.connection.authenticate()
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", token))
	// Send the request and wait for the response
	resp, err := p.AssignedAffinityLabelService.connection.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if p.AssignedAffinityLabelService.connection.logFunc != nil {
		dumpReq, err := httputil.DumpRequestOut(req, true)
		if err != nil {
			return nil, err
		}
		dumpResp, err := httputil.DumpResponse(resp, true)
		if err != nil {
			return nil, err
		}
		p.AssignedAffinityLabelService.connection.logFunc("<<<<<<Request:\n%sResponse:\n%s>>>>>>\n", string(dumpReq), string(dumpResp))
	}
	if !Contains(resp.StatusCode, []int{200}) {
		return nil, CheckFault(resp)
	}
	_, errReadBody := ioutil.ReadAll(resp.Body)
	if errReadBody != nil {
		return nil, errReadBody
	}
	return new(AssignedAffinityLabelServiceRemoveResponse), nil
}

func (p *AssignedAffinityLabelServiceRemoveRequest) MustSend() *AssignedAffinityLabelServiceRemoveResponse {
	if v, err := p.Send(); err != nil {
		panic(err)
	} else {
		return v
	}
}

//
// Removes the label from an entity. Does not touch the label itself.
//
type AssignedAffinityLabelServiceRemoveResponse struct {
}

//
// Removes the label from an entity. Does not touch the label itself.
//
func (p *AssignedAffinityLabelService) Remove() *AssignedAffinityLabelServiceRemoveRequest {
	return &AssignedAffinityLabelServiceRemoveRequest{AssignedAffinityLabelService: p}
}

//
// Service locator method, returns individual service on which the URI is dispatched.
//
func (op *AssignedAffinityLabelService) Service(path string) (Service, error) {
	if path == "" {
		return op, nil
	}
	return nil, fmt.Errorf("The path <%s> doesn't correspond to any service", path)
}

func (op *AssignedAffinityLabelService) String() string {
	return fmt.Sprintf("AssignedAffinityLabelService:%s", op.path)
}

//
// A service to manage a specific data center network.
//
type DataCenterNetworkService struct {
	BaseService
}

func NewDataCenterNetworkService(connection *Connection, path string) *DataCenterNetworkService {
	var result DataCenterNetworkService
	result.connection = connection
	result.path = path
	return &result
}

//
// Retrieves the data center network details.
//
type DataCenterNetworkServiceGetRequest struct {
	DataCenterNetworkService *DataCenterNetworkService
	heade