/*
 * Copyright (C) 2007 JasperSoft http://www.jaspersoft.com
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed WITHOUT ANY WARRANTY; and without the 
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt 
 * or write to:
 * 
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
#ifndef jasperserver_stringbuffer_H
#define jasperserver_stringbuffer_H


#define INCREMENT_STRING_SIZE 256
/**
*  Easy way to cast a jasperserver_string to a C-string.
*
*  \code
*  printf( "%s", JS_CSTR( my_jasperserver_string ) );
*  \endcode
*
*/
#define JS_CSTR(c) (const char *)c->buffer->content

/**
*  Easy way to cast a jasperserver_string to an xmlChar *.
*
*/
#define JS_UTFSTR(c) (const xmlChar *)c->buffer->content

/**
*  Easy syntax to check if a jasperserver_string or has length == 0
*
*/
#define JS_NOTNULL(c) (c && c->buffer->use >0)


#include <libxml/encoding.h>

/**
\struct jasperserver_string_t stringbuffer.h

This struct implements an UTF capable string (it wraps libxml2 xmlBuffer structure)
You should always use a pointer, create the string using jasperserver_string_new and free it using
jasperserver_string_free.

\see jasperserver_string_new
*/
typedef struct jasperserver_string
{
       xmlBufferPtr buffer;
} jasperserver_string_t;


/** \brief  Create a new blank string.
    
    This function initializes a string.
    It is responsability of the user  free the memory allocated by this function
    using jasperserver_string_free()
    
    \return a pointer to a new string
    \see jasperserver_string_free()
        
*/
jasperserver_string_t *jasperserver_string_new();

/** \brief  Free the memory allocated for a string
    
    \param jsstr a pointer to a jasperserver_string
    \see jasperserver_string_new()
        
*/
void            jasperserver_string_free(jasperserver_string_t *jsstr);

/** \brief  Append an string to a jasperserver_string
    
    \param jsstr a pointer to a jasperserver_string
    \param str an xmlChar *, it can contain UTF cahracters
        
*/
void            jasperserver_string_append(jasperserver_string_t *jsstr,const xmlChar *str);

/** \brief  Append an C-string to a jasperserver_string
    
    \param jsstr a pointer to a jasperserver_string
    \param str a C-string to append
        
*/
void            jasperserver_string_cappend(jasperserver_string_t *jsstr,const char *str);

/** \brief  set the value of the string
    
    \param jsstr a pointer to a jasperserver_string
    \param str an xmlChar *, it can contain UTF cahracters
        
*/
void            jasperserver_string_set(jasperserver_string_t *jsstr,const xmlChar *str);

/** \brief  set the value of the string
    
    \param jsstr a pointer to a jasperserver_string
    \param str a C-string to append
        
*/
void            jasperserver_string_cset(jasperserver_string_t *jsstr,const char *str);

/** \brief  set the formatted string as value of the string
    
    \param jsstr a pointer to a jasperserver_string
    \param format see printf
    \param ... set of parameters
    \return the bytes used by the buffer to store the string.    
*/
int             jasperserver_string_format(jasperserver_string_t *jsstr,const char *format, ...);
//void            jasperserver_replace_string(jasperserver_string_t *jsstr,const char *oldString, const char *newString);

#endif
