/*
 * Copyright (C) 2007 JasperSoft http://www.jaspersoft.com
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed WITHOUT ANY WARRANTY; and without the 
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt 
 * or write to:
 * 
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */

/** 
 *  \file jsput.c
 *  This sample shows how to put a file into the repository. In particular this sample shows how to upload a resource of type image.
 *  If a file is specified on the command line, the content of the file will be attached to the SOAP request and store as
 *  resource content on the server.
 *  
 *  In general, the put service can be used to create new folders, new resources of several types, modify a resource and 
 *  create/modify a report unit.
 *  
 *  \see  jasperserver_ws_put
*/

#include "../jasperserver/jasperserver.h"

const char server[] = "http://192.168.1.20:8080/jasperserver-pro/services/repository";

/** 
 *  \brief Sample function to put a resource in the repository.
 *
 *  If a file is specified on the command line, the content of the file will be attached to the SOAP request and store as
 *  resource content on the server.
 *  
 *  \param dir URI of the folder to list.
*/
void put_resource(char *name, char *parentUri, char *infile)
{
      jasperserver_request_t *request = NULL;
      jasperserver_server_t *jasperServer = NULL;
      jasperserver_resource_descriptor_t *res = NULL;
      jasperserver_operation_result_t *operationResult = NULL;
      jasperserver_resource_property_t *propHasData = NULL;
      jasperserver_resource_property_t *propParentUri = NULL;
 
      // Create a new structure of type jasperserver_server_t to save the
      // server url and credentials.
      jasperServer = jasperserver_server_new();
      jasperserver_string_cset(jasperServer->url, server);
      jasperserver_string_cset(jasperServer->username, "jasperadmin|organization_1");
      jasperserver_string_cset(jasperServer->password, "jasperadmin");
     
      // Create a new structure of type jasperserver_request_t to store
      // the request information like the resource to list    
      request = jasperserver_request_new();   
     
      //Creating and populating a new resource_descriptor...
      // The descriptor will be an image.
      res = jasperserver_resource_descriptor_new();
      request->resource = res;
      
      jasperserver_string_cset(res->name, name);
      jasperserver_string_cset(res->label, name);
      
      // If you are uploading a new resource, be sure you set the isNew flag
      // otherwise set it to 0 (that's when you just update a resource)
      res->isNew = 1;
      jasperserver_string_cset(res->uriString, parentUri);
      jasperserver_string_cset(res->wsType, JS_TYPE_IMAGE );
      jasperserver_string_cappend(res->uriString, "/");
      jasperserver_string_cappend(res->uriString, name);
      
      // Set the property JS_PROP_FILERESOURCE_HAS_DATA if there is come content
      propHasData = jasperserver_resource_property_new();
      jasperserver_string_cset(propHasData->name, JS_PROP_FILERESOURCE_HAS_DATA);
      jasperserver_string_cset(propHasData->value, (infile) ? "true" : "false");
      
      // In order to correctly save the resource in the right place, you need to set the JS_PROP_PARENT_FOLDER
      propParentUri = jasperserver_resource_property_new();
      jasperserver_string_cset(propParentUri->name, JS_PROP_PARENT_FOLDER);
      jasperserver_string_cset(propParentUri->value, parentUri);
      
      jasperserver_list_append((jasperserver_list_t **)&res->properties, (jasperserver_list_t *)propHasData);
      jasperserver_list_append((jasperserver_list_t **)&res->properties, (jasperserver_list_t *)propParentUri);
      
      // Call the web service
      operationResult = jasperserver_ws_put(jasperServer, request, infile);
      
      if (operationResult->returnCode == 0)
      {
         printf("Server version: %s, Return code: '%d', Return message '%s'\n", JS_CSTR( operationResult->version),  operationResult->returnCode, JS_CSTR( operationResult->returnMessage ));
         printf("Uploaded resource: %s\n\n", JS_CSTR( request->resource->uriString));
         fflush( stdout );
         
         if (operationResult->resources)
         {
            jasperserver_print_resource_descriptor( operationResult->resources, 0 );
         }
      }
      else
      {
          //printf("Error executing the service: %d (%s)", operationResult->returnCode, JS_UTFSTR(operationResult->returnMessage) );
          printf("Error executing the service: %d (%s)", operationResult->returnCode, JS_UTFSTR(operationResult->returnMessage) );
      }
                                      
      jasperserver_server_free( jasperServer );
      jasperserver_request_free( request );
      jasperserver_operation_result_free( operationResult );
      
}


int main(int argc, char **argv)
{ 
   char *parentUri = NULL;
   char *resourceName = NULL;
   char *inputfile = NULL;
   printf("JasperServer list sample (C) 2007 JasperSoft Corp.!\n");
   
   if (argc > 2)
   {
     resourceName = argv[1];
     parentUri = argv[2];
     
     if (argc > 3)
     {
        inputfile = argv[3]; 
     }
   }
   else
   {
     printf("Usage: jsput <parent uri> <resource name> [input file name]\n");
     printf("I.e.  jsput newImage /images  [myimage.gif]\n");
     exit(0);
   }
   put_resource(resourceName, parentUri, inputfile);

   return 0;
}
