#
# Copyright 2008-2016 Red Hat, Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
#
# Refer to the README and COPYING files for full details of the license
#
from __future__ import absolute_import

import logging

import six

from vdsm import numa
from vdsm import utils

from vdsm import v2v


JIFFIES_BOUND = 2 ** 32
NETSTATS_BOUND = 2 ** 32


_clock = utils.monotonic_time
_start_time = 0


def _elapsed_time():
    return _clock() - _start_time


def start(clock=utils.monotonic_time):
    global _clock
    global _start_time
    _clock = clock
    _start_time = _clock()


def produce(first_sample, last_sample):
    stats = _empty_stats()

    if first_sample is None:
        return stats

    stats.update(_get_interfaces_stats(first_sample, last_sample))

    interval = last_sample.timestamp - first_sample.timestamp

    jiffies = (
        last_sample.pidcpu.user - first_sample.pidcpu.user
    ) % JIFFIES_BOUND
    stats['cpuUserVdsmd'] = jiffies / interval
    jiffies = (
        last_sample.pidcpu.sys - first_sample.pidcpu.sys
    ) % JIFFIES_BOUND
    stats['cpuSysVdsmd'] = jiffies / interval

    jiffies = (
        last_sample.totcpu.user - first_sample.totcpu.user
    ) % JIFFIES_BOUND
    stats['cpuUser'] = jiffies / interval / last_sample.ncpus
    jiffies = (
        last_sample.totcpu.sys - first_sample.totcpu.sys
    ) % JIFFIES_BOUND
    stats['cpuSys'] = jiffies / interval / last_sample.ncpus
    stats['cpuIdle'] = max(0.0,
                           100.0 - stats['cpuUser'] - stats['cpuSys'])
    stats['memUsed'] = last_sample.memUsed
    stats['anonHugePages'] = last_sample.anonHugePages
    stats['cpuLoad'] = last_sample.cpuLoad

    stats['diskStats'] = last_sample.diskStats
    stats['thpState'] = last_sample.thpState

    if _boot_time():
        stats['bootTime'] = _boot_time()

    stats['numaNodeMemFree'] = last_sample.numaNodeMem.nodesMemSample
    stats['cpuStatistics'] = _get_cpu_core_stats(
        first_sample, last_sample)

    stats['v2vJobs'] = v2v.get_jobs_status()
    return stats


def _get_cpu_core_stats(first_sample, last_sample):
    interval = last_sample.timestamp - first_sample.timestamp

    def compute_cpu_usage(cpu_core, mode):
        first_core_sample = first_sample.cpuCores.getCoreSample(cpu_core)
        last_core_sample = last_sample.cpuCores.getCoreSample(cpu_core)
        if not first_core_sample or not last_core_sample:
            raise MissingSample()
        jiffies = (
            last_core_sample[mode] - first_core_sample[mode]
        ) % JIFFIES_BOUND
        return ("%.2f" % (jiffies / interval))

    cpu_core_stats = {}
    for node_index, numa_node in six.iteritems(numa.topology()):
        cpu_cores = numa_node['cpus']
        for cpu_core in cpu_cores:
            try:
                user_cpu_usage = compute_cpu_usage(cpu_core, 'user')
                system_cpu_usage = compute_cpu_usage(cpu_core, 'sys')
            except MissingSample:
                # Only collect data when all required samples already present
                continue
            core_stat = {
                'nodeIndex': int(node_index),
                'cpuUser': user_cpu_usage,
                'cpuSys': system_cpu_usage,
            }
            core_stat['cpuIdle'] = (
                "%.2f" % max(0.0,
                             100.0 -
                             float(core_stat['cpuUser']) -
                             float(core_stat['cpuSys'])))
            cpu_core_stats[str(cpu_core)] = core_stat
    return cpu_core_stats


class MissingSample(Exception):
    pass


def _get_interfaces_stats(first_sample, last_sample):
    rx = tx = rxDropped = txDropped = 0
    stats = {'network': {}}
    for ifid in last_sample.interfaces:
        # it skips hot-plugged devices if we haven't enough information
        # to count stats from it
        if ifid not in first_sample.interfaces:
            continue

        ifrate = last_sample.interfaces[ifid].speed or 1000
        thisRx = (
            last_sample.interfaces[ifid].rx -
            first_sample.interfaces[ifid].rx
        ) % NETSTATS_BOUND
        thisTx = (
            last_sample.interfaces[ifid].tx -
            first_sample.interfaces[ifid].tx
        ) % NETSTATS_BOUND
        iface = last_sample.interfaces[ifid]
        stats['network'][ifid] = {
            'name': ifid, 'speed': str(ifrate),
            'rxDropped': str(iface.rxDropped),
            'txDropped': str(iface.txDropped),
            'rxErrors': str(iface.rxErrors),
            'txErrors': str(iface.txErrors),
            'state': iface.operstate,
            'rx': str(iface.rx),
            'tx': str(iface.tx),
            'sampleTime': last_sample.timestamp,
        }
        rx += thisRx
        tx += thisTx
        rxDropped += last_sample.interfaces[ifid].rxDropped
        txDropped += last_sample.interfaces[ifid].txDropped

    stats['rxDropped'] = rxDropped
    stats['txDropped'] = txDropped

    return stats


_PROC_STAT_PATH = '/proc/stat'


def get_boot_time():
    """
    Returns the boot time of the machine in seconds since epoch.

    Raises IOError if file access fails, or ValueError if boot time not
    present in file.
    """
    with open(_PROC_STAT_PATH) as proc_stat:
        for line in proc_stat:
            if line.startswith('btime'):
                parts = line.split()
                if len(parts) > 1:
                    return int(parts[1])
                else:
                    break
    raise ValueError('Boot time not present')


@utils.memoized
def _boot_time():
    # Try to get boot time only once, if N/A just log the error and never
    # include it in the response.
    try:
        return get_boot_time()
    except (IOError, ValueError):
        logging.exception('Failed to get boot time')
        return None


def _empty_stats():
    return {
        'cpuLoad': 0.0,
        'cpuUser': 0.0,
        'cpuSys': 0.0,
        'cpuIdle': 100.0,
        'cpuSysVdsmd': 0.0,
        'cpuUserVdsmd': 0.0,
        'elapsedTime': _elapsed_time(),
        'memUsed': 0.0,
        'anonHugePages': 0.0,
    }
